package com.datadog.android.sessionreplay.model

import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Boolean
import kotlin.Long
import kotlin.Number
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of a Session Replay Segment context.
 * Schema of common properties for a Segment Metadata type.
 * @param application Application properties
 * @param session Session properties
 * @param view View properties
 * @param start The start UTC timestamp in milliseconds corresponding to the first record in the
 * Segment data. Each timestamp is computed as the UTC interval since 00:00:00.000 01.01.1970.
 * @param end The end UTC timestamp in milliseconds corresponding to the last record in the Segment
 * data. Each timestamp is computed as the UTC interval since 00:00:00.000 01.01.1970.
 * @param recordsCount The number of records in this Segment.
 * @param indexInView The index of this Segment in the segments list that was recorded for this view
 * ID. Starts from 0.
 * @param hasFullSnapshot Whether this Segment contains a full snapshot record or not.
 * @param source The source of this record
 * @param records The records contained by this Segment.
 */
public data class MobileSegment(
    public val application: Application,
    public val session: Session,
    public val view: View,
    public val start: Long,
    public val end: Long,
    public val recordsCount: Long,
    public val indexInView: Long? = null,
    public val hasFullSnapshot: Boolean? = null,
    public val source: Source,
    public val records: List<MobileRecord>,
) {
    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.add("application", application.toJson())
        json.add("session", session.toJson())
        json.add("view", view.toJson())
        json.addProperty("start", start)
        json.addProperty("end", end)
        json.addProperty("records_count", recordsCount)
        indexInView?.let { indexInViewNonNull ->
            json.addProperty("index_in_view", indexInViewNonNull)
        }
        hasFullSnapshot?.let { hasFullSnapshotNonNull ->
            json.addProperty("has_full_snapshot", hasFullSnapshotNonNull)
        }
        json.add("source", source.toJson())
        val recordsArray = JsonArray(records.size)
        records.forEach { recordsArray.add(it.toJson()) }
        json.add("records", recordsArray)
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): MobileSegment {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type MobileSegment",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): MobileSegment {
            try {
                val application = jsonObject.get("application").asJsonObject.let {
                    Application.fromJsonObject(it)
                }
                val session = jsonObject.get("session").asJsonObject.let {
                    Session.fromJsonObject(it)
                }
                val view = jsonObject.get("view").asJsonObject.let {
                    View.fromJsonObject(it)
                }
                val start = jsonObject.get("start").asLong
                val end = jsonObject.get("end").asLong
                val recordsCount = jsonObject.get("records_count").asLong
                val indexInView = jsonObject.get("index_in_view")?.asLong
                val hasFullSnapshot = jsonObject.get("has_full_snapshot")?.asBoolean
                val source = Source.fromJson(jsonObject.get("source").asString)
                val records = jsonObject.get("records").asJsonArray.let { jsonArray ->
                    val collection = ArrayList<MobileRecord>(jsonArray.size())
                    jsonArray.forEach {
                        collection.add(MobileRecord.fromJsonObject(it.asJsonObject))
                    }
                    collection
                }
                return MobileSegment(application, session, view, start, end, recordsCount,
                        indexInView, hasFullSnapshot, source, records)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type MobileSegment",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type MobileSegment",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type MobileSegment",
                    e
                )
            }
        }
    }

    /**
     * Application properties
     * @param id UUID of the application
     */
    public data class Application(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id").asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session properties
     * @param id UUID of the session
     */
    public data class Session(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Session {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Session {
                try {
                    val id = jsonObject.get("id").asString
                    return Session(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }
        }
    }

    /**
     * View properties
     * @param id UUID of the view
     */
    public data class View(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): View {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): View {
                try {
                    val id = jsonObject.get("id").asString
                    return View(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }
        }
    }

    /**
     * Mobile-specific. Schema of a Session Replay Record.
     */
    public sealed class MobileRecord {
        public abstract fun toJson(): JsonElement

        /**
         * Schema of common properties for a Record event type.
         * @param timestamp Defines the UTC time in milliseconds when this Record was performed.
         */
        public data class MobileFullSnapshotRecord(
            public val timestamp: Long,
            public val `data`: Data,
        ) : MobileRecord() {
            public val type: Long = 10L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("timestamp", timestamp)
                json.addProperty("type", type)
                json.add("data", data.toJson())
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): MobileFullSnapshotRecord {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileFullSnapshotRecord",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): MobileFullSnapshotRecord {
                    try {
                        val timestamp = jsonObject.get("timestamp").asLong
                        val type = jsonObject.get("type").asLong
                        val data = jsonObject.get("data").asJsonObject.let {
                            Data.fromJsonObject(it)
                        }
                        check(type == 10.0.toLong())
                        return MobileFullSnapshotRecord(timestamp, data)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileFullSnapshotRecord",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileFullSnapshotRecord",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileFullSnapshotRecord",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for a Record event type.
         * @param timestamp Defines the UTC time in milliseconds when this Record was performed.
         * @param data Mobile-specific. Schema of a Session Replay IncrementalData type.
         */
        public data class MobileIncrementalSnapshotRecord(
            public val timestamp: Long,
            public val `data`: MobileIncrementalData,
        ) : MobileRecord() {
            public val type: Long = 11L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("timestamp", timestamp)
                json.addProperty("type", type)
                json.add("data", data.toJson())
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): MobileIncrementalSnapshotRecord {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileIncrementalSnapshotRecord",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): MobileIncrementalSnapshotRecord {
                    try {
                        val timestamp = jsonObject.get("timestamp").asLong
                        val type = jsonObject.get("type").asLong
                        val data = jsonObject.get("data").asJsonObject.let {
                            MobileIncrementalData.fromJsonObject(it)
                        }
                        check(type == 11.0.toLong())
                        return MobileIncrementalSnapshotRecord(timestamp, data)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileIncrementalSnapshotRecord",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileIncrementalSnapshotRecord",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileIncrementalSnapshotRecord",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for a Record event type.
         * @param timestamp Defines the UTC time in milliseconds when this Record was performed.
         * @param slotId Unique ID of the slot that generated this record.
         * @param data The data contained by this record.
         */
        public data class MetaRecord(
            public val timestamp: Long,
            public val slotId: String? = null,
            public val `data`: Data1,
        ) : MobileRecord() {
            public val type: Long = 4L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("timestamp", timestamp)
                slotId?.let { slotIdNonNull ->
                    json.addProperty("slotId", slotIdNonNull)
                }
                json.addProperty("type", type)
                json.add("data", data.toJson())
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): MetaRecord {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type MetaRecord",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): MetaRecord {
                    try {
                        val timestamp = jsonObject.get("timestamp").asLong
                        val slotId = jsonObject.get("slotId")?.asString
                        val type = jsonObject.get("type").asLong
                        val data = jsonObject.get("data").asJsonObject.let {
                            Data1.fromJsonObject(it)
                        }
                        check(type == 4.0.toLong())
                        return MetaRecord(timestamp, slotId, data)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type MetaRecord",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type MetaRecord",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type MetaRecord",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for a Record event type.
         * @param timestamp Defines the UTC time in milliseconds when this Record was performed.
         * @param slotId Unique ID of the slot that generated this record.
         */
        public data class FocusRecord(
            public val timestamp: Long,
            public val slotId: String? = null,
            public val `data`: Data2,
        ) : MobileRecord() {
            public val type: Long = 6L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("timestamp", timestamp)
                slotId?.let { slotIdNonNull ->
                    json.addProperty("slotId", slotIdNonNull)
                }
                json.addProperty("type", type)
                json.add("data", data.toJson())
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): FocusRecord {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type FocusRecord",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): FocusRecord {
                    try {
                        val timestamp = jsonObject.get("timestamp").asLong
                        val slotId = jsonObject.get("slotId")?.asString
                        val type = jsonObject.get("type").asLong
                        val data = jsonObject.get("data").asJsonObject.let {
                            Data2.fromJsonObject(it)
                        }
                        check(type == 6.0.toLong())
                        return FocusRecord(timestamp, slotId, data)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type FocusRecord",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type FocusRecord",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type FocusRecord",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for a Record event type.
         * @param timestamp Defines the UTC time in milliseconds when this Record was performed.
         * @param slotId Unique ID of the slot that generated this record.
         */
        public data class ViewEndRecord(
            public val timestamp: Long,
            public val slotId: String? = null,
        ) : MobileRecord() {
            public val type: Long = 7L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("timestamp", timestamp)
                slotId?.let { slotIdNonNull ->
                    json.addProperty("slotId", slotIdNonNull)
                }
                json.addProperty("type", type)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): ViewEndRecord {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ViewEndRecord",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): ViewEndRecord {
                    try {
                        val timestamp = jsonObject.get("timestamp").asLong
                        val slotId = jsonObject.get("slotId")?.asString
                        val type = jsonObject.get("type").asLong
                        check(type == 7.0.toLong())
                        return ViewEndRecord(timestamp, slotId)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ViewEndRecord",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type ViewEndRecord",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type ViewEndRecord",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for a Record event type.
         * @param timestamp Defines the UTC time in milliseconds when this Record was performed.
         * @param slotId Unique ID of the slot that generated this record.
         */
        public data class VisualViewportRecord(
            public val timestamp: Long,
            public val slotId: String? = null,
            public val `data`: Data3,
        ) : MobileRecord() {
            public val type: Long = 8L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("timestamp", timestamp)
                slotId?.let { slotIdNonNull ->
                    json.addProperty("slotId", slotIdNonNull)
                }
                json.add("data", data.toJson())
                json.addProperty("type", type)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): VisualViewportRecord {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type VisualViewportRecord",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): VisualViewportRecord {
                    try {
                        val timestamp = jsonObject.get("timestamp").asLong
                        val slotId = jsonObject.get("slotId")?.asString
                        val data = jsonObject.get("data").asJsonObject.let {
                            Data3.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asLong
                        check(type == 8.0.toLong())
                        return VisualViewportRecord(timestamp, slotId, data)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type VisualViewportRecord",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type VisualViewportRecord",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type VisualViewportRecord",
                            e
                        )
                    }
                }
            }
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): MobileRecord {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into one of type MobileRecord",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): MobileRecord {
                val errors = mutableListOf<Throwable>()
                val asMobileFullSnapshotRecord = try {
                    MobileFullSnapshotRecord.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asMobileIncrementalSnapshotRecord = try {
                    MobileIncrementalSnapshotRecord.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asMetaRecord = try {
                    MetaRecord.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asFocusRecord = try {
                    FocusRecord.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asViewEndRecord = try {
                    ViewEndRecord.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asVisualViewportRecord = try {
                    VisualViewportRecord.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val result = arrayOf(
                    asMobileFullSnapshotRecord,
                    asMobileIncrementalSnapshotRecord,
                    asMetaRecord,
                    asFocusRecord,
                    asViewEndRecord,
                    asVisualViewportRecord,
                ).firstOrNull { it != null }
                if (result == null) {
                    val message = "Unable to parse json into one of type \n" + "MobileRecord\n" +
                        errors.joinToString("\n") { it.message.toString() }
                    throw JsonParseException(message)
                }
                return result
            }
        }
    }

    /**
     * @param wireframes The Wireframes contained by this Record.
     */
    public data class Data(
        public val wireframes: List<Wireframe>,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            val wireframesArray = JsonArray(wireframes.size)
            wireframes.forEach { wireframesArray.add(it.toJson()) }
            json.add("wireframes", wireframesArray)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Data {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Data {
                try {
                    val wireframes = jsonObject.get("wireframes").asJsonArray.let { jsonArray ->
                        val collection = ArrayList<Wireframe>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(Wireframe.fromJsonObject(it.asJsonObject))
                        }
                        collection
                    }
                    return Data(wireframes)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data",
                        e
                    )
                }
            }
        }
    }

    /**
     * Mobile-specific. Schema of a Session Replay IncrementalData type.
     */
    public sealed class MobileIncrementalData {
        public abstract fun toJson(): JsonElement

        /**
         * @param adds Contains the newly added wireframes.
         * @param removes Contains the removed wireframes as an array of ids.
         * @param updates Contains the updated wireframes mutations.
         */
        public data class MobileMutationData(
            public val adds: List<Add>? = null,
            public val removes: List<Remove>? = null,
            public val updates: List<WireframeUpdateMutation>? = null,
        ) : MobileIncrementalData() {
            public val source: Long = 0L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("source", source)
                adds?.let { addsNonNull ->
                    val addsArray = JsonArray(addsNonNull.size)
                    addsNonNull.forEach { addsArray.add(it.toJson()) }
                    json.add("adds", addsArray)
                }
                removes?.let { removesNonNull ->
                    val removesArray = JsonArray(removesNonNull.size)
                    removesNonNull.forEach { removesArray.add(it.toJson()) }
                    json.add("removes", removesArray)
                }
                updates?.let { updatesNonNull ->
                    val updatesArray = JsonArray(updatesNonNull.size)
                    updatesNonNull.forEach { updatesArray.add(it.toJson()) }
                    json.add("updates", updatesArray)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): MobileMutationData {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileMutationData",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): MobileMutationData {
                    try {
                        val source = jsonObject.get("source").asLong
                        val adds = jsonObject.get("adds")?.asJsonArray?.let { jsonArray ->
                            val collection = ArrayList<Add>(jsonArray.size())
                            jsonArray.forEach {
                                collection.add(Add.fromJsonObject(it.asJsonObject))
                            }
                            collection
                        }
                        val removes = jsonObject.get("removes")?.asJsonArray?.let { jsonArray ->
                            val collection = ArrayList<Remove>(jsonArray.size())
                            jsonArray.forEach {
                                collection.add(Remove.fromJsonObject(it.asJsonObject))
                            }
                            collection
                        }
                        val updates = jsonObject.get("updates")?.asJsonArray?.let { jsonArray ->
                            val collection = ArrayList<WireframeUpdateMutation>(jsonArray.size())
                            jsonArray.forEach {
                                collection.add(WireframeUpdateMutation.fromJsonObject(it.asJsonObject))
                            }
                            collection
                        }
                        check(source == 0.0.toLong())
                        return MobileMutationData(adds, removes, updates)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileMutationData",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileMutationData",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type MobileMutationData",
                            e
                        )
                    }
                }
            }
        }

        /**
         * @param positions Contains the positions of the finger on the screen during the
         * touchDown/touchUp event lifecycle.
         */
        public data class TouchData(
            public val positions: List<Position>? = null,
        ) : MobileIncrementalData() {
            public val source: Long = 2L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("source", source)
                positions?.let { positionsNonNull ->
                    val positionsArray = JsonArray(positionsNonNull.size)
                    positionsNonNull.forEach { positionsArray.add(it.toJson()) }
                    json.add("positions", positionsArray)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): TouchData {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type TouchData",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): TouchData {
                    try {
                        val source = jsonObject.get("source").asLong
                        val positions = jsonObject.get("positions")?.asJsonArray?.let { jsonArray ->
                            val collection = ArrayList<Position>(jsonArray.size())
                            jsonArray.forEach {
                                collection.add(Position.fromJsonObject(it.asJsonObject))
                            }
                            collection
                        }
                        check(source == 2.0.toLong())
                        return TouchData(positions)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type TouchData",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type TouchData",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type TouchData",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of a ViewportResizeDimension.
         * @param width The new width of the screen in pixels, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width is divided by 2 to get
         * a normalized width.
         * @param height The new height of the screen in pixels, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height is divided by
         * 2 to get a normalized height.
         */
        public data class ViewportResizeData(
            public val width: Long,
            public val height: Long,
        ) : MobileIncrementalData() {
            public val source: Long = 4L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("source", source)
                json.addProperty("width", width)
                json.addProperty("height", height)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): ViewportResizeData {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ViewportResizeData",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): ViewportResizeData {
                    try {
                        val source = jsonObject.get("source").asLong
                        val width = jsonObject.get("width").asLong
                        val height = jsonObject.get("height").asLong
                        check(source == 4.0.toLong())
                        return ViewportResizeData(width, height)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ViewportResizeData",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type ViewportResizeData",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type ViewportResizeData",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of a PointerInteraction.
         * @param pointerEventType Schema of an PointerEventType
         * @param pointerType Schema of an PointerType
         * @param pointerId Id of the pointer of this PointerInteraction.
         * @param x X-axis coordinate for this PointerInteraction.
         * @param y Y-axis coordinate for this PointerInteraction.
         */
        public data class PointerInteractionData(
            public val pointerEventType: PointerEventType,
            public val pointerType: PointerType,
            public val pointerId: Long,
            public val x: Number,
            public val y: Number,
        ) : MobileIncrementalData() {
            public val source: Long = 9L

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("source", source)
                json.add("pointerEventType", pointerEventType.toJson())
                json.add("pointerType", pointerType.toJson())
                json.addProperty("pointerId", pointerId)
                json.addProperty("x", x)
                json.addProperty("y", y)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): PointerInteractionData {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type PointerInteractionData",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): PointerInteractionData {
                    try {
                        val source = jsonObject.get("source").asLong
                        val pointerEventType =
                                PointerEventType.fromJson(jsonObject.get("pointerEventType").asString)
                        val pointerType =
                                PointerType.fromJson(jsonObject.get("pointerType").asString)
                        val pointerId = jsonObject.get("pointerId").asLong
                        val x = jsonObject.get("x").asNumber
                        val y = jsonObject.get("y").asNumber
                        check(source == 9.0.toLong())
                        return PointerInteractionData(pointerEventType, pointerType, pointerId, x,
                                y)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type PointerInteractionData",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type PointerInteractionData",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type PointerInteractionData",
                            e
                        )
                    }
                }
            }
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): MobileIncrementalData {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into one of type MobileIncrementalData",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): MobileIncrementalData {
                val errors = mutableListOf<Throwable>()
                val asMobileMutationData = try {
                    MobileMutationData.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asTouchData = try {
                    TouchData.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asViewportResizeData = try {
                    ViewportResizeData.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asPointerInteractionData = try {
                    PointerInteractionData.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val result = arrayOf(
                    asMobileMutationData,
                    asTouchData,
                    asViewportResizeData,
                    asPointerInteractionData,
                ).firstOrNull { it != null }
                if (result == null) {
                    val message = "Unable to parse json into one of type \n" +
                            "MobileIncrementalData\n" +
                        errors.joinToString("\n") { it.message.toString() }
                    throw JsonParseException(message)
                }
                return result
            }
        }
    }

    /**
     * The data contained by this record.
     * @param width The width of the screen in pixels, normalized based on the device pixels per
     * inch density (DPI). Example: if a device has a DPI = 2, the normalized width is the current
     * width divided by 2.
     * @param height The height of the screen in pixels, normalized based on the device pixels per
     * inch density (DPI). Example: if a device has a DPI = 2, the normalized height is the current
     * height divided by 2.
     * @param href Browser-specific. URL of the view described by this record.
     */
    public data class Data1(
        public val width: Long,
        public val height: Long,
        public val href: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("width", width)
            json.addProperty("height", height)
            href?.let { hrefNonNull ->
                json.addProperty("href", hrefNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Data1 {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data1",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Data1 {
                try {
                    val width = jsonObject.get("width").asLong
                    val height = jsonObject.get("height").asLong
                    val href = jsonObject.get("href")?.asString
                    return Data1(width, height, href)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data1",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data1",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data1",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param hasFocus Whether this screen has a focus or not. For now it will always be true for
     * mobile.
     */
    public data class Data2(
        public val hasFocus: Boolean,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("has_focus", hasFocus)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Data2 {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data2",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Data2 {
                try {
                    val hasFocus = jsonObject.get("has_focus").asBoolean
                    return Data2(hasFocus)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data2",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data2",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data2",
                        e
                    )
                }
            }
        }
    }

    public data class Data3(
        public val height: Number,
        public val offsetLeft: Number,
        public val offsetTop: Number,
        public val pageLeft: Number,
        public val pageTop: Number,
        public val scale: Number,
        public val width: Number,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("height", height)
            json.addProperty("offsetLeft", offsetLeft)
            json.addProperty("offsetTop", offsetTop)
            json.addProperty("pageLeft", pageLeft)
            json.addProperty("pageTop", pageTop)
            json.addProperty("scale", scale)
            json.addProperty("width", width)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Data3 {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data3",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Data3 {
                try {
                    val height = jsonObject.get("height").asNumber
                    val offsetLeft = jsonObject.get("offsetLeft").asNumber
                    val offsetTop = jsonObject.get("offsetTop").asNumber
                    val pageLeft = jsonObject.get("pageLeft").asNumber
                    val pageTop = jsonObject.get("pageTop").asNumber
                    val scale = jsonObject.get("scale").asNumber
                    val width = jsonObject.get("width").asNumber
                    return Data3(height, offsetLeft, offsetTop, pageLeft, pageTop, scale, width)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data3",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data3",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Data3",
                        e
                    )
                }
            }
        }
    }

    /**
     * Schema of a Wireframe type.
     */
    public sealed class Wireframe {
        public abstract fun toJson(): JsonElement

        /**
         * Schema of common properties for Wireframe events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         */
        public data class ShapeWireframe(
            public val id: Long,
            public val x: Long,
            public val y: Long,
            public val width: Long,
            public val height: Long,
            public val clip: WireframeClip? = null,
            public val shapeStyle: ShapeStyle? = null,
            public val border: ShapeBorder? = null,
        ) : Wireframe() {
            public val type: String = "shape"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                json.addProperty("x", x)
                json.addProperty("y", y)
                json.addProperty("width", width)
                json.addProperty("height", height)
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                shapeStyle?.let { shapeStyleNonNull ->
                    json.add("shapeStyle", shapeStyleNonNull.toJson())
                }
                border?.let { borderNonNull ->
                    json.add("border", borderNonNull.toJson())
                }
                json.addProperty("type", type)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): ShapeWireframe {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ShapeWireframe",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): ShapeWireframe {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x").asLong
                        val y = jsonObject.get("y").asLong
                        val width = jsonObject.get("width").asLong
                        val height = jsonObject.get("height").asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val shapeStyle = jsonObject.get("shapeStyle")?.asJsonObject?.let {
                            ShapeStyle.fromJsonObject(it)
                        }
                        val border = jsonObject.get("border")?.asJsonObject?.let {
                            ShapeBorder.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        check(type == "shape")
                        return ShapeWireframe(id, x, y, width, height, clip, shapeStyle, border)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ShapeWireframe",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type ShapeWireframe",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type ShapeWireframe",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for Wireframe events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         * @param text The text value of the wireframe.
         */
        public data class TextWireframe(
            public val id: Long,
            public val x: Long,
            public val y: Long,
            public val width: Long,
            public val height: Long,
            public val clip: WireframeClip? = null,
            public val shapeStyle: ShapeStyle? = null,
            public val border: ShapeBorder? = null,
            public var text: String,
            public val textStyle: TextStyle,
            public val textPosition: TextPosition? = null,
        ) : Wireframe() {
            public val type: String = "text"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                json.addProperty("x", x)
                json.addProperty("y", y)
                json.addProperty("width", width)
                json.addProperty("height", height)
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                shapeStyle?.let { shapeStyleNonNull ->
                    json.add("shapeStyle", shapeStyleNonNull.toJson())
                }
                border?.let { borderNonNull ->
                    json.add("border", borderNonNull.toJson())
                }
                json.addProperty("type", type)
                json.addProperty("text", text)
                json.add("textStyle", textStyle.toJson())
                textPosition?.let { textPositionNonNull ->
                    json.add("textPosition", textPositionNonNull.toJson())
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): TextWireframe {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type TextWireframe",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): TextWireframe {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x").asLong
                        val y = jsonObject.get("y").asLong
                        val width = jsonObject.get("width").asLong
                        val height = jsonObject.get("height").asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val shapeStyle = jsonObject.get("shapeStyle")?.asJsonObject?.let {
                            ShapeStyle.fromJsonObject(it)
                        }
                        val border = jsonObject.get("border")?.asJsonObject?.let {
                            ShapeBorder.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        val text = jsonObject.get("text").asString
                        val textStyle = jsonObject.get("textStyle").asJsonObject.let {
                            TextStyle.fromJsonObject(it)
                        }
                        val textPosition = jsonObject.get("textPosition")?.asJsonObject?.let {
                            TextPosition.fromJsonObject(it)
                        }
                        check(type == "text")
                        return TextWireframe(id, x, y, width, height, clip, shapeStyle, border,
                                text, textStyle, textPosition)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type TextWireframe",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type TextWireframe",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type TextWireframe",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for Wireframe events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         * @param base64 base64 representation of the image. Not required as the ImageWireframe can
         * be initialised without any base64
         * @param resourceId Unique identifier of the image resource
         * @param mimeType MIME type of the image file
         * @param isEmpty Flag describing an image wireframe that should render an empty state
         * placeholder
         */
        public data class ImageWireframe(
            public val id: Long,
            public val x: Long,
            public val y: Long,
            public val width: Long,
            public val height: Long,
            public val clip: WireframeClip? = null,
            public val shapeStyle: ShapeStyle? = null,
            public val border: ShapeBorder? = null,
            public var base64: String? = null,
            public var resourceId: String? = null,
            public var mimeType: String? = null,
            public var isEmpty: Boolean? = null,
        ) : Wireframe() {
            public val type: String = "image"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                json.addProperty("x", x)
                json.addProperty("y", y)
                json.addProperty("width", width)
                json.addProperty("height", height)
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                shapeStyle?.let { shapeStyleNonNull ->
                    json.add("shapeStyle", shapeStyleNonNull.toJson())
                }
                border?.let { borderNonNull ->
                    json.add("border", borderNonNull.toJson())
                }
                json.addProperty("type", type)
                base64?.let { base64NonNull ->
                    json.addProperty("base64", base64NonNull)
                }
                resourceId?.let { resourceIdNonNull ->
                    json.addProperty("resourceId", resourceIdNonNull)
                }
                mimeType?.let { mimeTypeNonNull ->
                    json.addProperty("mimeType", mimeTypeNonNull)
                }
                isEmpty?.let { isEmptyNonNull ->
                    json.addProperty("isEmpty", isEmptyNonNull)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): ImageWireframe {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ImageWireframe",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): ImageWireframe {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x").asLong
                        val y = jsonObject.get("y").asLong
                        val width = jsonObject.get("width").asLong
                        val height = jsonObject.get("height").asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val shapeStyle = jsonObject.get("shapeStyle")?.asJsonObject?.let {
                            ShapeStyle.fromJsonObject(it)
                        }
                        val border = jsonObject.get("border")?.asJsonObject?.let {
                            ShapeBorder.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        val base64 = jsonObject.get("base64")?.asString
                        val resourceId = jsonObject.get("resourceId")?.asString
                        val mimeType = jsonObject.get("mimeType")?.asString
                        val isEmpty = jsonObject.get("isEmpty")?.asBoolean
                        check(type == "image")
                        return ImageWireframe(id, x, y, width, height, clip, shapeStyle, border,
                                base64, resourceId, mimeType, isEmpty)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ImageWireframe",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type ImageWireframe",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type ImageWireframe",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for Wireframe events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         * @param label Label of the placeholder
         */
        public data class PlaceholderWireframe(
            public val id: Long,
            public val x: Long,
            public val y: Long,
            public val width: Long,
            public val height: Long,
            public val clip: WireframeClip? = null,
            public var label: String? = null,
        ) : Wireframe() {
            public val type: String = "placeholder"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                json.addProperty("x", x)
                json.addProperty("y", y)
                json.addProperty("width", width)
                json.addProperty("height", height)
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                json.addProperty("type", type)
                label?.let { labelNonNull ->
                    json.addProperty("label", labelNonNull)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): PlaceholderWireframe {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type PlaceholderWireframe",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): PlaceholderWireframe {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x").asLong
                        val y = jsonObject.get("y").asLong
                        val width = jsonObject.get("width").asLong
                        val height = jsonObject.get("height").asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        val label = jsonObject.get("label")?.asString
                        check(type == "placeholder")
                        return PlaceholderWireframe(id, x, y, width, height, clip, label)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type PlaceholderWireframe",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type PlaceholderWireframe",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type PlaceholderWireframe",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for Wireframe events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         * @param slotId Unique Id of the slot containing this webview.
         * @param isVisible Whether this webview is visible or not.
         */
        public data class WebviewWireframe(
            public val id: Long,
            public val x: Long,
            public val y: Long,
            public val width: Long,
            public val height: Long,
            public val clip: WireframeClip? = null,
            public val shapeStyle: ShapeStyle? = null,
            public val border: ShapeBorder? = null,
            public val slotId: String,
            public val isVisible: Boolean? = null,
        ) : Wireframe() {
            public val type: String = "webview"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                json.addProperty("x", x)
                json.addProperty("y", y)
                json.addProperty("width", width)
                json.addProperty("height", height)
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                shapeStyle?.let { shapeStyleNonNull ->
                    json.add("shapeStyle", shapeStyleNonNull.toJson())
                }
                border?.let { borderNonNull ->
                    json.add("border", borderNonNull.toJson())
                }
                json.addProperty("type", type)
                json.addProperty("slotId", slotId)
                isVisible?.let { isVisibleNonNull ->
                    json.addProperty("isVisible", isVisibleNonNull)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): WebviewWireframe {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type WebviewWireframe",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): WebviewWireframe {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x").asLong
                        val y = jsonObject.get("y").asLong
                        val width = jsonObject.get("width").asLong
                        val height = jsonObject.get("height").asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val shapeStyle = jsonObject.get("shapeStyle")?.asJsonObject?.let {
                            ShapeStyle.fromJsonObject(it)
                        }
                        val border = jsonObject.get("border")?.asJsonObject?.let {
                            ShapeBorder.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        val slotId = jsonObject.get("slotId").asString
                        val isVisible = jsonObject.get("isVisible")?.asBoolean
                        check(type == "webview")
                        return WebviewWireframe(id, x, y, width, height, clip, shapeStyle, border,
                                slotId, isVisible)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type WebviewWireframe",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type WebviewWireframe",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type WebviewWireframe",
                            e
                        )
                    }
                }
            }
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Wireframe {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into one of type Wireframe",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Wireframe {
                val errors = mutableListOf<Throwable>()
                val asShapeWireframe = try {
                    ShapeWireframe.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asTextWireframe = try {
                    TextWireframe.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asImageWireframe = try {
                    ImageWireframe.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asPlaceholderWireframe = try {
                    PlaceholderWireframe.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asWebviewWireframe = try {
                    WebviewWireframe.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val result = arrayOf(
                    asShapeWireframe,
                    asTextWireframe,
                    asImageWireframe,
                    asPlaceholderWireframe,
                    asWebviewWireframe,
                ).firstOrNull { it != null }
                if (result == null) {
                    val message = "Unable to parse json into one of type \n" + "Wireframe\n" +
                        errors.joinToString("\n") { it.message.toString() }
                    throw JsonParseException(message)
                }
                return result
            }
        }
    }

    /**
     * @param previousId The previous wireframe id next or after which this new wireframe is drawn
     * or attached to, respectively.
     * @param wireframe Schema of a Wireframe type.
     */
    public data class Add(
        public val previousId: Long? = null,
        public val wireframe: Wireframe,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            previousId?.let { previousIdNonNull ->
                json.addProperty("previousId", previousIdNonNull)
            }
            json.add("wireframe", wireframe.toJson())
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Add {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Add",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Add {
                try {
                    val previousId = jsonObject.get("previousId")?.asLong
                    val wireframe = jsonObject.get("wireframe").asJsonObject.let {
                        Wireframe.fromJsonObject(it)
                    }
                    return Add(previousId, wireframe)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Add",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Add",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Add",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param id The id of the wireframe that needs to be removed.
     */
    public data class Remove(
        public val id: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Remove {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Remove",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Remove {
                try {
                    val id = jsonObject.get("id").asLong
                    return Remove(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Remove",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Remove",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Remove",
                        e
                    )
                }
            }
        }
    }

    /**
     * Schema of a WireframeUpdateMutation type.
     */
    public sealed class WireframeUpdateMutation {
        public abstract fun toJson(): JsonElement

        /**
         * Schema of common properties for WireframeUpdate events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         * @param text The text value of the wireframe.
         */
        public data class TextWireframeUpdate(
            public val id: Long,
            public val x: Long? = null,
            public val y: Long? = null,
            public val width: Long? = null,
            public val height: Long? = null,
            public val clip: WireframeClip? = null,
            public val shapeStyle: ShapeStyle? = null,
            public val border: ShapeBorder? = null,
            public var text: String? = null,
            public val textStyle: TextStyle? = null,
            public val textPosition: TextPosition? = null,
        ) : WireframeUpdateMutation() {
            public val type: String = "text"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                x?.let { xNonNull ->
                    json.addProperty("x", xNonNull)
                }
                y?.let { yNonNull ->
                    json.addProperty("y", yNonNull)
                }
                width?.let { widthNonNull ->
                    json.addProperty("width", widthNonNull)
                }
                height?.let { heightNonNull ->
                    json.addProperty("height", heightNonNull)
                }
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                shapeStyle?.let { shapeStyleNonNull ->
                    json.add("shapeStyle", shapeStyleNonNull.toJson())
                }
                border?.let { borderNonNull ->
                    json.add("border", borderNonNull.toJson())
                }
                json.addProperty("type", type)
                text?.let { textNonNull ->
                    json.addProperty("text", textNonNull)
                }
                textStyle?.let { textStyleNonNull ->
                    json.add("textStyle", textStyleNonNull.toJson())
                }
                textPosition?.let { textPositionNonNull ->
                    json.add("textPosition", textPositionNonNull.toJson())
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): TextWireframeUpdate {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type TextWireframeUpdate",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): TextWireframeUpdate {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x")?.asLong
                        val y = jsonObject.get("y")?.asLong
                        val width = jsonObject.get("width")?.asLong
                        val height = jsonObject.get("height")?.asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val shapeStyle = jsonObject.get("shapeStyle")?.asJsonObject?.let {
                            ShapeStyle.fromJsonObject(it)
                        }
                        val border = jsonObject.get("border")?.asJsonObject?.let {
                            ShapeBorder.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        val text = jsonObject.get("text")?.asString
                        val textStyle = jsonObject.get("textStyle")?.asJsonObject?.let {
                            TextStyle.fromJsonObject(it)
                        }
                        val textPosition = jsonObject.get("textPosition")?.asJsonObject?.let {
                            TextPosition.fromJsonObject(it)
                        }
                        check(type == "text")
                        return TextWireframeUpdate(id, x, y, width, height, clip, shapeStyle,
                                border, text, textStyle, textPosition)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type TextWireframeUpdate",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type TextWireframeUpdate",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type TextWireframeUpdate",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for WireframeUpdate events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         */
        public data class ShapeWireframeUpdate(
            public val id: Long,
            public val x: Long? = null,
            public val y: Long? = null,
            public val width: Long? = null,
            public val height: Long? = null,
            public val clip: WireframeClip? = null,
            public val shapeStyle: ShapeStyle? = null,
            public val border: ShapeBorder? = null,
        ) : WireframeUpdateMutation() {
            public val type: String = "shape"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                x?.let { xNonNull ->
                    json.addProperty("x", xNonNull)
                }
                y?.let { yNonNull ->
                    json.addProperty("y", yNonNull)
                }
                width?.let { widthNonNull ->
                    json.addProperty("width", widthNonNull)
                }
                height?.let { heightNonNull ->
                    json.addProperty("height", heightNonNull)
                }
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                shapeStyle?.let { shapeStyleNonNull ->
                    json.add("shapeStyle", shapeStyleNonNull.toJson())
                }
                border?.let { borderNonNull ->
                    json.add("border", borderNonNull.toJson())
                }
                json.addProperty("type", type)
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): ShapeWireframeUpdate {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ShapeWireframeUpdate",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): ShapeWireframeUpdate {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x")?.asLong
                        val y = jsonObject.get("y")?.asLong
                        val width = jsonObject.get("width")?.asLong
                        val height = jsonObject.get("height")?.asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val shapeStyle = jsonObject.get("shapeStyle")?.asJsonObject?.let {
                            ShapeStyle.fromJsonObject(it)
                        }
                        val border = jsonObject.get("border")?.asJsonObject?.let {
                            ShapeBorder.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        check(type == "shape")
                        return ShapeWireframeUpdate(id, x, y, width, height, clip, shapeStyle,
                                border)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ShapeWireframeUpdate",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type ShapeWireframeUpdate",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type ShapeWireframeUpdate",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for WireframeUpdate events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         * @param base64 base64 representation of the image. Not required as the ImageWireframe can
         * be initialised without any base64
         * @param resourceId Unique identifier of the image resource
         * @param mimeType MIME type of the image file
         * @param isEmpty Flag describing an image wireframe that should render an empty state
         * placeholder
         */
        public data class ImageWireframeUpdate(
            public val id: Long,
            public val x: Long? = null,
            public val y: Long? = null,
            public val width: Long? = null,
            public val height: Long? = null,
            public val clip: WireframeClip? = null,
            public val shapeStyle: ShapeStyle? = null,
            public val border: ShapeBorder? = null,
            public var base64: String? = null,
            public var resourceId: String? = null,
            public var mimeType: String? = null,
            public var isEmpty: Boolean? = null,
        ) : WireframeUpdateMutation() {
            public val type: String = "image"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                x?.let { xNonNull ->
                    json.addProperty("x", xNonNull)
                }
                y?.let { yNonNull ->
                    json.addProperty("y", yNonNull)
                }
                width?.let { widthNonNull ->
                    json.addProperty("width", widthNonNull)
                }
                height?.let { heightNonNull ->
                    json.addProperty("height", heightNonNull)
                }
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                shapeStyle?.let { shapeStyleNonNull ->
                    json.add("shapeStyle", shapeStyleNonNull.toJson())
                }
                border?.let { borderNonNull ->
                    json.add("border", borderNonNull.toJson())
                }
                json.addProperty("type", type)
                base64?.let { base64NonNull ->
                    json.addProperty("base64", base64NonNull)
                }
                resourceId?.let { resourceIdNonNull ->
                    json.addProperty("resourceId", resourceIdNonNull)
                }
                mimeType?.let { mimeTypeNonNull ->
                    json.addProperty("mimeType", mimeTypeNonNull)
                }
                isEmpty?.let { isEmptyNonNull ->
                    json.addProperty("isEmpty", isEmptyNonNull)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): ImageWireframeUpdate {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ImageWireframeUpdate",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): ImageWireframeUpdate {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x")?.asLong
                        val y = jsonObject.get("y")?.asLong
                        val width = jsonObject.get("width")?.asLong
                        val height = jsonObject.get("height")?.asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val shapeStyle = jsonObject.get("shapeStyle")?.asJsonObject?.let {
                            ShapeStyle.fromJsonObject(it)
                        }
                        val border = jsonObject.get("border")?.asJsonObject?.let {
                            ShapeBorder.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        val base64 = jsonObject.get("base64")?.asString
                        val resourceId = jsonObject.get("resourceId")?.asString
                        val mimeType = jsonObject.get("mimeType")?.asString
                        val isEmpty = jsonObject.get("isEmpty")?.asBoolean
                        check(type == "image")
                        return ImageWireframeUpdate(id, x, y, width, height, clip, shapeStyle,
                                border, base64, resourceId, mimeType, isEmpty)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type ImageWireframeUpdate",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type ImageWireframeUpdate",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type ImageWireframeUpdate",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for WireframeUpdate events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         * @param label Label of the placeholder
         */
        public data class PlaceholderWireframeUpdate(
            public val id: Long,
            public val x: Long? = null,
            public val y: Long? = null,
            public val width: Long? = null,
            public val height: Long? = null,
            public val clip: WireframeClip? = null,
            public var label: String? = null,
        ) : WireframeUpdateMutation() {
            public val type: String = "placeholder"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                x?.let { xNonNull ->
                    json.addProperty("x", xNonNull)
                }
                y?.let { yNonNull ->
                    json.addProperty("y", yNonNull)
                }
                width?.let { widthNonNull ->
                    json.addProperty("width", widthNonNull)
                }
                height?.let { heightNonNull ->
                    json.addProperty("height", heightNonNull)
                }
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                json.addProperty("type", type)
                label?.let { labelNonNull ->
                    json.addProperty("label", labelNonNull)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): PlaceholderWireframeUpdate {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type PlaceholderWireframeUpdate",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): PlaceholderWireframeUpdate {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x")?.asLong
                        val y = jsonObject.get("y")?.asLong
                        val width = jsonObject.get("width")?.asLong
                        val height = jsonObject.get("height")?.asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        val label = jsonObject.get("label")?.asString
                        check(type == "placeholder")
                        return PlaceholderWireframeUpdate(id, x, y, width, height, clip, label)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type PlaceholderWireframeUpdate",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type PlaceholderWireframeUpdate",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type PlaceholderWireframeUpdate",
                            e
                        )
                    }
                }
            }
        }

        /**
         * Schema of common properties for WireframeUpdate events type.
         * @param id Defines the unique ID of the wireframe. This is persistent throughout the view
         * lifetime.
         * @param x The position in pixels on X axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param y The position in pixels on Y axis of the UI element in absolute coordinates. The
         * anchor point is always the top-left corner of the wireframe.
         * @param width The width in pixels of the UI element, normalized based on the device pixels
         * per inch density (DPI). Example: if a device has a DPI = 2, the width of all UI elements is
         * divided by 2 to get a normalized width.
         * @param height The height in pixels of the UI element, normalized based on the device
         * pixels per inch density (DPI). Example: if a device has a DPI = 2, the height of all UI
         * elements is divided by 2 to get a normalized height.
         * @param clip Schema of clipping information for a Wireframe.
         * @param slotId Unique Id of the slot containing this webview.
         * @param isVisible Whether this webview is visible or not.
         */
        public data class WebviewWireframeUpdate(
            public val id: Long,
            public val x: Long? = null,
            public val y: Long? = null,
            public val width: Long? = null,
            public val height: Long? = null,
            public val clip: WireframeClip? = null,
            public val shapeStyle: ShapeStyle? = null,
            public val border: ShapeBorder? = null,
            public val slotId: String,
            public val isVisible: Boolean? = null,
        ) : WireframeUpdateMutation() {
            public val type: String = "webview"

            override fun toJson(): JsonElement {
                val json = JsonObject()
                json.addProperty("id", id)
                x?.let { xNonNull ->
                    json.addProperty("x", xNonNull)
                }
                y?.let { yNonNull ->
                    json.addProperty("y", yNonNull)
                }
                width?.let { widthNonNull ->
                    json.addProperty("width", widthNonNull)
                }
                height?.let { heightNonNull ->
                    json.addProperty("height", heightNonNull)
                }
                clip?.let { clipNonNull ->
                    json.add("clip", clipNonNull.toJson())
                }
                shapeStyle?.let { shapeStyleNonNull ->
                    json.add("shapeStyle", shapeStyleNonNull.toJson())
                }
                border?.let { borderNonNull ->
                    json.add("border", borderNonNull.toJson())
                }
                json.addProperty("type", type)
                json.addProperty("slotId", slotId)
                isVisible?.let { isVisibleNonNull ->
                    json.addProperty("isVisible", isVisibleNonNull)
                }
                return json
            }

            public companion object {
                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJson(jsonString: String): WebviewWireframeUpdate {
                    try {
                        val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                        return fromJsonObject(jsonObject)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type WebviewWireframeUpdate",
                            e
                        )
                    }
                }

                @JvmStatic
                @Throws(JsonParseException::class)
                public fun fromJsonObject(jsonObject: JsonObject): WebviewWireframeUpdate {
                    try {
                        val id = jsonObject.get("id").asLong
                        val x = jsonObject.get("x")?.asLong
                        val y = jsonObject.get("y")?.asLong
                        val width = jsonObject.get("width")?.asLong
                        val height = jsonObject.get("height")?.asLong
                        val clip = jsonObject.get("clip")?.asJsonObject?.let {
                            WireframeClip.fromJsonObject(it)
                        }
                        val shapeStyle = jsonObject.get("shapeStyle")?.asJsonObject?.let {
                            ShapeStyle.fromJsonObject(it)
                        }
                        val border = jsonObject.get("border")?.asJsonObject?.let {
                            ShapeBorder.fromJsonObject(it)
                        }
                        val type = jsonObject.get("type").asString
                        val slotId = jsonObject.get("slotId").asString
                        val isVisible = jsonObject.get("isVisible")?.asBoolean
                        check(type == "webview")
                        return WebviewWireframeUpdate(id, x, y, width, height, clip, shapeStyle,
                                border, slotId, isVisible)
                    } catch (e: IllegalStateException) {
                        throw JsonParseException(
                            "Unable to parse json into type WebviewWireframeUpdate",
                            e
                        )
                    } catch (e: NumberFormatException) {
                        throw JsonParseException(
                            "Unable to parse json into type WebviewWireframeUpdate",
                            e
                        )
                    } catch (e: NullPointerException) {
                        throw JsonParseException(
                            "Unable to parse json into type WebviewWireframeUpdate",
                            e
                        )
                    }
                }
            }
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): WireframeUpdateMutation {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into one of type WireframeUpdateMutation",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): WireframeUpdateMutation {
                val errors = mutableListOf<Throwable>()
                val asTextWireframeUpdate = try {
                    TextWireframeUpdate.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asShapeWireframeUpdate = try {
                    ShapeWireframeUpdate.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asImageWireframeUpdate = try {
                    ImageWireframeUpdate.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asPlaceholderWireframeUpdate = try {
                    PlaceholderWireframeUpdate.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val asWebviewWireframeUpdate = try {
                    WebviewWireframeUpdate.fromJsonObject(jsonObject)
                } catch (e: JsonParseException) {
                    errors.add(e)
                    null
                }
                val result = arrayOf(
                    asTextWireframeUpdate,
                    asShapeWireframeUpdate,
                    asImageWireframeUpdate,
                    asPlaceholderWireframeUpdate,
                    asWebviewWireframeUpdate,
                ).firstOrNull { it != null }
                if (result == null) {
                    val message = "Unable to parse json into one of type \n" +
                            "WireframeUpdateMutation\n" +
                        errors.joinToString("\n") { it.message.toString() }
                    throw JsonParseException(message)
                }
                return result
            }
        }
    }

    /**
     * @param id The touch id of the touch event this position corresponds to. In mobile it is
     * possible to have multiple touch events (fingers touching the screen) happening at the same time.
     * @param x The x coordinate value of the position.
     * @param y The y coordinate value of the position.
     * @param timestamp The UTC timestamp in milliseconds corresponding to the moment the position
     * change was recorded. Each timestamp is computed as the UTC interval since 00:00:00.000
     * 01.01.1970.
     */
    public data class Position(
        public val id: Long,
        public val x: Long,
        public val y: Long,
        public val timestamp: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            json.addProperty("x", x)
            json.addProperty("y", y)
            json.addProperty("timestamp", timestamp)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Position {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Position",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Position {
                try {
                    val id = jsonObject.get("id").asLong
                    val x = jsonObject.get("x").asLong
                    val y = jsonObject.get("y").asLong
                    val timestamp = jsonObject.get("timestamp").asLong
                    return Position(id, x, y, timestamp)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Position",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Position",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Position",
                        e
                    )
                }
            }
        }
    }

    /**
     * Schema of clipping information for a Wireframe.
     * @param top The amount of space in pixels that needs to be clipped (masked) at the top of the
     * wireframe.
     * @param bottom The amount of space in pixels that needs to be clipped (masked) at the bottom
     * of the wireframe.
     * @param left The amount of space in pixels that needs to be clipped (masked) at the left of
     * the wireframe.
     * @param right The amount of space in pixels that needs to be clipped (masked) at the right of
     * the wireframe.
     */
    public data class WireframeClip(
        public val top: Long? = null,
        public val bottom: Long? = null,
        public val left: Long? = null,
        public val right: Long? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            top?.let { topNonNull ->
                json.addProperty("top", topNonNull)
            }
            bottom?.let { bottomNonNull ->
                json.addProperty("bottom", bottomNonNull)
            }
            left?.let { leftNonNull ->
                json.addProperty("left", leftNonNull)
            }
            right?.let { rightNonNull ->
                json.addProperty("right", rightNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): WireframeClip {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type WireframeClip",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): WireframeClip {
                try {
                    val top = jsonObject.get("top")?.asLong
                    val bottom = jsonObject.get("bottom")?.asLong
                    val left = jsonObject.get("left")?.asLong
                    val right = jsonObject.get("right")?.asLong
                    return WireframeClip(top, bottom, left, right)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type WireframeClip",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type WireframeClip",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type WireframeClip",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param backgroundColor The background color for this wireframe as a String hexadecimal.
     * Follows the #RRGGBBAA color format with the alpha value as optional. The default value is
     * #FFFFFF00.
     * @param opacity The opacity of this wireframe. Takes values from 0 to 1, default value is 1.
     * @param cornerRadius The corner(border) radius of this wireframe in pixels. The default value
     * is 0.
     */
    public data class ShapeStyle(
        public val backgroundColor: String? = null,
        public val opacity: Number? = null,
        public val cornerRadius: Number? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            backgroundColor?.let { backgroundColorNonNull ->
                json.addProperty("backgroundColor", backgroundColorNonNull)
            }
            opacity?.let { opacityNonNull ->
                json.addProperty("opacity", opacityNonNull)
            }
            cornerRadius?.let { cornerRadiusNonNull ->
                json.addProperty("cornerRadius", cornerRadiusNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ShapeStyle {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ShapeStyle",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ShapeStyle {
                try {
                    val backgroundColor = jsonObject.get("backgroundColor")?.asString
                    val opacity = jsonObject.get("opacity")?.asNumber
                    val cornerRadius = jsonObject.get("cornerRadius")?.asNumber
                    return ShapeStyle(backgroundColor, opacity, cornerRadius)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ShapeStyle",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ShapeStyle",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ShapeStyle",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param color The border color as a String hexadecimal. Follows the #RRGGBBAA color format
     * with the alpha value as optional.
     * @param width The width of the border in pixels.
     */
    public data class ShapeBorder(
        public val color: String,
        public val width: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("color", color)
            json.addProperty("width", width)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ShapeBorder {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ShapeBorder",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ShapeBorder {
                try {
                    val color = jsonObject.get("color").asString
                    val width = jsonObject.get("width").asLong
                    return ShapeBorder(color, width)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ShapeBorder",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ShapeBorder",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ShapeBorder",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param family The preferred font family collection, ordered by preference and formatted as a
     * String list: e.g. Century Gothic, Verdana, sans-serif
     * @param size The font size in pixels.
     * @param color The font color as a string hexadecimal. Follows the #RRGGBBAA color format with
     * the alpha value as optional.
     */
    public data class TextStyle(
        public val family: String,
        public val size: Long,
        public val color: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("family", family)
            json.addProperty("size", size)
            json.addProperty("color", color)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): TextStyle {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type TextStyle",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): TextStyle {
                try {
                    val family = jsonObject.get("family").asString
                    val size = jsonObject.get("size").asLong
                    val color = jsonObject.get("color").asString
                    return TextStyle(family, size, color)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type TextStyle",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type TextStyle",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type TextStyle",
                        e
                    )
                }
            }
        }
    }

    public data class TextPosition(
        public val padding: Padding? = null,
        public val alignment: Alignment? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            padding?.let { paddingNonNull ->
                json.add("padding", paddingNonNull.toJson())
            }
            alignment?.let { alignmentNonNull ->
                json.add("alignment", alignmentNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): TextPosition {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type TextPosition",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): TextPosition {
                try {
                    val padding = jsonObject.get("padding")?.asJsonObject?.let {
                        Padding.fromJsonObject(it)
                    }
                    val alignment = jsonObject.get("alignment")?.asJsonObject?.let {
                        Alignment.fromJsonObject(it)
                    }
                    return TextPosition(padding, alignment)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type TextPosition",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type TextPosition",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type TextPosition",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param top The top padding in pixels. The default value is 0.
     * @param bottom The bottom padding in pixels. The default value is 0.
     * @param left The left padding in pixels. The default value is 0.
     * @param right The right padding in pixels. The default value is 0.
     */
    public data class Padding(
        public val top: Long? = null,
        public val bottom: Long? = null,
        public val left: Long? = null,
        public val right: Long? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            top?.let { topNonNull ->
                json.addProperty("top", topNonNull)
            }
            bottom?.let { bottomNonNull ->
                json.addProperty("bottom", bottomNonNull)
            }
            left?.let { leftNonNull ->
                json.addProperty("left", leftNonNull)
            }
            right?.let { rightNonNull ->
                json.addProperty("right", rightNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Padding {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Padding",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Padding {
                try {
                    val top = jsonObject.get("top")?.asLong
                    val bottom = jsonObject.get("bottom")?.asLong
                    val left = jsonObject.get("left")?.asLong
                    val right = jsonObject.get("right")?.asLong
                    return Padding(top, bottom, left, right)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Padding",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Padding",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Padding",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param horizontal The horizontal text alignment. The default value is `left`.
     * @param vertical The vertical text alignment. The default value is `top`.
     */
    public data class Alignment(
        public val horizontal: Horizontal? = null,
        public val vertical: Vertical? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            horizontal?.let { horizontalNonNull ->
                json.add("horizontal", horizontalNonNull.toJson())
            }
            vertical?.let { verticalNonNull ->
                json.add("vertical", verticalNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Alignment {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Alignment",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Alignment {
                try {
                    val horizontal = jsonObject.get("horizontal")?.asString?.let {
                        Horizontal.fromJson(it)
                    }
                    val vertical = jsonObject.get("vertical")?.asString?.let {
                        Vertical.fromJson(it)
                    }
                    return Alignment(horizontal, vertical)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Alignment",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Alignment",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Alignment",
                        e
                    )
                }
            }
        }
    }

    /**
     * The source of this record
     */
    public enum class Source(
        private val jsonValue: String,
    ) {
        ANDROID("android"),
        IOS("ios"),
        FLUTTER("flutter"),
        REACT_NATIVE("react-native"),
        KOTLIN_MULTIPLATFORM("kotlin-multiplatform"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Source = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Schema of an PointerEventType
     */
    public enum class PointerEventType(
        private val jsonValue: String,
    ) {
        DOWN("down"),
        UP("up"),
        MOVE("move"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): PointerEventType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Schema of an PointerType
     */
    public enum class PointerType(
        private val jsonValue: String,
    ) {
        MOUSE("mouse"),
        TOUCH("touch"),
        PEN("pen"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): PointerType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * The horizontal text alignment. The default value is `left`.
     */
    public enum class Horizontal(
        private val jsonValue: String,
    ) {
        LEFT("left"),
        RIGHT("right"),
        CENTER("center"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Horizontal = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * The vertical text alignment. The default value is `top`.
     */
    public enum class Vertical(
        private val jsonValue: String,
    ) {
        TOP("top"),
        BOTTOM("bottom"),
        CENTER("center"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Vertical = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
