package com.datadog.android.sessionreplay.model

import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Number
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of all properties of a ResourceHashesEntry.
 * @param lastUpdateDateNs The last time we updated the date for the datastore.
 * @param resourceHashes The collection of resource hashes.
 */
public data class ResourceHashesEntry(
    public val lastUpdateDateNs: Number,
    public val resourceHashes: List<String>,
) {
    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.addProperty("last_update_date_ns", lastUpdateDateNs)
        val resourceHashesArray = JsonArray(resourceHashes.size)
        resourceHashes.forEach { resourceHashesArray.add(it) }
        json.add("resource_hashes", resourceHashesArray)
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): ResourceHashesEntry {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type ResourceHashesEntry",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): ResourceHashesEntry {
            try {
                val lastUpdateDateNs = jsonObject.get("last_update_date_ns").asNumber
                val resourceHashes =
                        jsonObject.get("resource_hashes").asJsonArray.let { jsonArray ->
                    val collection = ArrayList<String>(jsonArray.size())
                    jsonArray.forEach {
                        collection.add(it.asString)
                    }
                    collection
                }
                return ResourceHashesEntry(lastUpdateDateNs, resourceHashes)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type ResourceHashesEntry",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type ResourceHashesEntry",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type ResourceHashesEntry",
                    e
                )
            }
        }
    }
}
