package com.datadog.android.trace.model

import com.datadog.android.core.`internal`.utils.JsonSerializer
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Any
import kotlin.Array
import kotlin.Long
import kotlin.Number
import kotlin.String
import kotlin.collections.Map
import kotlin.collections.MutableMap
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Structure holding information about a Span
 * @param traceId The id of the trace this Span belongs to
 * @param spanId The unique id of this Span
 * @param parentId The id this Span's parent or 0 if this is the root Span
 * @param resource The resource name
 * @param name The name of this Span
 * @param service The service name
 * @param duration The duration of this Span in nanoseconds
 * @param start The Span start time in nanoseconds
 * @param error Span error flag. If 1 that means there was an error thrown during the duration of
 * the Span
 * @param metrics The metrics data of this Span event
 * @param meta The metadata of this Span event
 */
public data class SpanEvent(
    public val traceId: String,
    public val spanId: String,
    public val parentId: String,
    public var resource: String,
    public var name: String,
    public val service: String,
    public val duration: Long,
    public val start: Long,
    public val error: Long = 0L,
    public val metrics: Metrics,
    public val meta: Meta,
) {
    public val type: String = "custom"

    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.addProperty("trace_id", traceId)
        json.addProperty("span_id", spanId)
        json.addProperty("parent_id", parentId)
        json.addProperty("resource", resource)
        json.addProperty("name", name)
        json.addProperty("service", service)
        json.addProperty("duration", duration)
        json.addProperty("start", start)
        json.addProperty("error", error)
        json.addProperty("type", type)
        json.add("metrics", metrics.toJson())
        json.add("meta", meta.toJson())
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): SpanEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type SpanEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): SpanEvent {
            try {
                val traceId = jsonObject.get("trace_id").asString
                val spanId = jsonObject.get("span_id").asString
                val parentId = jsonObject.get("parent_id").asString
                val resource = jsonObject.get("resource").asString
                val name = jsonObject.get("name").asString
                val service = jsonObject.get("service").asString
                val duration = jsonObject.get("duration").asLong
                val start = jsonObject.get("start").asLong
                val error = jsonObject.get("error").asLong
                val type = jsonObject.get("type").asString
                val metrics = jsonObject.get("metrics").asJsonObject.let {
                    Metrics.fromJsonObject(it)
                }
                val meta = jsonObject.get("meta").asJsonObject.let {
                    Meta.fromJsonObject(it)
                }
                check(type == "custom")
                return SpanEvent(traceId, spanId, parentId, resource, name, service, duration,
                        start, error, metrics, meta)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type SpanEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type SpanEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type SpanEvent",
                    e
                )
            }
        }
    }

    /**
     * The metrics data of this Span event
     * @param topLevel Top level flag. If 1 means that this Span is the root of this Trace
     */
    public data class Metrics(
        public val topLevel: Long? = null,
        public val additionalProperties: Map<String, Number> = mapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            topLevel?.let { topLevelNonNull ->
                json.addProperty("_top_level", topLevelNonNull)
            }
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.addProperty(k, v)
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("_top_level")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Metrics {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Metrics",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Metrics {
                try {
                    val topLevel = jsonObject.get("_top_level")?.asLong
                    val additionalProperties = mutableMapOf<String, Number>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value.asNumber
                        }
                    }
                    return Metrics(topLevel, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Metrics",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Metrics",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Metrics",
                        e
                    )
                }
            }
        }
    }

    /**
     * The metadata of this Span event
     * @param version The client application package version
     * @param usr User properties
     * @param network The network information in the moment the Span was created
     * @param device Device properties
     * @param os Operating system properties
     */
    public data class Meta(
        public val version: String,
        public val dd: Dd,
        public val span: Span,
        public val tracer: Tracer,
        public val usr: Usr,
        public val network: Network? = null,
        public val device: Device,
        public val os: Os,
        public val additionalProperties: Map<String, String> = mapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("version", version)
            json.add("_dd", dd.toJson())
            json.add("span", span.toJson())
            json.add("tracer", tracer.toJson())
            json.add("usr", usr.toJson())
            network?.let { networkNonNull ->
                json.add("network", networkNonNull.toJson())
            }
            json.add("device", device.toJson())
            json.add("os", os.toJson())
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.addProperty(k, v)
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("version", "_dd", "span",
                    "tracer", "usr", "network", "device", "os")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Meta {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Meta",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Meta {
                try {
                    val version = jsonObject.get("version").asString
                    val dd = jsonObject.get("_dd").asJsonObject.let {
                        Dd.fromJsonObject(it)
                    }
                    val span = jsonObject.get("span").asJsonObject.let {
                        Span.fromJsonObject(it)
                    }
                    val tracer = jsonObject.get("tracer").asJsonObject.let {
                        Tracer.fromJsonObject(it)
                    }
                    val usr = jsonObject.get("usr").asJsonObject.let {
                        Usr.fromJsonObject(it)
                    }
                    val network = jsonObject.get("network")?.asJsonObject?.let {
                        Network.fromJsonObject(it)
                    }
                    val device = jsonObject.get("device").asJsonObject.let {
                        Device.fromJsonObject(it)
                    }
                    val os = jsonObject.get("os").asJsonObject.let {
                        Os.fromJsonObject(it)
                    }
                    val additionalProperties = mutableMapOf<String, String>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value.asString
                        }
                    }
                    return Meta(version, dd, span, tracer, usr, network, device, os,
                            additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Meta",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Meta",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Meta",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param source The trace source
     * @param application The RUM Application attributes
     * @param session The active RUM Session attributes
     * @param view The active RUM View attributes
     */
    public data class Dd(
        public val source: String? = "android",
        public val application: Application? = null,
        public val session: Session? = null,
        public val view: View? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            source?.let { sourceNonNull ->
                json.addProperty("source", sourceNonNull)
            }
            application?.let { applicationNonNull ->
                json.add("application", applicationNonNull.toJson())
            }
            session?.let { sessionNonNull ->
                json.add("session", sessionNonNull.toJson())
            }
            view?.let { viewNonNull ->
                json.add("view", viewNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dd {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dd {
                try {
                    val source = jsonObject.get("source")?.asString
                    val application = jsonObject.get("application")?.asJsonObject?.let {
                        Application.fromJsonObject(it)
                    }
                    val session = jsonObject.get("session")?.asJsonObject?.let {
                        Session.fromJsonObject(it)
                    }
                    val view = jsonObject.get("view")?.asJsonObject?.let {
                        View.fromJsonObject(it)
                    }
                    return Dd(source, application, session, view)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }
        }
    }

    public class Span {
        public val kind: String = "client"

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("kind", kind)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Span {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Span",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Span {
                try {
                    val kind = jsonObject.get("kind")?.asString
                    if (kind != null) {
                        check(kind == "client")
                    }
                    return Span()
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Span",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Span",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Span",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param version The SDK version name
     */
    public data class Tracer(
        public val version: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("version", version)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Tracer {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Tracer",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Tracer {
                try {
                    val version = jsonObject.get("version").asString
                    return Tracer(version)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Tracer",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Tracer",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Tracer",
                        e
                    )
                }
            }
        }
    }

    /**
     * User properties
     * @param id Identifier of the user
     * @param name Name of the user
     * @param email Email of the user
     */
    public data class Usr(
        public val id: String? = null,
        public val name: String? = null,
        public val email: String? = null,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            email?.let { emailNonNull ->
                json.addProperty("email", emailNonNull)
            }
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("id", "name", "email")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Usr {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Usr {
                try {
                    val id = jsonObject.get("id")?.asString
                    val name = jsonObject.get("name")?.asString
                    val email = jsonObject.get("email")?.asString
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    return Usr(id, name, email, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }
        }
    }

    /**
     * The network information in the moment the Span was created
     */
    public data class Network(
        public val client: Client? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            client?.let { clientNonNull ->
                json.add("client", clientNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Network {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Network",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Network {
                try {
                    val client = jsonObject.get("client")?.asJsonObject?.let {
                        Client.fromJsonObject(it)
                    }
                    return Network(client)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Network",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Network",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Network",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device properties
     * @param type Device type info
     * @param name Device marketing name, e.g. Xiaomi Redmi Note 8 Pro, Pixel 5, etc.
     * @param model Device SKU model, e.g. Samsung SM-988GN, etc. Quite often name and model can be
     * the same.
     * @param brand Device marketing brand, e.g. Apple, OPPO, Xiaomi, etc.
     * @param architecture The CPU architecture of the device that is reporting the error
     */
    public data class Device(
        public val type: Type,
        public val name: String? = null,
        public val model: String? = null,
        public val brand: String? = null,
        public val architecture: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("type", type.toJson())
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            model?.let { modelNonNull ->
                json.addProperty("model", modelNonNull)
            }
            brand?.let { brandNonNull ->
                json.addProperty("brand", brandNonNull)
            }
            architecture?.let { architectureNonNull ->
                json.addProperty("architecture", architectureNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Device {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Device {
                try {
                    val type = Type.fromJson(jsonObject.get("type").asString)
                    val name = jsonObject.get("name")?.asString
                    val model = jsonObject.get("model")?.asString
                    val brand = jsonObject.get("brand")?.asString
                    val architecture = jsonObject.get("architecture")?.asString
                    return Device(type, name, model, brand, architecture)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }
        }
    }

    /**
     * Operating system properties
     * @param name Operating system name, e.g. Android, iOS
     * @param version Full operating system version, e.g. 8.1.1
     * @param build Operating system build number, e.g. 15D21
     * @param versionMajor Major operating system version, e.g. 8
     */
    public data class Os(
        public val name: String,
        public val version: String,
        public val build: String? = null,
        public val versionMajor: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("name", name)
            json.addProperty("version", version)
            build?.let { buildNonNull ->
                json.addProperty("build", buildNonNull)
            }
            json.addProperty("version_major", versionMajor)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Os {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Os {
                try {
                    val name = jsonObject.get("name").asString
                    val version = jsonObject.get("version").asString
                    val build = jsonObject.get("build")?.asString
                    val versionMajor = jsonObject.get("version_major").asString
                    return Os(name, version, build, versionMajor)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }
        }
    }

    /**
     * The RUM Application attributes
     * @param id RUM Application ID
     */
    public data class Application(
        public val id: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id")?.asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * The active RUM Session attributes
     * @param id The RUM Session ID
     */
    public data class Session(
        public val id: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Session {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Session {
                try {
                    val id = jsonObject.get("id")?.asString
                    return Session(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }
        }
    }

    /**
     * The active RUM View attributes
     * @param id The RUM View ID
     */
    public data class View(
        public val id: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): View {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): View {
                try {
                    val id = jsonObject.get("id")?.asString
                    return View(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param connectivity The active network
     */
    public data class Client(
        public val simCarrier: SimCarrier? = null,
        public val signalStrength: String? = null,
        public val downlinkKbps: String? = null,
        public val uplinkKbps: String? = null,
        public val connectivity: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            simCarrier?.let { simCarrierNonNull ->
                json.add("sim_carrier", simCarrierNonNull.toJson())
            }
            signalStrength?.let { signalStrengthNonNull ->
                json.addProperty("signal_strength", signalStrengthNonNull)
            }
            downlinkKbps?.let { downlinkKbpsNonNull ->
                json.addProperty("downlink_kbps", downlinkKbpsNonNull)
            }
            uplinkKbps?.let { uplinkKbpsNonNull ->
                json.addProperty("uplink_kbps", uplinkKbpsNonNull)
            }
            connectivity?.let { connectivityNonNull ->
                json.addProperty("connectivity", connectivityNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Client {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Client",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Client {
                try {
                    val simCarrier = jsonObject.get("sim_carrier")?.asJsonObject?.let {
                        SimCarrier.fromJsonObject(it)
                    }
                    val signalStrength = jsonObject.get("signal_strength")?.asString
                    val downlinkKbps = jsonObject.get("downlink_kbps")?.asString
                    val uplinkKbps = jsonObject.get("uplink_kbps")?.asString
                    val connectivity = jsonObject.get("connectivity")?.asString
                    return Client(simCarrier, signalStrength, downlinkKbps, uplinkKbps,
                            connectivity)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Client",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Client",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Client",
                        e
                    )
                }
            }
        }
    }

    public data class SimCarrier(
        public val id: String? = null,
        public val name: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): SimCarrier {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type SimCarrier",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): SimCarrier {
                try {
                    val id = jsonObject.get("id")?.asString
                    val name = jsonObject.get("name")?.asString
                    return SimCarrier(id, name)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type SimCarrier",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type SimCarrier",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type SimCarrier",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device type info
     */
    public enum class Type(
        private val jsonValue: String,
    ) {
        MOBILE("mobile"),
        DESKTOP("desktop"),
        TABLET("tablet"),
        TV("tv"),
        GAMING_CONSOLE("gaming_console"),
        BOT("bot"),
        OTHER("other"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Type = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
