/*
 * Decompiled with CFR 0.152.
 */
package datadog.trace.util;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class ClassNameTrie {
    private static final char LEAF_MARKER = '\u8000';
    private static final char BUD_MARKER = '\u4000';
    private static final char GLOB_MARKER = '\u2000';
    private static final char MAX_NODE_VALUE = '\u1fff';
    private static final char LONG_JUMP_MARKER = '\u8000';
    private static final int[] NO_LONG_JUMPS = new int[0];
    private final char[] trieData;
    private final int[] longJumps;

    public int apply(String key) {
        char[] data = this.trieData;
        int keyLength = key.length();
        int keyIndex = 0;
        int dataIndex = 0;
        int result = -1;
        while (keyIndex < keyLength) {
            char c;
            char branchCount;
            int branchIndex;
            if ((branchIndex = Arrays.binarySearch(data, dataIndex, dataIndex + (branchCount = data[dataIndex++]), (c = key.charAt(keyIndex++)) == '/' ? (char)'.' : (char)c)) < 0) {
                return result;
            }
            int valueIndex = branchIndex + branchCount;
            char value = data[valueIndex];
            char segmentLength = '\u0000';
            if ((value & 0xC000) != 0) {
                if (keyIndex == keyLength || (value & 0x2000) != 0) {
                    result = value & 0x1FFF;
                }
                if (keyIndex == keyLength || (value & 0x8000) != 0) {
                    return result;
                }
            } else {
                segmentLength = value;
            }
            if (branchIndex > dataIndex) {
                int jumpIndex = valueIndex + branchCount - 1;
                int nextJump = data[jumpIndex];
                if ((nextJump & 0x8000) != 0) {
                    nextJump = this.longJumps[nextJump & 0xFFFF7FFF];
                }
                dataIndex += nextJump;
            }
            dataIndex += branchCount * 3 - 1;
            if (segmentLength <= '\u0000') continue;
            if (keyLength - keyIndex < segmentLength) {
                return result;
            }
            int segmentEnd = dataIndex + segmentLength;
            while (dataIndex < segmentEnd) {
                if (((c = key.charAt(keyIndex++)) == '/' ? (char)'.' : (char)c) == data[dataIndex++]) continue;
                return result;
            }
            value = data[dataIndex];
            if ((value & 0x8000) == 0) continue;
            if (keyIndex == keyLength || (value & 0x2000) != 0) {
                result = value & 0x1FFF;
            }
            return result;
        }
        return result;
    }

    public static ClassNameTrie create(String trieData) {
        return ClassNameTrie.create(trieData, NO_LONG_JUMPS);
    }

    public static ClassNameTrie create(String[] trieData) {
        return ClassNameTrie.create(trieData, NO_LONG_JUMPS);
    }

    public static ClassNameTrie create(String trieData, int[] longJumps) {
        return new ClassNameTrie(trieData.toCharArray(), longJumps);
    }

    public static ClassNameTrie create(String[] trieData, int[] longJumps) {
        int dataLength = 0;
        for (String chunk : trieData) {
            dataLength += chunk.length();
        }
        char[] data = new char[dataLength];
        int dataIndex = 0;
        for (String chunk : trieData) {
            int chunkLength = chunk.length();
            System.arraycopy(chunk.toCharArray(), 0, data, dataIndex, chunkLength);
            dataIndex += chunkLength;
        }
        return new ClassNameTrie(data, longJumps);
    }

    private ClassNameTrie(char[] trieData, int[] longJumps) {
        this.trieData = trieData;
        this.longJumps = longJumps;
    }

    public static class Generator {
        private static final Pattern MAPPING_LINE = Pattern.compile("^\\s*([0-9]+)\\s+([^\\s#]+)");

        public static void main(String[] args) throws IOException {
            if (args.length < 2) {
                throw new IllegalArgumentException("Expected: trie-dir java-dir [file.trie ...]");
            }
            Path trieDir = Paths.get(args[0], new String[0]).toAbsolutePath().normalize();
            if (!Files.isDirectory(trieDir, new LinkOption[0])) {
                throw new IllegalArgumentException("Bad trie directory: " + trieDir);
            }
            Path javaDir = Paths.get(args[1], new String[0]).toAbsolutePath().normalize();
            if (!Files.isDirectory(javaDir, new LinkOption[0])) {
                throw new IllegalArgumentException("Bad java directory: " + javaDir);
            }
            for (int i = 2; i < args.length; ++i) {
                Path triePath = trieDir.resolve(args[i]).normalize();
                String className = Generator.toClassName(triePath.getFileName().toString());
                Path pkgPath = trieDir.relativize(triePath.getParent());
                String pkgName = pkgPath.toString().replace(File.separatorChar, '.');
                Path javaPath = javaDir.resolve(pkgPath).resolve(className + ".java");
                Generator.generateJavaFile(triePath, javaPath, pkgName, className);
            }
        }

        private static String toClassName(String trieName) {
            StringBuilder className = new StringBuilder();
            boolean upperNext = true;
            for (int i = 0; i < trieName.length(); ++i) {
                char c = trieName.charAt(i);
                if (c == '_' | c == '.') {
                    upperNext = true;
                    continue;
                }
                className.append(upperNext ? Character.toUpperCase(c) : c);
                upperNext = false;
            }
            return className.toString();
        }

        private static void generateJavaFile(Path triePath, Path javaPath, String pkgName, String className) throws IOException {
            Builder builder = new Builder();
            for (String l : Files.readAllLines(triePath, StandardCharsets.UTF_8)) {
                Matcher m = MAPPING_LINE.matcher(l);
                if (!m.find()) continue;
                builder.put(m.group(2), Integer.parseInt(m.group(1)));
            }
            ArrayList<String> lines = new ArrayList<String>();
            if (!pkgName.isEmpty()) {
                lines.add("package " + pkgName + ';');
            }
            lines.add("");
            lines.add("import datadog.trace.util.ClassNameTrie;");
            lines.add("");
            lines.add("// Generated from '" + triePath.getFileName() + "' - DO NOT EDIT!");
            lines.add("public final class " + className + " {");
            lines.add("  public static int apply(String key) {");
            lines.add("    return TRIE.apply(key);");
            lines.add("  }");
            lines.add("");
            Generator.generateJavaTrie(lines, "", builder.buildTrie());
            lines.add("  private " + className + "() {}");
            lines.add("}");
            Files.write(javaPath, lines, StandardCharsets.UTF_8, new OpenOption[0]);
        }

        public static void generateJavaTrie(List<String> lines, String prefix, ClassNameTrie trie) {
            boolean hasLongJumps = trie.longJumps.length > 0;
            int firstLineNumber = lines.size();
            int chunk = 1;
            lines.add("  private static final String " + prefix + "TRIE_DATA_" + chunk + " =");
            int chunkSize = 0;
            StringBuilder buf = new StringBuilder();
            buf.append("      \"");
            for (char c : trie.trieData) {
                if (++chunkSize > 10000) {
                    chunkSize = 0;
                    lines.add(buf + "\";");
                    lines.add("  private static final String " + prefix + "TRIE_DATA_" + ++chunk + " =");
                    buf.setLength(0);
                    buf.append("      \"");
                } else if (buf.length() > 120) {
                    lines.add(buf + "\"");
                    buf.setLength(0);
                    buf.append("          + \"");
                }
                if (c <= '\u00ff') {
                    buf.append(String.format("\\%03o", c));
                    continue;
                }
                buf.append(String.format("\\u%04x", c));
            }
            lines.add(buf + "\";");
            lines.add("");
            if (chunk > 1) {
                lines.add("  private static final String[] " + prefix + "TRIE_DATA = {");
                for (int n = 1; n < chunk; ++n) {
                    lines.add("    TRIE_DATA_" + n + ',');
                }
                lines.add("  };");
                lines.add("");
            } else {
                lines.set(firstLineNumber, "  private static final String " + prefix + "TRIE_DATA =");
            }
            if (hasLongJumps) {
                lines.add("  private static final int[] " + prefix + "LONG_JUMPS = {");
                buf.setLength(0);
                buf.append("   ");
                for (int j : trie.longJumps) {
                    if (buf.length() > 90) {
                        lines.add(buf.toString());
                        buf.setLength(0);
                        buf.append("   ");
                    }
                    buf.append(' ').append(String.format("0x%06X", j)).append(',');
                }
                lines.add(buf.toString());
                lines.add("  };");
                lines.add("");
            }
            if (hasLongJumps) {
                lines.add("  private static final ClassNameTrie " + prefix + "TRIE = ClassNameTrie.create(" + prefix + "TRIE_DATA, " + prefix + "LONG_JUMPS);");
            } else {
                lines.add("  private static final ClassNameTrie " + prefix + "TRIE = ClassNameTrie.create(" + prefix + "TRIE_DATA);");
            }
            lines.add("");
        }
    }

    public static class Builder {
        private final List<String> keys = new ArrayList<String>();
        private final StringBuilder values = new StringBuilder();
        private final StringBuilder buf = new StringBuilder();
        private int[] longJumps = new int[0];

        public void put(String className, int number) {
            char value;
            String key;
            if (number < 0) {
                throw new IllegalArgumentException("Number for " + className + " is negative: " + number);
            }
            if (number > 8191) {
                throw new IllegalArgumentException("Number for " + className + " is too big: " + number);
            }
            if (className.charAt(className.length() - 1) == '*') {
                key = className.substring(0, className.length() - 1);
                value = (char)(number | 0x2000);
            } else {
                key = className;
                value = (char)number;
            }
            int index = ~Collections.binarySearch(this.keys, key);
            if (index >= 0) {
                this.keys.add(index, key);
                this.values.insert(index, value);
            }
        }

        public ClassNameTrie buildTrie() {
            this.buildSubTrie(0, 0, this.keys.size());
            char[] data = new char[this.buf.length()];
            this.buf.getChars(0, data.length, data, 0);
            return new ClassNameTrie(data, this.longJumps);
        }

        private void buildSubTrie(int column, int row, int rowLimit) {
            int trieStart = this.buf.length();
            int prevRow = row;
            int branchCount = 0;
            int nextJump = 0;
            while (prevRow < rowLimit) {
                String key = this.keys.get(prevRow);
                int columnLimit = key.length();
                char pivot = key.charAt(column);
                int nextRow = this.nextPivotRow(pivot, column, prevRow, rowLimit);
                int nextColumn = this.nextPivotColumn(column, prevRow, nextRow);
                if (nextColumn == columnLimit && nextColumn - column > 1 && nextRow - prevRow > 1) {
                    --nextColumn;
                }
                int branchIndex = trieStart + branchCount;
                this.buf.insert(branchIndex, pivot);
                int valueIndex = branchIndex + 1 + branchCount;
                int subTrieStart = this.buf.length() + 1;
                if (nextColumn < columnLimit) {
                    if (nextColumn - column > 1) {
                        String segment = key.substring(column + 1, nextColumn);
                        this.buf.insert(valueIndex, (char)segment.length());
                        this.buf.append(segment);
                    } else {
                        this.buf.insert(valueIndex, '\u0000');
                    }
                    this.buildSubTrie(nextColumn, prevRow, nextRow);
                } else {
                    this.buildSubTrie(nextColumn, prevRow + 1, nextRow);
                    if (subTrieStart > this.buf.length()) {
                        if (nextColumn - column > 1) {
                            String segment = key.substring(column + 1, nextColumn);
                            this.buf.insert(valueIndex, (char)segment.length());
                            this.buf.append(segment);
                            this.buf.append((char)(this.values.charAt(prevRow) | 0x8000));
                        } else {
                            this.buf.insert(valueIndex, (char)(this.values.charAt(prevRow) | 0x8000));
                        }
                    } else {
                        this.buf.insert(valueIndex, (char)(this.values.charAt(prevRow) | 0x4000));
                    }
                }
                if (nextRow < rowLimit) {
                    int jumpIndex = valueIndex + 1 + branchCount;
                    if ((nextJump += this.buf.length() - subTrieStart) >= 32768) {
                        int longJumpIndex = this.longJumps.length;
                        this.longJumps = Arrays.copyOf(this.longJumps, longJumpIndex + 1);
                        this.longJumps[longJumpIndex] = nextJump;
                        this.buf.insert(jumpIndex, (char)(0x8000 | longJumpIndex));
                    } else {
                        this.buf.insert(jumpIndex, (char)nextJump);
                    }
                }
                prevRow = nextRow;
                ++branchCount;
            }
            if (branchCount > 0) {
                this.buf.insert(trieStart, (char)branchCount);
            }
        }

        private int nextPivotRow(char pivot, int column, int row, int rowLimit) {
            for (int r = row + 1; r < rowLimit; ++r) {
                String key = this.keys.get(r);
                if (key.length() > column && key.charAt(column) == pivot) continue;
                return r;
            }
            return rowLimit;
        }

        private int nextPivotColumn(int column, int row, int rowLimit) {
            String key = this.keys.get(row);
            int columnLimit = key.length();
            for (int c = column + 1; c < columnLimit; ++c) {
                if (this.nextPivotRow(key.charAt(c), c, row, rowLimit) >= rowLimit) continue;
                return c;
            }
            return columnLimit;
        }
    }
}

