/*
 * Decompiled with CFR 0.152.
 */
package datadog.trace.common.writer;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.util.concurrent.RateLimiter;
import datadog.opentracing.DDSpan;
import datadog.opentracing.DDTraceOTInfo;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.msgpack.jackson.dataformat.MessagePackFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DDApi {
    private static final Logger log = LoggerFactory.getLogger(DDApi.class);
    private static final String DATADOG_META_LANG = "Datadog-Meta-Lang";
    private static final String DATADOG_META_LANG_VERSION = "Datadog-Meta-Lang-Version";
    private static final String DATADOG_META_LANG_INTERPRETER = "Datadog-Meta-Lang-Interpreter";
    private static final String DATADOG_META_TRACER_VERSION = "Datadog-Meta-Tracer-Version";
    private static final String X_DATADOG_TRACE_COUNT = "X-Datadog-Trace-Count";
    private static final String TRACES_ENDPOINT_V3 = "/v0.3/traces";
    private static final String TRACES_ENDPOINT_V4 = "/v0.4/traces";
    private static final long SECONDS_BETWEEN_ERROR_LOG = TimeUnit.MINUTES.toSeconds(5L);
    private final String tracesEndpoint;
    private final List<ResponseListener> responseListeners = new ArrayList<ResponseListener>();
    private AtomicInteger traceCount;
    private final RateLimiter loggingRateLimiter = RateLimiter.create((double)(1.0 / (double)SECONDS_BETWEEN_ERROR_LOG));
    private static final ObjectMapper objectMapper = new ObjectMapper((JsonFactory)new MessagePackFactory());

    public DDApi(String host, int port) {
        this(host, port, DDApi.traceEndpointAvailable("http://" + host + ":" + port + TRACES_ENDPOINT_V4));
    }

    DDApi(String host, int port, boolean v4EndpointsAvailable) {
        if (v4EndpointsAvailable) {
            this.tracesEndpoint = "http://" + host + ":" + port + TRACES_ENDPOINT_V4;
        } else {
            log.debug("API v0.4 endpoints not available. Downgrading to v0.3");
            this.tracesEndpoint = "http://" + host + ":" + port + TRACES_ENDPOINT_V3;
        }
    }

    public void addResponseListener(ResponseListener listener) {
        if (!this.responseListeners.contains(listener)) {
            this.responseListeners.add(listener);
        }
    }

    public void addTraceCounter(AtomicInteger traceCount) {
        this.traceCount = traceCount;
    }

    public boolean sendTraces(List<List<DDSpan>> traces) {
        int totalSize = this.traceCount == null ? traces.size() : this.traceCount.getAndSet(0);
        try {
            HttpURLConnection httpCon = DDApi.getHttpURLConnection(this.tracesEndpoint);
            httpCon.setRequestProperty(X_DATADOG_TRACE_COUNT, String.valueOf(totalSize));
            OutputStream out = httpCon.getOutputStream();
            objectMapper.writeValue(out, traces);
            out.flush();
            out.close();
            String responseString = null;
            BufferedReader responseReader = new BufferedReader(new InputStreamReader(httpCon.getInputStream()));
            StringBuilder sb = new StringBuilder();
            String line = null;
            while ((line = responseReader.readLine()) != null) {
                sb.append(line);
            }
            responseReader.close();
            responseString = sb.toString();
            int responseCode = httpCon.getResponseCode();
            if (responseCode != 200) {
                if (log.isDebugEnabled()) {
                    log.debug("Error while sending {} of {} traces to the DD agent. Status: {}, ResponseMessage: ", new Object[]{traces.size(), totalSize, responseCode, httpCon.getResponseMessage()});
                } else if (this.loggingRateLimiter.tryAcquire()) {
                    log.warn("Error while sending {} of {} traces to the DD agent. Status: {} (going silent for {} seconds)", new Object[]{traces.size(), totalSize, responseCode, httpCon.getResponseMessage(), SECONDS_BETWEEN_ERROR_LOG});
                }
                return false;
            }
            log.debug("Succesfully sent {} of {} traces to the DD agent.", (Object)traces.size(), (Object)totalSize);
            try {
                if (null != responseString && !"".equals(responseString.trim()) && !"OK".equalsIgnoreCase(responseString.trim())) {
                    JsonNode response = objectMapper.readTree(responseString);
                    for (ResponseListener listener : this.responseListeners) {
                        listener.onResponse(this.tracesEndpoint, response);
                    }
                }
            }
            catch (IOException e) {
                log.debug("failed to parse DD agent response: " + responseString, (Throwable)e);
            }
            return true;
        }
        catch (IOException e) {
            if (log.isDebugEnabled()) {
                log.debug("Error while sending " + traces.size() + " of " + totalSize + " traces to the DD agent.", (Throwable)e);
            } else if (this.loggingRateLimiter.tryAcquire()) {
                log.warn("Error while sending {} of {} traces to the DD agent. {}: {} (going silent for {} seconds)", new Object[]{traces.size(), totalSize, e.getClass().getName(), e.getMessage(), SECONDS_BETWEEN_ERROR_LOG});
            }
            return false;
        }
    }

    private static boolean traceEndpointAvailable(String endpoint) {
        return DDApi.endpointAvailable(endpoint, Collections.emptyList(), true);
    }

    private static boolean serviceEndpointAvailable(String endpoint) {
        return DDApi.endpointAvailable(endpoint, Collections.emptyMap(), true);
    }

    private static boolean endpointAvailable(String endpoint, Object data, boolean retry) {
        try {
            HttpURLConnection httpCon = DDApi.getHttpURLConnection(endpoint);
            httpCon.setConnectTimeout((int)TimeUnit.SECONDS.toMillis(1L));
            httpCon.setReadTimeout((int)TimeUnit.SECONDS.toMillis(1L));
            OutputStream out = httpCon.getOutputStream();
            objectMapper.writeValue(out, data);
            out.flush();
            out.close();
            return httpCon.getResponseCode() == 200;
        }
        catch (IOException e) {
            if (retry) {
                return DDApi.endpointAvailable(endpoint, data, false);
            }
            return false;
        }
    }

    private static HttpURLConnection getHttpURLConnection(String endpoint) throws IOException {
        URL url = new URL(endpoint);
        HttpURLConnection httpCon = (HttpURLConnection)url.openConnection();
        httpCon.setDoOutput(true);
        httpCon.setDoInput(true);
        httpCon.setRequestMethod("PUT");
        httpCon.setRequestProperty("Content-Type", "application/msgpack");
        httpCon.setRequestProperty(DATADOG_META_LANG, "java");
        httpCon.setRequestProperty(DATADOG_META_LANG_VERSION, DDTraceOTInfo.JAVA_VERSION);
        httpCon.setRequestProperty(DATADOG_META_LANG_INTERPRETER, DDTraceOTInfo.JAVA_VM_NAME);
        httpCon.setRequestProperty(DATADOG_META_TRACER_VERSION, DDTraceOTInfo.VERSION);
        return httpCon;
    }

    public String toString() {
        return "DDApi { tracesEndpoint=" + this.tracesEndpoint + " }";
    }

    public static interface ResponseListener {
        public void onResponse(String var1, JsonNode var2);
    }
}

