/*
 * Decompiled with CFR 0.152.
 */
package ddtrot.dd.trace.util;

import datadog.trace.api.config.ProfilingConfig;
import ddtrot.dd.trace.api.telemetry.LogCollector;
import ddtrot.dd.trace.bootstrap.config.provider.ConfigProvider;
import ddtrot.dd.trace.util.AgentTaskScheduler;
import ddtrot.dd.trace.util.PidHelper;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TempLocationManager {
    private static final Logger log = LoggerFactory.getLogger(TempLocationManager.class);
    private static final Pattern JFR_DIR_PATTERN = Pattern.compile("\\d{4}_\\d{2}_\\d{2}_\\d{2}_\\d{2}_\\d{2}_\\d{6}");
    private static final String TEMPDIR_PREFIX = "pid_";
    private final boolean isPosixFs;
    private final Path baseTempDir;
    private final Path tempDir;
    private final long cutoffSeconds;
    private final CleanupTask cleanupTask = new CleanupTask();
    private final CleanupHook cleanupTestHook;

    public static TempLocationManager getInstance() {
        return TempLocationManager.getInstance(false);
    }

    static TempLocationManager getInstance(boolean waitForCleanup) {
        TempLocationManager instance = SingletonHolder.INSTANCE;
        if (waitForCleanup) {
            instance.waitForCleanup(5L, TimeUnit.SECONDS);
        }
        return instance;
    }

    private TempLocationManager() {
        this(ConfigProvider.getInstance());
    }

    TempLocationManager(ConfigProvider configProvider) {
        this(configProvider, true, CleanupHook.EMPTY);
    }

    TempLocationManager(ConfigProvider configProvider, boolean runStartupCleanup, CleanupHook testHook) {
        this.cleanupTestHook = testHook;
        Set<String> supportedViews = FileSystems.getDefault().supportedFileAttributeViews();
        this.isPosixFs = supportedViews.contains("posix");
        this.cutoffSeconds = configProvider.getLong("profiling.upload.period", 60L, new String[0]);
        Path configuredTempDir = Paths.get(configProvider.getString("profiling.tempdir", ProfilingConfig.PROFILING_TEMP_DIR_DEFAULT, new String[0]), new String[0]);
        if (!Files.exists(configuredTempDir, new LinkOption[0])) {
            log.warn(LogCollector.SEND_TELEMETRY, "Base temp directory, as defined in 'profiling.tempdir' does not exist: " + configuredTempDir);
            throw new IllegalStateException("Base temp directory, as defined in 'profiling.tempdir' does not exist: " + configuredTempDir);
        }
        String pid = PidHelper.getPid();
        this.baseTempDir = configuredTempDir.resolve(TempLocationManager.getBaseTempDirName());
        this.baseTempDir.toFile().deleteOnExit();
        this.tempDir = this.baseTempDir.resolve(TEMPDIR_PREFIX + pid);
        if (runStartupCleanup) {
            AgentTaskScheduler.INSTANCE.execute(this.cleanupTask);
        }
        this.createTempDir(this.tempDir);
    }

    static String getBaseTempDirName() {
        String userName = System.getProperty("user.name");
        userName = userName == null ? System.getenv("USER") : userName;
        userName = userName != null ? userName.replace('.', '_').replace('/', '_').replace(' ', '_') : null;
        return "ddprof" + (userName != null ? "_" + userName : "");
    }

    public Path getTempDir() {
        return this.getTempDir(null);
    }

    public Path getTempDir(Path subPath) {
        return this.getTempDir(subPath, true);
    }

    public Path getTempDir(Path subPath, boolean create) {
        Path rslt;
        Path path = rslt = subPath != null && !subPath.toString().isEmpty() ? this.tempDir.resolve(subPath) : this.tempDir;
        if (create && !Files.exists(rslt, new LinkOption[0])) {
            this.createTempDir(rslt);
        }
        return rslt;
    }

    boolean cleanup() {
        return this.cleanup(-1L, TimeUnit.SECONDS);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    boolean cleanup(long timeout, TimeUnit unit) {
        try {
            if (!Files.exists(this.baseTempDir, new LinkOption[0])) {
                return true;
            }
            try (Stream<Path> paths = Files.walk(this.baseTempDir, new FileVisitOption[0]);){
                if (paths.noneMatch(path -> Files.isDirectory(path, new LinkOption[0]) && path.getFileName().toString().startsWith(TEMPDIR_PREFIX))) {
                    boolean bl = true;
                    return bl;
                }
            }
            this.cleanupTestHook.onCleanupStart(timeout, unit);
            CleanupVisitor visitor = new CleanupVisitor(timeout, unit);
            Files.walkFileTree(this.baseTempDir, visitor);
            if (visitor.isTerminated()) return false;
            return true;
        }
        catch (IOException e) {
            if (log.isDebugEnabled()) {
                log.warn("Unable to cleanup temp location {}", (Object)this.baseTempDir, (Object)e);
                return false;
            }
            log.warn("Unable to cleanup temp location {}", (Object)this.baseTempDir);
            return false;
        }
    }

    boolean waitForCleanup(long timeout, TimeUnit unit) {
        try {
            return this.cleanupTask.await(timeout, unit);
        }
        catch (InterruptedException e) {
            log.debug("Temp directory cleanup was interrupted");
            Thread.currentThread().interrupt();
        }
        catch (Throwable t) {
            if (log.isDebugEnabled()) {
                log.debug("Failed to cleanup temp directory: {}", (Object)this.tempDir, (Object)t);
            }
            log.debug("Failed to cleanup temp directory: {}", (Object)this.tempDir);
        }
        return false;
    }

    void createDirStructure() throws IOException {
        Files.createDirectories(this.baseTempDir, new FileAttribute[0]);
    }

    private void createTempDir(Path tempDir) {
        String msg = "Failed to create temp directory: " + tempDir;
        try {
            if (this.isPosixFs) {
                Files.createDirectories(tempDir, PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rwx------")));
            } else {
                Files.createDirectories(tempDir, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            log.error("Failed to create temp directory {}", (Object)tempDir, (Object)e);
            if (this.isPosixFs) {
                Path root = this.baseTempDir.resolve(this.baseTempDir.relativize(tempDir).getName(0));
                try {
                    final AtomicReference failed = new AtomicReference();
                    Files.walkFileTree(root, (FileVisitor<? super Path>)new FileVisitor<Path>(){

                        @Override
                        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                            Set<PosixFilePermission> perms = Files.getPosixFilePermissions(dir, new LinkOption[0]);
                            if (!(perms.contains((Object)PosixFilePermission.OWNER_READ) && perms.contains((Object)PosixFilePermission.OWNER_WRITE) && perms.contains((Object)PosixFilePermission.OWNER_EXECUTE))) {
                                failed.set(dir);
                                return FileVisitResult.TERMINATE;
                            }
                            return FileVisitResult.CONTINUE;
                        }

                        @Override
                        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                            return FileVisitResult.SKIP_SIBLINGS;
                        }

                        @Override
                        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                            return FileVisitResult.TERMINATE;
                        }

                        @Override
                        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                            return FileVisitResult.CONTINUE;
                        }
                    });
                    Path failedDir = (Path)failed.get();
                    if (failedDir != null) {
                        msg = msg + " (offender: " + failedDir + ", permissions: " + PosixFilePermissions.toString(Files.getPosixFilePermissions(failedDir, new LinkOption[0])) + ")";
                        log.warn(LogCollector.SEND_TELEMETRY, msg, (Throwable)e);
                    }
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                throw new IllegalStateException(msg, e);
            }
            log.warn(LogCollector.SEND_TELEMETRY, msg, (Throwable)e);
            throw new IllegalStateException(msg, e);
        }
    }

    private final class CleanupTask
    implements Runnable {
        private final CountDownLatch latch = new CountDownLatch(1);
        private volatile Throwable throwable = null;

        private CleanupTask() {
        }

        @Override
        public void run() {
            try {
                TempLocationManager.this.cleanup();
            }
            catch (OutOfMemoryError oom) {
                throw oom;
            }
            catch (Throwable t) {
                this.throwable = t;
            }
            finally {
                this.latch.countDown();
            }
        }

        boolean await(long timeout, TimeUnit unit) throws Throwable {
            boolean ret = this.latch.await(timeout, unit);
            if (this.throwable != null) {
                throw this.throwable;
            }
            return ret;
        }
    }

    private final class CleanupVisitor
    implements FileVisitor<Path> {
        private boolean shouldClean;
        private Set<String> pidSet;
        private final Instant cutoff;
        private final Instant timeoutTarget;
        private boolean terminated = false;

        CleanupVisitor(long timeout, TimeUnit unit) {
            this.cutoff = Instant.now().minus(TempLocationManager.this.cutoffSeconds, ChronoUnit.SECONDS);
            this.timeoutTarget = timeout > -1L ? Instant.now().plus(TimeUnit.MILLISECONDS.convert(timeout, unit), ChronoUnit.MILLIS) : null;
        }

        boolean isTerminated() {
            return this.terminated;
        }

        private boolean isTimedOut() {
            return this.timeoutTarget != null && Instant.now().isAfter(this.timeoutTarget);
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            boolean isSelfPid;
            if (this.isTimedOut()) {
                log.debug("Cleaning task timed out");
                this.terminated = true;
                return FileVisitResult.TERMINATE;
            }
            TempLocationManager.this.cleanupTestHook.preVisitDirectory(dir, attrs);
            if (dir.equals(TempLocationManager.this.baseTempDir)) {
                return FileVisitResult.CONTINUE;
            }
            String fileName = dir.getFileName().toString();
            String pid = fileName.startsWith(TempLocationManager.TEMPDIR_PREFIX) ? fileName.substring(4) : null;
            boolean bl = isSelfPid = pid != null && pid.equals(PidHelper.getPid());
            if (!isSelfPid) {
                if (this.pidSet == null) {
                    this.pidSet = PidHelper.getJavaPids();
                }
                this.shouldClean |= !this.pidSet.contains(pid);
            }
            if (this.shouldClean) {
                log.debug("Cleaning temporary location {}", (Object)dir);
            }
            return this.shouldClean ? FileVisitResult.CONTINUE : FileVisitResult.SKIP_SUBTREE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (this.isTimedOut()) {
                log.debug("Cleaning task timed out");
                this.terminated = true;
                return FileVisitResult.TERMINATE;
            }
            TempLocationManager.this.cleanupTestHook.visitFile(file, attrs);
            try {
                if (Files.getLastModifiedTime(file, new LinkOption[0]).toInstant().isAfter(this.cutoff)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                Files.delete(file);
            }
            catch (NoSuchFileException noSuchFileException) {
                // empty catch block
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            if (this.isTimedOut()) {
                log.debug("Cleaning task timed out");
                this.terminated = true;
                return FileVisitResult.TERMINATE;
            }
            TempLocationManager.this.cleanupTestHook.visitFileFailed(file, exc);
            if (!(exc instanceof NoSuchFileException) && log.isDebugEnabled()) {
                log.debug("Failed to delete file {}", (Object)file, (Object)exc);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            if (this.isTimedOut()) {
                log.debug("Cleaning task timed out");
                this.terminated = true;
                return FileVisitResult.TERMINATE;
            }
            TempLocationManager.this.cleanupTestHook.postVisitDirectory(dir, exc);
            if (exc instanceof NoSuchFileException) {
                return FileVisitResult.CONTINUE;
            }
            if (this.shouldClean) {
                try {
                    Files.delete(dir);
                }
                catch (NoSuchFileException noSuchFileException) {
                    // empty catch block
                }
                String fileName = dir.getFileName().toString();
                this.shouldClean = !fileName.startsWith(TempLocationManager.TEMPDIR_PREFIX);
            }
            return FileVisitResult.CONTINUE;
        }
    }

    static interface CleanupHook
    extends FileVisitor<Path> {
        public static final CleanupHook EMPTY = new CleanupHook(){};

        @Override
        default public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            return null;
        }

        @Override
        default public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            return null;
        }

        @Override
        default public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            return null;
        }

        @Override
        default public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            return null;
        }

        default public void onCleanupStart(long timeout, TimeUnit unit) {
        }
    }

    private static final class SingletonHolder {
        private static final TempLocationManager INSTANCE = new TempLocationManager();

        private SingletonHolder() {
        }
    }
}

