/*
 * Decompiled with CFR 0.152.
 */
package com.dataliquid.asciidoc.linter.validator;

import com.dataliquid.asciidoc.linter.config.DocumentConfiguration;
import com.dataliquid.asciidoc.linter.config.Severity;
import com.dataliquid.asciidoc.linter.config.rule.SectionConfig;
import com.dataliquid.asciidoc.linter.config.rule.TitleConfig;
import com.dataliquid.asciidoc.linter.validator.SourceLocation;
import com.dataliquid.asciidoc.linter.validator.ValidationMessage;
import com.dataliquid.asciidoc.linter.validator.ValidationResult;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.asciidoctor.ast.Document;
import org.asciidoctor.ast.Section;
import org.asciidoctor.ast.StructuralNode;

public final class SectionValidator {
    private final DocumentConfiguration configuration;
    private final Map<String, Integer> sectionOccurrences;
    private final List<SectionConfig> rootSections;

    private SectionValidator(Builder builder) {
        this.configuration = Objects.requireNonNull(builder.configuration, "[" + this.getClass().getName() + "] configuration must not be null");
        this.sectionOccurrences = new HashMap<String, Integer>();
        this.rootSections = this.configuration.sections() != null ? this.configuration.sections() : Collections.emptyList();
    }

    public ValidationResult validate(Document document) {
        long startTime = System.currentTimeMillis();
        ValidationResult.Builder resultBuilder = ValidationResult.builder().startTime(startTime);
        String filename = this.extractFilename(document);
        this.validateDocumentTitle(document, filename, resultBuilder);
        List<StructuralNode> sections = document.getBlocks().stream().filter(block -> block instanceof Section).collect(Collectors.toList());
        this.validateRootSections(sections, filename, resultBuilder);
        this.validateMinMaxOccurrences(filename, resultBuilder);
        this.validateSectionOrder(sections, filename, resultBuilder);
        return resultBuilder.complete().build();
    }

    private void validateRootSections(List<StructuralNode> sections, String filename, ValidationResult.Builder resultBuilder) {
        List<SectionConfig> level1Configs = this.determineLevel1Configs();
        for (StructuralNode node : sections) {
            if (!(node instanceof Section)) continue;
            Section section = (Section)node;
            this.validateSection(section, level1Configs, filename, resultBuilder, null);
        }
    }

    private List<SectionConfig> determineLevel1Configs() {
        for (SectionConfig config2 : this.rootSections) {
            if (config2.level() != 0 || config2.subsections() == null || config2.subsections().isEmpty()) continue;
            return config2.subsections();
        }
        return this.rootSections.stream().filter(config -> config.level() != 0).collect(Collectors.toList());
    }

    private void validateSection(Section section, List<SectionConfig> allowedConfigs, String filename, ValidationResult.Builder resultBuilder, SectionConfig parentConfig) {
        int level = section.getLevel();
        String title = section.getTitle();
        SectionConfig matchingConfig = this.findMatchingConfig(section, allowedConfigs);
        if (matchingConfig == null && !allowedConfigs.isEmpty()) {
            List<SectionConfig> levelConfigs = this.findConfigsForLevel(level, allowedConfigs);
            SourceLocation location = this.createLocation(filename, section);
            if (!levelConfigs.isEmpty()) {
                SectionConfig configWithPattern = levelConfigs.stream().filter(config -> config.title() != null && (config.title().pattern() != null || config.title().exactMatch() != null)).findFirst().orElse(levelConfigs.get(0));
                String expectedPattern = null;
                if (configWithPattern.title() != null) {
                    if (configWithPattern.title().pattern() != null) {
                        expectedPattern = "Pattern: " + configWithPattern.title().pattern();
                    } else if (configWithPattern.title().exactMatch() != null) {
                        expectedPattern = "Exact match: " + configWithPattern.title().exactMatch();
                    }
                }
                ValidationMessage message = ValidationMessage.builder().severity(Severity.ERROR).ruleId("section.title.pattern").location(location).message("Section title doesn't match required pattern at level " + level + ": '" + title + "'").actualValue(title).expectedValue((String)(expectedPattern != null ? expectedPattern : "One of configured patterns")).build();
                resultBuilder.addMessage(message);
            } else {
                ValidationMessage message = ValidationMessage.builder().severity(Severity.ERROR).ruleId("section.unexpected").location(location).message("Section not allowed at level " + level + ": '" + title + "'").actualValue(title).expectedValue("No sections configured for level " + level).build();
                resultBuilder.addMessage(message);
            }
            return;
        }
        if (matchingConfig != null) {
            this.trackOccurrence(matchingConfig);
            this.validateTitle(section, matchingConfig.title(), filename, resultBuilder);
            this.validateLevel(section, matchingConfig, filename, resultBuilder);
            List subsections = section.getBlocks().stream().filter(block -> block instanceof Section).collect(Collectors.toList());
            for (StructuralNode subsection : subsections) {
                this.validateSection((Section)subsection, matchingConfig.subsections(), filename, resultBuilder, matchingConfig);
            }
        }
    }

    private void validateTitle(Section section, TitleConfig titleConfig, String filename, ValidationResult.Builder resultBuilder) {
        Pattern pattern;
        if (titleConfig == null) {
            return;
        }
        String title = section.getTitle();
        SourceLocation location = this.createLocation(filename, section);
        if (titleConfig.pattern() != null && !(pattern = Pattern.compile(titleConfig.pattern())).matcher(title).matches()) {
            ValidationMessage message = ValidationMessage.builder().severity(titleConfig.severity()).ruleId("section.title.pattern").location(location).message("Section title does not match required pattern").actualValue(title).expectedValue("Pattern: " + titleConfig.pattern()).build();
            resultBuilder.addMessage(message);
        }
        if (titleConfig.exactMatch() != null && !title.equals(titleConfig.exactMatch())) {
            ValidationMessage message = ValidationMessage.builder().severity(titleConfig.severity()).ruleId("section.title.exact").location(location).message("Section title does not match expected value").actualValue(title).expectedValue(titleConfig.exactMatch()).build();
            resultBuilder.addMessage(message);
        }
    }

    private void validateLevel(Section section, SectionConfig config, String filename, ValidationResult.Builder resultBuilder) {
        int expectedLevel;
        int actualLevel = section.getLevel();
        if (actualLevel != (expectedLevel = config.level())) {
            SourceLocation location = this.createLocation(filename, section);
            ValidationMessage message = ValidationMessage.builder().severity(Severity.ERROR).ruleId("section.level").location(location).message("Section level mismatch").actualValue(String.valueOf(actualLevel)).expectedValue(String.valueOf(expectedLevel)).build();
            resultBuilder.addMessage(message);
        }
    }

    private void validateMinMaxOccurrences(String filename, ValidationResult.Builder resultBuilder) {
        for (SectionConfig config : this.rootSections) {
            this.validateOccurrenceForConfig(config, filename, resultBuilder);
        }
        for (SectionConfig config : this.rootSections) {
            if (config.level() != 0 || config.subsections() == null) continue;
            for (SectionConfig subsection : config.subsections()) {
                this.validateOccurrenceForConfig(subsection, filename, resultBuilder);
            }
        }
    }

    private void validateOccurrenceForConfig(SectionConfig config, String filename, ValidationResult.Builder resultBuilder) {
        ValidationMessage message;
        SourceLocation location;
        String key = this.createOccurrenceKey(config);
        int occurrences = this.sectionOccurrences.getOrDefault(key, 0);
        if (occurrences < config.min()) {
            location = SourceLocation.builder().filename(filename).line(1).build();
            message = ValidationMessage.builder().severity(Severity.ERROR).ruleId("section.min-occurrences").location(location).message("Too few occurrences of section: " + config.name()).actualValue(String.valueOf(occurrences)).expectedValue("At least " + config.min()).build();
            resultBuilder.addMessage(message);
        }
        if (occurrences > config.max()) {
            location = SourceLocation.builder().filename(filename).line(1).build();
            message = ValidationMessage.builder().severity(Severity.ERROR).ruleId("section.max-occurrences").location(location).message("Too many occurrences of section: " + config.name()).actualValue(String.valueOf(occurrences)).expectedValue("At most " + config.max()).build();
            resultBuilder.addMessage(message);
        }
        if (config.subsections() != null) {
            for (SectionConfig subsection : config.subsections()) {
                this.validateOccurrenceForConfig(subsection, filename, resultBuilder);
            }
        }
    }

    private void validateDocumentTitleConfig(String title, TitleConfig titleConfig, SourceLocation location, ValidationResult.Builder resultBuilder) {
        Pattern pattern;
        if (titleConfig.pattern() != null && !(pattern = Pattern.compile(titleConfig.pattern())).matcher(title).matches()) {
            ValidationMessage message = ValidationMessage.builder().severity(titleConfig.severity()).ruleId("section.title.pattern").location(location).message("Document title does not match required pattern").actualValue(title).expectedValue("Pattern: " + titleConfig.pattern()).build();
            resultBuilder.addMessage(message);
        }
        if (titleConfig.exactMatch() != null && !title.equals(titleConfig.exactMatch())) {
            ValidationMessage message = ValidationMessage.builder().severity(titleConfig.severity()).ruleId("section.title.exactMatch").location(location).message("Document title does not match expected value").actualValue(title).expectedValue(titleConfig.exactMatch()).build();
            resultBuilder.addMessage(message);
        }
    }

    private void validateDocumentTitle(Document document, String filename, ValidationResult.Builder resultBuilder) {
        SectionConfig titleConfig = this.rootSections.stream().filter(config -> config.level() == 0).findFirst().orElse(null);
        if (titleConfig == null) {
            return;
        }
        String documentTitle = document.getTitle();
        SourceLocation location = SourceLocation.builder().filename(filename).line(1).build();
        if (titleConfig.min() > 0 && (documentTitle == null || documentTitle.trim().isEmpty())) {
            ValidationMessage message = ValidationMessage.builder().severity(Severity.ERROR).ruleId("section.level0.missing").location(location).message("Document title is required").actualValue("No title").expectedValue("Document title").build();
            resultBuilder.addMessage(message);
            return;
        }
        if (documentTitle != null && titleConfig.title() != null) {
            this.validateDocumentTitleConfig(documentTitle, titleConfig.title(), location, resultBuilder);
        }
        String configKey = this.createOccurrenceKey(titleConfig);
        this.sectionOccurrences.put(configKey, documentTitle != null ? 1 : 0);
    }

    private void validateSectionOrder(List<StructuralNode> sections, String filename, ValidationResult.Builder resultBuilder) {
        int i;
        List orderedConfigs = this.rootSections.stream().filter(config -> config.order() != null).sorted(Comparator.comparing(SectionConfig::order)).collect(Collectors.toList());
        if (orderedConfigs.isEmpty()) {
            return;
        }
        HashMap<String, Integer> actualOrder = new HashMap<String, Integer>();
        for (i = 0; i < sections.size(); ++i) {
            Section section;
            SectionConfig config2;
            if (!(sections.get(i) instanceof Section) || (config2 = this.findMatchingConfig(section = (Section)sections.get(i), this.rootSections)) == null || config2.order() == null) continue;
            actualOrder.put(config2.name(), i);
        }
        for (i = 0; i < orderedConfigs.size() - 1; ++i) {
            SectionConfig current = (SectionConfig)orderedConfigs.get(i);
            SectionConfig next = (SectionConfig)orderedConfigs.get(i + 1);
            Integer currentPos = (Integer)actualOrder.get(current.name());
            Integer nextPos = (Integer)actualOrder.get(next.name());
            if (currentPos == null || nextPos == null || currentPos <= nextPos) continue;
            SourceLocation location = SourceLocation.builder().filename(filename).line(1).build();
            ValidationMessage message = ValidationMessage.builder().severity(Severity.ERROR).ruleId("section.order").location(location).message("Section order violation").actualValue(current.name() + " appears after " + next.name()).expectedValue(current.name() + " should appear before " + next.name()).build();
            resultBuilder.addMessage(message);
        }
    }

    private SectionConfig findMatchingConfig(Section section, List<SectionConfig> configs) {
        String title = section.getTitle();
        int level = section.getLevel();
        for (SectionConfig config : configs) {
            if (config.level() != level) continue;
            if (config.title() != null) {
                Pattern pattern;
                if (config.title().exactMatch() != null && config.title().exactMatch().equals(title)) {
                    return config;
                }
                if (config.title().pattern() == null || !(pattern = Pattern.compile(config.title().pattern())).matcher(title).matches()) continue;
                return config;
            }
            if (config.name() == null) continue;
            return config;
        }
        return null;
    }

    private List<SectionConfig> findConfigsForLevel(int level, List<SectionConfig> configs) {
        return configs.stream().filter(config -> config.level() == level).collect(Collectors.toList());
    }

    private void trackOccurrence(SectionConfig config) {
        String key = this.createOccurrenceKey(config);
        this.sectionOccurrences.merge(key, 1, Integer::sum);
    }

    private String createOccurrenceKey(SectionConfig config) {
        return config.name() + "_" + config.level();
    }

    private String extractFilename(Document document) {
        Map attrs = document.getAttributes();
        if (attrs.containsKey("docfile")) {
            return attrs.get("docfile").toString();
        }
        return "unknown";
    }

    private SourceLocation createLocation(String filename, Section section) {
        return SourceLocation.builder().filename(filename).line(section.getSourceLocation() != null ? section.getSourceLocation().getLineNumber() : 1).build();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder fromConfiguration(DocumentConfiguration configuration) {
        return SectionValidator.builder().configuration(configuration);
    }

    public static final class Builder {
        private DocumentConfiguration configuration;

        private Builder() {
        }

        public Builder configuration(DocumentConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public SectionValidator build() {
            return new SectionValidator(this);
        }
    }
}

