package com.datarobot;

import java.io.FileNotFoundException;

import com.datarobot.impl.ClientException;
import com.datarobot.model.LearningSession;
import com.datarobot.model.LearningSessionList;
import com.datarobot.model.Output;
import com.datarobot.model.OutputFeatures;
import com.datarobot.model.OutputList;
import com.datarobot.model.PagingParams;
import com.datarobot.model.PredictionList;
import com.datarobot.model.AI;
import com.datarobot.model.AIList;
import com.datarobot.model.Dataset;
import com.datarobot.util.Action;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;

/**
 * The {@link IAIClient IAIClient} interface provides access to {@link AI}
 * related endpoints for the AI API
 */
public interface IAIClient {
    Action<HttpRequest, HttpResponse> getHttpMessageTransformer();

    void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer);

    /**
     * Retrieve an AI
     * 
     * @param id The ID of the AI to retrieve
     * 
     * @return The queried {@link AI}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    AI get(String id) throws ClientException;

    /**
     * Retrieve a list of all AIs associated with this account
     * 
     * @return {@link AIList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    AIList list() throws ClientException;

    /**
     * Retrieve a list of all AIs associated with this account
     * 
     * @param params The {@link PagingParams} object for this list
     * 
     * @return {@link AIList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    AIList list(PagingParams params) throws ClientException;

    /**
     * Create an AI with the given name
     * 
     * @param name The name of the AI
     * 
     * @return The new {@link AI}
     * 
     * @throws ClientException      when 4xx or 5xx response is received from
     *                              server, or errors in parsing the response.
     * @throws InterruptedException when a thread is waiting, sleeping, or otherwise
     *                              occupied, and the thread is interrupted, either
     *                              before or during the activity.
     */
    AI createAI(String name) throws ClientException, InterruptedException;

    // Support for allowing AIs w/out names to be created - can't remember if we
    // allow this or not, pending
    // AI createAI() throws ClientException, InterruptedException;

    /**
     * Delete an AI
     * 
     * @param aiId The ID of the AI to delete
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    void delete(String aiId) throws ClientException;

    /**
     * Add a {@link LearningSession} to the AI
     * 
     * @param aiId              The ID of the AI to which you want to add the
     *                          learning session
     * @param learningSessionId The ID of the learning session to add
     * @param outputName        Name of the output this AI will learn
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    void addLearningSession(String aiId, String learningSessionId, String outputName) throws ClientException;

    /**
     * Add a {@link Dataset} to the AI
     * 
     * @param aiId      The ID of the AI to which you want to add the dataset
     * @param datasetId The ID of the dataset to add
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    void addDataset(String aiId, String datasetId) throws ClientException;

    /**
     * Retrieve a list of all learning sessions associated with an AI
     * 
     * @param aiId The ID of the AI to retrieve learning sessions for
     * 
     * @return {@link LearningSessionList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    LearningSessionList getLearningSessions(String aiId) throws ClientException;

    /**
     * Retrieve a list of all learning sessions associated with an AI
     * 
     * @param aiId   The ID of the AI to retrieve learning sessions for
     * @param params The {@link PagingParams} object for this list
     * 
     * @return {@link LearningSessionList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    LearningSessionList getLearningSessions(String aiId, PagingParams params) throws ClientException;

    // TODO - add try/catch for output that does not exist
    /**
     * Retrieve the {@link Output} associated with an AI and target
     * 
     * @param aiId         The ID of the AI to retrieve an output for
     * @param outputTarget The target of this output
     * 
     * @return {@link Output}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    Output getOutput(String aiId, String outputTarget) throws ClientException;

    /**
     * Make predictions on an AI, target and dataset. Note an AI must be trained
     * with {@link AI#learn} or an existing learning session must be added to the AI
     * with {@link AI#addLearningSession} or {@link IAIClient#addLearningSession}
     * before predictions can occur.
     * 
     * @param aiId     The ID of the AI to retrieve predictions for
     * @param target   The name of the selected target feature to predict on
     * @param filePath The data on which to predict via a filepath on the local
     *                 system
     * 
     * @return {@link PredictionList}
     * 
     * @throws ClientException       when 4xx or 5xx response is received from
     *                               server, or errors in parsing the response.
     * @throws FileNotFoundException when a file with the specified pathname does
     *                               not exist, or if the file does exist but is
     *                               inaccessible for some reason.
     */
    PredictionList predict(String aiId, String target, String filePath) throws ClientException, FileNotFoundException;

    /**
     * Make predictions on an AI, target and dataset. Note an AI must be trained
     * with {@link AI#learn} or an existing learning session must be added to the AI
     * with {@link AI#addLearningSession} or {@link IAIClient#addLearningSession}
     * before predictions can occur.
     * 
     * @param aiId     The ID of the AI to retrieve predictions for
     * @param target   The name of the selected target feature to predict on
     * @param filePath The data on which to predict via a filepath on the local
     *                 system
     * 
     * @return {@link PredictionList}
     * 
     * @throws ClientException       when 4xx or 5xx response is received from
     *                               server, or errors in parsing the response.
     * @throws FileNotFoundException when a file with the specified pathname does
     *                               not exist, or if the file does exist but is
     *                               inaccessible for some reason.
     */
    PredictionList infer(String aiId, String target, String filePath) throws ClientException, FileNotFoundException;

    /**
     * Retrieve a list of all outputs associated with an AI.
     * 
     * @param aiId The ID of the AI to retrieve outputs for
     * 
     * @return {@link OutputList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    OutputList getOutputs(String aiId) throws ClientException;

    /**
     * Retrieve a list of all outputs associated with an AI.
     * 
     * @param aiId   The ID of the AI to retrieve outputs for
     * @param params The {@link PagingParams} object for this list
     * 
     * @return {@link OutputList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    OutputList getOutputs(String aiId, PagingParams params) throws ClientException;

    /**
     * Add an {@link Output} to the AI
     * 
     * @param aiId              The ID of the AI to which you want to add the
     *                          dataset
     * @param learningSessionId The ID of the learning session
     * @param outputName        The name of the output
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    void addOutput(String aiId, String learningSessionId, String outputName) throws ClientException;

    /**
     * Retrieve features of an output associated with an AI and a target
     * 
     * @param aiId         The ID of the AI to retrieve output features for
     * @param outputTarget The target of this output
     * 
     * @return {@link OutputFeatures}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    OutputFeatures getOutputFeatures(String aiId, String outputTarget) throws ClientException;

    /*
     * TODO: - Prediction predict(String aiId, String target, ArrayList) <-- ? idea
     * for prediction via an upload *
     */

}
