package com.datarobot;

import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;
import com.datarobot.impl.ClientException;
import com.datarobot.model.*;
import com.datarobot.util.Action;

import java.io.FileNotFoundException;
import java.util.concurrent.Future;

/**
 * The {@link IDatasetClient IDatasetClient} interface provides access to
 * {@link Dataset} related endpoints in the AI API
 */
public interface IDatasetClient {
    Action<HttpRequest, HttpResponse> getHttpMessageTransformer();

    void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer);

    /**
     * Upload a new dataset. See the Dataset page in the documentation for size
     * limitations.
     * 
     * @param sourceFile The path of the local file to upload
     * 
     * @return The new {@link Dataset}
     * 
     * @throws ClientException       when 4xx or 5xx response is received from
     *                               server, or errors in parsing the response.
     * @throws FileNotFoundException when a file with the specified pathname does
     *                               not exist, or if the file does exist but is
     *                               inaccessible for some reason.
     * @throws InterruptedException  when a thread is waiting, sleeping, or
     *                               otherwise occupied, and the thread is
     *                               interrupted, either before or during the
     *                               activity.
     */
    Dataset importFile(String sourceFile) throws ClientException, FileNotFoundException, InterruptedException;

    /**
     * Import a dataset from a url. See the Dataset page in the documentation for
     * size limitations.
     * 
     * @param url The url to a publicly available file
     * 
     * @return The new {@link Dataset}
     * 
     * @throws ClientException      when 4xx or 5xx response is received from
     *                              server, or errors in parsing the response.
     * @throws InterruptedException when a thread is waiting, sleeping, or otherwise
     *                              occupied, and the thread is interrupted, either
     *                              before or during the activity.
     */
    Dataset importUrl(String url) throws ClientException, InterruptedException;

    /**
     * Start the process of uploading a dataset from a local file. This method can
     * be used to avoid blocking during large file imports. See the Dataset page in
     * the documentation for size limitations.
     * 
     * @param sourceFile The path of the local file to upload
     * 
     * @return {@link Future}{@link Dataset}
     * 
     * @throws FileNotFoundException when a file with the specified pathname does
     *                               not exist, or if the file does exist but is
     *                               inaccessible for some reason.
     * @throws ClientException       when 4xx or 5xx response is received from
     *                               server, or errors in parsing the response.
     */
    Future<Dataset> startImportFile(final String sourceFile) throws FileNotFoundException, ClientException;

    /**
     * Start the process of uploading a dataset from a url. This method can be used
     * to avoid blocking during large file imports. See the Dataset page in the
     * documentation for size limitations.
     * 
     * @param url The url to a publicly available file
     * 
     * @return {@link Future}{@link Dataset}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    Future<Dataset> startimportUrl(final String url) throws ClientException;

    /**
     * Retrieve a dataset
     * 
     * @param datasetId The ID of the dataset to retrieve
     * 
     * @return The queried {@link Dataset}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    Dataset get(String datasetId) throws ClientException;

    /**
     * Retrieve a list of uploaded datasets associated with this account
     * 
     * @return {@link DatasetList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    DatasetList list() throws ClientException;

    /**
     * Retrieve a list of uploaded datasets associated with this AI
     * 
     * @param aiId Datasets for this AI will only be returned
     * 
     * @return {@link DatasetList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    DatasetList list(String aiId) throws ClientException;

    /**
     * Retrieve a list of uploaded datasets associated with this account
     * 
     * @param params The {@link PagingParams} object for this list
     * 
     * @return {@link DatasetList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    DatasetList list(PagingParams params) throws ClientException;

    /**
     * Delete a dataset
     * 
     * @param datasetId The ID of the dataset to delete
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    void delete(String datasetId) throws ClientException;

    /*
     * TODO: 1. Not in the Python client, but would be helpful for Java users:
     * Dataset importStream(String fileName, InputStream stream) throws
     * ClientException, InterruptedException; Future<Dataset>
     * startImportStream(String fileName, InputStream stream);
     * 
     * 2. Still needs implemented (not all of these are even possible, just keeping
     * this as a reminder for now): Dataset importDataFrame() Dataset
     * startImportDataFrame() Dataset importListOfRecords() Dataset
     * startImportListOfRecords()
     * 
     */
}
