package com.datarobot;

import java.util.concurrent.Future;

import com.datarobot.impl.ClientException;
import com.datarobot.impl.StatusTask;
import com.datarobot.model.*;
import com.datarobot.model.Evaluation;
import com.datarobot.util.Action;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;

/**
 * The {@link ILearningSessionClient ILearningSessionClient} interface provides
 * access to {@link LearningSession} related endpoints in the AI API
 */
public interface ILearningSessionClient {
    Action<HttpRequest, HttpResponse> getHttpMessageTransformer();

    void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer);

    /*
     * TODO: 1. Refactor paging params object so a user can just List(10, 30)
     * (offset, max, etc)
     * 
     */

    /**
     * Retrieve a {@link LearningSession} by id
     * 
     * @param id The ID of the learning session to retrieve
     * 
     * @return The queried {@link LearningSession}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    LearningSession get(String id) throws ClientException;

    /**
     * Retrieve a list of learning sessions associated with this account
     * 
     * @return {@link LearningSessionList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    LearningSessionList list() throws ClientException;

    /**
     * Retrieve a list of learning sessions associated with this account
     * 
     * @param params The {@link PagingParams} object for this list
     * 
     * @return {@link LearningSessionList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    LearningSessionList list(PagingParams params) throws ClientException;

    /**
     * Delete a learning session
     * 
     * @param id The ID of the learning session to delete
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    void delete(String id) throws ClientException;

    /**
     * Create a learning session and wait for it to be able to predict
     * 
     * @param datasetId The ID of the dataset to learn on
     * @param target    The name of the feature from the dataset to learn how to
     *                  predict
     * 
     * @return The new {@link LearningSession}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    StatusTask<LearningSession> learn(String datasetId, String target) throws ClientException;

    /**
     * Set up a learning session without waiting for it to be ready. To block until
     * this process finishes, use {@link ILearningSessionClient#learn} instead or
     * call {@code .get()} on the return {@code Future<LearningSession>}.
     * 
     * @param datasetId The ID of the dataset to learn on
     * @param target    The name of the feature from the dataset to learn how to
     *                  predict
     * 
     * @return The new {@link Future}{@link LearningSession}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    Future<LearningSession> startLearn(String datasetId, String target) throws ClientException;

    /**
     * Get the deployment associated with this learning session. The resulting
     * deployment can be used to predict via
     * {@link IPredictionClient#deploymentPredict} although
     * {@link LearningSession#predict} can also be called directly on the
     * {@link LearningSession} object.
     * 
     * @param learningSessionId The ID of the learning session
     * 
     * @return {@link Deployment}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    Deployment getDeployment(String learningSessionId) throws ClientException;

    /**
     * Retrieve features association with a {@link LearningSession}. Values for
     * these features must be provided when making a prediction.
     * 
     * @param learningSessionId The ID of the learning session
     * 
     * @return {@link LearningSessionFeatures}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    LearningSessionFeatures getFeatures(String learningSessionId) throws ClientException;

    /**
     * Get performance information for the given learning session. For details on
     * how to interpret this information see the learning session documentation.
     * 
     * @param learningSessionId The ID of the learning session
     * 
     * @return {@link Evaluation}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    Evaluation getEvaluation(String learningSessionId) throws ClientException;

}
