package com.datarobot.impl;

import java.io.FileNotFoundException;
import java.util.HashMap;
import java.util.Map;

import com.datarobot.IDataRobotAIClient;
import com.datarobot.IAIClient;
import com.datarobot.model.Dataset;
import com.datarobot.model.IDoGetCallback;
import com.datarobot.model.LearningSession;
import com.datarobot.model.LearningSessionList;
import com.datarobot.model.Output;
import com.datarobot.model.OutputFeatures;
import com.datarobot.model.OutputList;
import com.datarobot.model.PagingParams;
import com.datarobot.model.PredictionList;
import com.datarobot.model.AI;
import com.datarobot.model.AICreationResponse;
import com.datarobot.model.AIEmptyResponse;
import com.datarobot.model.AIList;
import com.datarobot.model.AIBody;
import com.datarobot.util.Action;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;

/**
 * The {@link AIClient} object provides access to the AI endpoints of the
 * DataRobot AI API. This object is not meant to be used directly but through
 * the {@code DataRobotAIClient.ais()} method.
 */
public class AIClient implements IAIClient, IDoGetCallback<AI> {
     private IDataRobotAIClient client;
     private Action<HttpRequest, HttpResponse> httpMessageTransformer = null;

     /**
      * {@link AI IAIClient} based API operations. Users will not need to instantiate
      * this object directly. It can be accessed through
      * {@code DataRobotAIClient.ais()}.
      * 
      * @param client {@link DataRobotAIClient}
      */
     public AIClient(IDataRobotAIClient client) {
          this.client = client;
     }

     /**
      * internal
      */
     @Override
     public Action<HttpRequest, HttpResponse> getHttpMessageTransformer() {
          return httpMessageTransformer;
     }

     /**
      * internal
      */
     @Override
     public void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer) {
          this.httpMessageTransformer = httpMessageTransformer;
     }

     /**
      * Retrieve an AI
      * 
      * @param id The ID of the AI to retrieve
      * 
      * @return The queried {@link AI}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public AI get(String id) throws ClientException {
          Argument.IsNotNullOrEmpty(id, "id");

          return client.getConnection().get(AI.class, "ais/" + id + "/", null, this.httpMessageTransformer);
     }

     /**
      * Retrieve a list of all AIs associated with this account
      * 
      * @return {@link AIList}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public AIList list() throws ClientException {
          return this.list(null);
     }

     /**
      * Retrieve a list of all AIs associated with this account
      * 
      * @param params The {@link PagingParams} object for this list
      * 
      * @return {@link AIList}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public AIList list(PagingParams params) throws ClientException {
          Map<String, Object> parameters = null;

          if (params != null) {
               parameters = params.toParameters();
          }
          AIList list = client.getConnection().get(AIList.class, "ais/", parameters, httpMessageTransformer);

          for (AI ai : list.getItems()) {
               ai.setClient(this.client);
          }
          return list;
     }

     /**
      * Create an AI with the given name
      * 
      * @param name The name of the AI
      * 
      * @return The new {@link AI}
      * 
      * @throws ClientException      when 4xx or 5xx response is received from
      *                              server, or errors in parsing the response.
      * @throws InterruptedException when a thread is waiting, sleeping, or otherwise
      *                              occupied, and the thread is interrupted, either
      *                              before or during the activity.
      */
     @Override
     public AI createAI(String name) throws ClientException, InterruptedException {
          AICreationResponse aiResponse = client.getConnection().post(AICreationResponse.class, "ais/", null,
                    new AIBody(name, 1), this.httpMessageTransformer);
          String aiId = aiResponse.getAIId();
          return get(aiId);
     }

     // Support for allowing AIs w/out names to be created - can't remember if we
     // allow this or not, pending
     // @Override
     // public AI createAI() throws ClientException, InterruptedException{
     // AICreationResponse aiResponse =
     // client.getConnection().post(AICreationResponse.class, "ais/", null, null,
     // this.httpMessageTransformer);
     // String aiId = aiResponse.getAIId();
     // return get(aiId);
     // }

     /**
      * Delete an AI
      * 
      * @param aiId The ID of the AI to delete
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public void delete(String aiId) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          client.getConnection().delete("/ais/" + aiId + "/", null, this.httpMessageTransformer);
     }

     /**
      * Add a {@link LearningSession} to the AI
      * 
      * @param aiId              The ID of the AI to which you want to add the
      *                          learning session
      * @param learningSessionId The ID of the learning session to add
      * @param outputName        Name of the output this AI will learn
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public void addLearningSession(String aiId, String learningSessionId, String outputName) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          Argument.IsNotNullOrEmpty(learningSessionId, "learningSessionId");
          Argument.IsNotNullOrEmpty(outputName, "outputName");

          // TODO support for customer who tries to add an ls that does not exist
          String url = "/ais/" + aiId + "/learningSessions/";
          client.getConnection().post(AIEmptyResponse.class, url, null, new AIBody(learningSessionId, outputName),
                    this.httpMessageTransformer);
     }

     /**
      * Add a {@link Dataset} to the AI
      * 
      * @param aiId      The ID of the AI to which you want to add the dataset
      * @param datasetId The ID of the dataset to add
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public void addDataset(String aiId, String datasetId) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          Argument.IsNotNullOrEmpty(datasetId, "datasetId");

          String url = "/ais/" + aiId + "/datasets/";
          client.getConnection().post(AIEmptyResponse.class, url, null, new AIBody(datasetId, 2),
                    this.httpMessageTransformer);
     }

     /**
      * Retrieve a list of all learning sessions associated with an AI
      * 
      * @param aiId The ID of the AI to retrieve learning sessions for
      * 
      * @return {@link LearningSessionList}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public LearningSessionList getLearningSessions(String aiId) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          LearningSessionList list = client.getConnection().get(LearningSessionList.class, "/learningSessions/",
                    new AIBody(aiId, 3).toParameters(), this.httpMessageTransformer);
          return setClient(list);
     }

     /**
      * Retrieve a list of all learning sessions associated with an AI
      * 
      * @param aiId   The ID of the AI to retrieve learning sessions for
      * @param params The {@link PagingParams} object for this list
      * 
      * @return {@link LearningSessionList}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public LearningSessionList getLearningSessions(String aiId, PagingParams params) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          Map<String, Object> parm = params.toParameters();
          parm.put("aiId", aiId);
          LearningSessionList list = client.getConnection().get(LearningSessionList.class, "/learningSessions/", parm,
                    this.httpMessageTransformer);
          return setClient(list);
     }

     private LearningSessionList setClient(LearningSessionList list) {
          for (LearningSession ls : list.getItems()) {
               ls.setClient(this.client);
          }
          return list;
     }

     /**
      * Retrieve the {@link Output} associated with an AI and target
      * 
      * @param aiId         The ID of the AI to retrieve an output for
      * @param outputTarget The target of this output
      * 
      * @return {@link Output}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     public Output getOutput(String aiId, String outputTarget) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          Argument.IsNotNullOrEmpty(outputTarget, "outputTarget");
          String url = "ais/" + aiId + "/outputs/" + outputTarget;
          Output output = client.getConnection().get(Output.class, url, null, this.httpMessageTransformer);
          output.setAiId(aiId);
          return output;
     }

     /**
      * Make predictions on an AI, target and dataset. Note an AI must be trained
      * with {@link AI#learn} or an existing learning session must be added to the AI
      * with {@link AI#addLearningSession} or {@link IAIClient#addLearningSession}
      * before predictions can occur.
      * 
      * @param aiId     The ID of the AI to retrieve predictions for
      * @param target   The name of the selected target feature to predict on
      * @param filePath The data on which to predict via a filepath on the local
      *                 system
      * 
      * @return {@link PredictionList}
      * 
      * @throws ClientException       when 4xx or 5xx response is received from
      *                               server, or errors in parsing the response.
      * @throws FileNotFoundException when a file with the specified pathname does
      *                               not exist, or if the file does exist but is
      *                               inaccessible for some reason.
      */
     @Override
     public PredictionList predict(String aiId, String target, String filePath)
               throws ClientException, FileNotFoundException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          Argument.IsNotNullOrEmpty(target, "target");
          Argument.IsNotNullOrEmpty(filePath, "filePath");

          return client.predictions().aiPredict(aiId, target, filePath);
     }

     /**
      * Make predictions on an AI, target and dataset. Note an AI must be trained
      * with {@link AI#learn} or an existing learning session must be added to the AI
      * with {@link AI#addLearningSession} or {@link IAIClient#addLearningSession}
      * before predictions can occur.
      * 
      * @param aiId     The ID of the AI to retrieve predictions for
      * @param target   The name of the selected target feature to predict on
      * @param filePath The data on which to predict via a filepath on the local
      *                 system
      * 
      * @return {@link PredictionList}
      * 
      * @throws ClientException       when 4xx or 5xx response is received from
      *                               server, or errors in parsing the response.
      * @throws FileNotFoundException when a file with the specified pathname does
      *                               not exist, or if the file does exist but is
      *                               inaccessible for some reason.
      */
     @Override
     public PredictionList infer(String aiId, String target, String filePath)
               throws ClientException, FileNotFoundException {
          return predict(aiId, target, filePath);
     }

     /**
      * Retrieve a list of all outputs associated with an AI.
      * 
      * @param aiId The ID of the AI to retrieve outputs for
      * 
      * @return {@link OutputList}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public OutputList getOutputs(String aiId) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          String url = "ais/" + aiId + "/outputs/";
          OutputList list = client.getConnection().get(OutputList.class, url, null, this.httpMessageTransformer);
          return setClientAndAIId(list, aiId);
     }

     /**
      * Retrieve a list of all outputs associated with an AI.
      * 
      * @param aiId   The ID of the AI to retrieve outputs for
      * @param params The {@link PagingParams} object for this list
      * 
      * @return {@link OutputList}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public OutputList getOutputs(String aiId, PagingParams params) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          String url = "ais/" + aiId + "/outputs/";
          Map<String, Object> parm = params.toParameters();
          OutputList list = client.getConnection().get(OutputList.class, url, parm, this.httpMessageTransformer);
          return setClientAndAIId(list, aiId);
     }

     private OutputList setClientAndAIId(OutputList list, String aiId) {
          for (Output output : list.getItems()) {
               output.setClient(this.client);
               output.setAiId(aiId);
          }
          return list;
     }

     /**
      * Add an {@link Output} to the AI
      * 
      * @param aiId              The ID of the AI to which you want to add the
      *                          dataset
      * @param learningSessionId The ID of the learning session
      * @param outputName        The name of the output
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public void addOutput(String aiId, String learningSessionId, String outputName) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          Argument.IsNotNullOrEmpty(learningSessionId, "learningSessionId");
          Argument.IsNotNullOrEmpty(outputName, "outputName");

          String url = "ais/" + aiId + "/outputs/";
          Map<String, Object> body = new HashMap<>();
          body.put("learningSessionId", learningSessionId);
          body.put("outputName", outputName);

          client.getConnection().put(Output.class, url, null, body, this.httpMessageTransformer);
     }

     /**
      * Retrieve features of an output associated with an AI and a target
      * 
      * @param aiId       The ID of the AI to retrieve output features for
      * @param outputName The name of the output
      * 
      * @return {@link OutputFeatures}
      * 
      * @throws ClientException when 4xx or 5xx response is received from server, or
      *                         errors in parsing the response.
      */
     @Override
     public OutputFeatures getOutputFeatures(String aiId, String outputName) throws ClientException {
          Argument.IsNotNullOrEmpty(aiId, "aiId");
          Argument.IsNotNullOrEmpty(outputName, "outputName");

          String url = "ais/" + aiId + "/outputs/" + outputName + "/features/";

          return client.getConnection().get(OutputFeatures.class, url, null, this.httpMessageTransformer);
     }

     @Override
     public String toString() {
          return "AIClient";
     }

}
