package com.datarobot.impl;

import com.datarobot.IDataRobotAIClient;
import com.datarobot.model.INeedClient;
import com.datarobot.model.AICreationResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.google.api.client.http.InputStreamContent;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.MultipartContent;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.http.HttpRequestFactory;
import com.google.api.client.http.HttpMediaType;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.http.FileContent;
import com.google.api.client.http.GenericUrl;
import com.google.api.client.http.HttpContent;
import com.google.api.client.http.HttpResponseException;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.datarobot.model.Deployment;
import com.datarobot.model.ErrorResponse;
import com.datarobot.util.Action;
import com.datarobot.util.HttpMethod;
import com.datarobot.util.JacksonMapperHttpContent;
import com.datarobot.util.JacksonMapperParser;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.*;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.UUID;


/**
 * The {@link ApiConnection ApiConnection} object is responsible for low-level HTTP requests and mapping JSON responses
 * to POJO objects. It is not meant to be used directly.
 */
public class ApiConnection {
    private IDataRobotAIClient client;
    private String key;
    private String endpoint;
    private HttpTransport httpTransport;
    private ObjectMapper mapper = new ObjectMapper();
    public static final String JSON_TYPE = "application/json";
    private static final Logger logger = LogManager.getLogger(ApiConnection.class);
    private int readTimeout = 300000;

    ObjectMapper getObjectMapper(){
        return mapper;
    }

    String getApiKey() {
        return key;
    }
    private HttpRequestFactory requestFactory;

    /**
     * The {@code ApiConnection} object is responsible for low-level HTTP requests /responses and mapping JSON
     * responses to POJO objects.
     *
     * @param endpoint The URL of the DataRobot AI API.
     * @param key The API key used in the Authorization header to gain access to the DataRobot AI API.
     * @param client A reference to the DataRobot AI API Client so it can assign it to models
     */
    public ApiConnection(String endpoint, String key, IDataRobotAIClient client) {
        this(endpoint, key, new NetHttpTransport(), client);
    }

    /**
     * The {@code ApiConnection} object is responsible for low-level HTTP requests /responses and mapping JSON
     * responses to POJO objects and this constructor is used for mocking HTTP transport objects.
     * <P>
     * @param key The api key from your account.
     * @param endpoint URL of DataRobot AI API
     * @param httpTransport HttpTransport to provide mock class for unit tests
     * @param client A reference to the DataRobot AI API Client so it can assign it to models
     */
    ApiConnection(String endpoint, String key, HttpTransport httpTransport, IDataRobotAIClient client) {
        this.client = client;
        this.endpoint = endpoint;
        this.httpTransport = httpTransport;
        this.key = key;
        mapper.registerModule(new JodaModule());
        mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);

        this.requestFactory = httpTransport.createRequestFactory(new HttpRequestInitializer() {
            @Override
            public void initialize(HttpRequest request) throws IOException {
                request.setSuppressUserAgentSuffix(true);
                request.setParser(new JacksonMapperParser(getObjectMapper()));
                // TODO: look into usage of UnsuccessfulResponseHandler
                //request.setUnsuccessfulResponseHandler();
                // DO NOT FOLLOW REDIRECTS
                request.setFollowRedirects(false);
                // Since 301's throw exceptions, we need to handle this ourselves.
                request.setThrowExceptionOnExecuteError(false);
                request.setReadTimeout(readTimeout);
                request.setHeaders(new HttpHeaders()
                        .setAuthorization("Bearer " + getApiKey() )
                        .setUserAgent(DataRobotAIClient.CLIENT_VERSION));

            }
        });
    }

    public void get(String path, Map<String,Object>parameters, Action<HttpRequest,
            HttpResponse> httpMessageTransformer, OutputStream output) throws ClientException {
        get(path, parameters, httpMessageTransformer, output, null);
    }

    public void get(String path, Map<String,Object> parameters, Action<HttpRequest,
            HttpResponse> httpMessageTransformer, OutputStream output, String acceptType)
            throws ClientException {
        try {

            if (StringUtils.isEmpty(acceptType)) {
                acceptType = JSON_TYPE;
            }

            GenericUrl uri = prepareURI(path, parameters);

            HttpRequest request = requestFactory.buildGetRequest(uri);
            request.getHeaders().setAccept(acceptType);
            makeRequest(request, httpMessageTransformer, output);
        } catch (URISyntaxException uris) {
            throw new ClientException("Invalid URI.", uris);
        } catch (IOException ioe) {
            throw new ClientException("Internal Error.", ioe);
        }
    }

    public <T> T get(Class<T> type, String path, Map<String,Object> parameters,
                     Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException {
        return get(type, path, parameters, httpMessageTransformer, (String)null);
    }

    public <T> T get(Class<T> type, String path, Map<String,Object> parameters,
                     Action<HttpRequest, HttpResponse> httpMessageTransformer, String acceptType) throws ClientException {
        try {
            if (StringUtils.isEmpty(acceptType)) {
                acceptType = JSON_TYPE;
            }

            GenericUrl uri = prepareURI(path, parameters);

            HttpRequest request = requestFactory.buildGetRequest(uri);
            request.getHeaders().setAccept(acceptType);

            return makeRequest(type, request, httpMessageTransformer);
        } catch (URISyntaxException uris) {
            throw new ClientException("Invalid URI.", uris);
        } catch (IOException ioe) {
            throw new ClientException("Internal Error.", ioe);
        }
    }

    public HttpHeaders head(String path, Map<String,Object> parameters,
                            Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException
    {
        String acceptType = JSON_TYPE;
        try {
            GenericUrl uri = prepareURI(path, parameters);

            HttpRequest request = requestFactory.buildHeadRequest(uri);
            request.getHeaders().setAccept(acceptType);

            return makeRequest(request, httpMessageTransformer).getHeaders();
        } catch (URISyntaxException uris) {
            throw new ClientException("Invalid URI.", uris);
        } catch (IOException ioe) {
            throw new ClientException("Internal Error.", ioe);
        }
    }

    public <T> T put(Class<T> type, String path, Map<String, Object> parameters, Object body,
                     Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException
    {
        return sendObjectContent(type, path, parameters, HttpMethod.PUT, body, httpMessageTransformer);
    }

    public <T> T putFile(Class<T> type, String path, Map<String, Object> parameters, InputStream body, String streamName,
                     String contentType, Action<HttpRequest, HttpResponse> httpMessageTransformer)
            throws ClientException
    {
        return sendFileContent(type, path, parameters, HttpMethod.PUT, body, streamName,
                contentType, httpMessageTransformer);
    }

    public <T> T putStream(Class<T> type, String path, Map<String, Object> parameters, File body,
                         String contentType, Action<HttpRequest, HttpResponse> httpMessageTransformer)
            throws ClientException
    {
        return sendStreamContent(type, path, parameters, HttpMethod.PUT, body, contentType,
                httpMessageTransformer);
    }

    public <T> T post(Class<T> type, String path, Map<String,Object> parameters, Object body,
                      Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException {
        return sendObjectContent(type, path, parameters, HttpMethod.POST, body, httpMessageTransformer);
    }

    public <T> T postStream(Class<T> type, String path, Map<String, Object> parameters, File body,
                          String contentType, Action<HttpRequest, HttpResponse> httpMessageTransformer)
            throws ClientException {
        return sendStreamContent(type, path, parameters, HttpMethod.POST, body , contentType,
                httpMessageTransformer);
    }

    public <T> T postFile(Class<T> type, String path, Map<String, Object> parameters, String streamName, InputStream body,
                      String contentType, Action<HttpRequest, HttpResponse> httpMessageTransformer)
            throws ClientException {
        return sendFileContent(type, path, parameters, HttpMethod.POST, body, streamName, contentType,
                httpMessageTransformer);
    }

    private <T> T sendObjectContent(Class<T> type, String path, Map<String, Object> parameters, HttpMethod method,
                                    Object body, Action<HttpRequest, HttpResponse> httpMessageTransformer)
            throws ClientException {
        String acceptType = JSON_TYPE;
        try {

            GenericUrl uri = prepareURI(path, parameters);

            HttpRequest request = null;
            HttpContent contentSend = new JacksonMapperHttpContent(getObjectMapper(), body);

            switch (method) {
                case PUT:
                    request = requestFactory.buildPutRequest(uri, contentSend);
                    break;
                case POST:
                    request = requestFactory.buildPostRequest(uri, contentSend);
                    break;
            }
            request.getHeaders().setAccept(acceptType).setContentType(acceptType);
            return makeRequest(type, request, httpMessageTransformer);
        } catch (URISyntaxException uris) {
            throw new ClientException("Invalid URI.", uris);
        } catch (IOException ioe) {
            throw new ClientException("Internal Error.", ioe);
        }
    }


    private <T> T sendStreamContent(Class<T> type, String path, Map<String, Object> parameters,
                                    HttpMethod method, File file, String contentType,
                                    Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException
    {
        String acceptType = JSON_TYPE;
        try {
            GenericUrl uri = prepareURI(path, parameters);
            HttpRequest request = null;
            FileContent body = new FileContent("text/csv", file);
            switch (method) {
                case PUT:
                    request = requestFactory.buildPutRequest(uri, body);
                    break;
                case POST:
                    request = requestFactory.buildPostRequest(uri, body);
                    break;
            }
            // datarobot-key needs to be added into the header for prediction requests
            if (path.contains("/predApi/v1.0/")) {
            	request.getHeaders()
            		.setAccept(acceptType)
            		.setContentType(contentType)
            		.set("datarobot-key", ((Deployment) parameters.get("deployment")).getDataRobotKey());
            }
            else {
            	request.getHeaders().setAccept(acceptType).setContentType(contentType);
            	
            }
            return makeRequest(type, request, httpMessageTransformer);
        } catch (URISyntaxException use) {
            throw new ClientException("Invalid URI.", use);
        } catch (IOException ioe) {
            throw new ClientException("Internal Error.", ioe);
        }
    }


    private <T> T sendFileContent(Class<T> type, String path, Map<String, Object> parameters,
                                  HttpMethod method, InputStream body, String streamName, String contentType,
                                  Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException
    {
        String acceptType = JSON_TYPE;
        try {
            GenericUrl uri = prepareURI(path, parameters);
            HttpRequest request = null;
            
            // Note that the prediction server does not accept Multipart requests - this should only be used for dataset uploads to main DR
            MultipartContent.Part part = new MultipartContent.Part()
                        .setContent(new InputStreamContent(contentType, body))
                        .setHeaders(new HttpHeaders().set("Content-Disposition",
                                String.format("form-data; name=\"file\"; filename=\"%s\"", streamName)
                        ));

            MultipartContent content = new MultipartContent()
                    .setMediaType(new HttpMediaType("multipart/form-data")
                            .setParameter("boundary", UUID.randomUUID().toString()))
                    .addPart(part);

            switch (method) {
                case PUT:
                    request = requestFactory.buildPutRequest(uri, content);
                    break;
                case POST:
                    request = requestFactory.buildPostRequest(uri, content);
                    break;
            }
            request.getHeaders().setAccept(acceptType);
            return makeRequest(type, request, httpMessageTransformer);
        } catch (URISyntaxException use) {
            throw new ClientException("Invalid URI.", use);
        } catch (IOException ioe) {
            throw new ClientException("Internal Error.", ioe);
        }
    }


    public void delete(String path, Map<String, Object> parameters, Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException {
        String acceptType = JSON_TYPE;
        try {
            GenericUrl uri = prepareURI(path, parameters);

            HttpRequest request = requestFactory.buildDeleteRequest(uri);
            request.getHeaders().setAccept(acceptType);
            HttpResponse response = makeRequest(request, httpMessageTransformer);
            response.disconnect();
        } catch (URISyntaxException uris) {
            throw new ClientException("Invalid URI.", uris);
        } catch (IOException ioe) {
            throw new ClientException("Internal Error.", ioe);
        }
    }

    public <T> T makeRequest(Class<T> type, HttpRequest request, Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException {
        HttpResponse response;
        T object = null;
        try {
            response = makeRequest(request, httpMessageTransformer);            

            try {
                // Adding 201 for AI creation
            	// Hack --> type.getTypeName().equals(AICreationResponse.class.getName()) is because ai.create() returns an AI; ai.addLearningSession() returns nothing (both have 201)
            	// This class is being configured in follow up
                if (response.getStatusCode() == 200 || response.getStatusCode() == 201 && type.getTypeName().equals(AICreationResponse.class.getName()) || response.getStatusCode() == 202) {
                    object = response.parseAs(type);
                }
                // If no body to deserialize response into an object, create a new empty one one
                if (object == null) {
                    object = type.newInstance();
                }
                
                if (INeedClient.class.isAssignableFrom(object.getClass())) {
                    ((INeedClient)object).setClient(this.client);
                }
                return object;
            } finally {
                response.disconnect();
            }
        } catch (HttpResponseException hre) {
            throw GenerateClientException(hre);
        } catch (IOException ioe) {
            throw new ClientException("IO Error while making HTTP Request", ioe);
        } catch (ClientException nce) {
            throw nce;
        } catch (Exception e) {
            throw new ClientException("Error while making HTTP Request: " + e.getMessage(), e);
        }
    }

    private void makeRequest(HttpRequest request, Action<HttpRequest, HttpResponse> httpMessageTransformer, OutputStream output) throws ClientException {
        HttpResponse response;

        try {
            response = makeRequest(request, httpMessageTransformer);
            try {
                // Write content to stream if status is complete
                response.download(output);
            } finally {
                response.disconnect();
            }
        } catch (HttpResponseException hre) {
            throw GenerateClientException(hre);
        } catch (IOException ioe) {
            throw new ClientException("IO Error while making HTTP Request: " + ioe.getMessage());
        }
    }

    private HttpResponse makeRequest(HttpRequest request, Action<HttpRequest, HttpResponse> httpMessageTransformer) throws ClientException {
        HttpResponse response;

        try {
            if (httpMessageTransformer != null)
                httpMessageTransformer.invoke(request, null);
            
            logger.debug(String.format("HTTP %s Request: %s", request.getRequestMethod(), request.getUrl().toString()));
            response = request.execute();
            logger.debug(String.format("HTTP %s: %s", response.getStatusCode(), request.getUrl().toString()));

            if (!response.isSuccessStatusCode()
                    && response.getStatusCode() != 301
                    && response.getStatusCode() != 303) {
                throw new HttpResponseException(response);
            }

            if (httpMessageTransformer != null)
                httpMessageTransformer.invoke(request, response);

            return response;
        } catch (HttpResponseException hre) {
            throw GenerateClientException(hre);
        } catch (IOException ioe) {
            throw new ClientException("IO Error while making HTTP Request: " + ioe.getMessage());
        } catch (Exception e) {
            throw new ClientException("Error while making HTTP Request: " + e.getMessage());
        }
    }

    GenericUrl prepareURI(String path, Map<String,Object> parameters) throws URISyntaxException {
    	java.net.URI uri;
    	GenericUrl url;
    	
    	// if this is a predictions request, a different base URL is used
    	if (path.contains("predApi/v1.0/")) {
    		String fullPath = path + "/";
    		uri = new URI(fullPath);
    		url = new GenericUrl(fullPath);
    		return url;
    	}
    	// Removes the trailing slash if it was sent (internal)
    	else if (path.startsWith("/")) {
            path = path.substring(1);
            uri = new URI(endpoint + path);
            url = new GenericUrl(endpoint + path);
        }
    	else {
    		uri = new URI(endpoint + path);
            url = new GenericUrl(endpoint + path);
    	}
    	// eliminates "deployment" parameter from being added on to the end of the request URL
    	if (path.contains("predApi/v1.0/")) {
    		return url;
    	}
    	else if (!(parameters == null || parameters.size() == 0)) {
            url.putAll(parameters);
        }

        return url;
    }

    private ClientException GenerateClientException(HttpResponseException responseException) {
        try {

            // map the json error content to ErrorResponse object
            ErrorResponse errorResponse = mapper.readValue(responseException.getContent(), ErrorResponse.class);
            // Include fields that aren't in content body
            errorResponse.setStatusCode(responseException.getStatusCode());
            errorResponse.setStatusMessage(responseException.getStatusMessage());

            logger.error(errorResponse.toString());

            if (errorResponse != null) {
                StringBuilder errorBuilder = new StringBuilder();
                errorBuilder.append("API Error: ")
                        .append(errorResponse.getStatusCode())
                        .append(" -");
                if (errorResponse.getMessage() != null) {
                    errorBuilder.append(" ").append(errorResponse.getMessage());
                }
                if (errorResponse.getStatusMessage() != null) {
                    errorBuilder.append(" ").append(errorResponse.getStatusMessage());
                }
                return new ClientException(errorBuilder.toString(), errorResponse, responseException);
            }
            return new ClientException("API Error: " + errorResponse.getStatusCode() + " - no details provided.", responseException.getStatusCode(), responseException);

        } catch (IOException ioe) {
            ClientException nce = new ClientException("Error processing error response content.", responseException);

            try {
                ErrorResponse e = new ErrorResponse();
                e.setAdditionalProperty("Error Content", responseException.getContent());
                e.setStatusCode(responseException.getStatusCode());
                nce.setStatusCode(responseException.getStatusCode());
            } catch (Exception ex) {}

            return nce;
        }
    }
}
