package com.datarobot.impl;

import com.datarobot.*;
import com.datarobot.model.*;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.datarobot.util.Action;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.FileNotFoundException;

/**
 * The {@link DataRobotAIClient} object is the main class for using the
 * DataRobot AI API.
 */
public class DataRobotAIClient implements IDataRobotAIClient {
    private String key;
    private String configuredUrl;
    private ApiConnection apiConnection;
    private IDatasetClient data;
    private IStatusClient status;
    private ILearningSessionClient learning;
    private IPredictionClient prediction;
    private IAIClient ais;
    private static final Logger logger = LogManager.getLogger(DataRobotAIClient.class);
    private Action<HttpRequest, HttpResponse> httpMessageTransformer;
    /**
     * The current version of the client
     */
    public final static String CLIENT_VERSION = "DataRobotAIClient-Java-API-Client/1.0.0";
    /**
     * The base URL
     */
    public final static String BASE_URL = "https://developers.datarobot.com/aiapi/";

    /**
     * The currently configured API key used by this instance of the client.
     *
     * @return The currently configured API Key.
     */
    public String getApiKey() {
        return key;
    }

    /**
     * Gets the URL endpoint the client is currently connected to.
     *
     * @return the URL of the DataRobot AI API that is currently being used.
     */
    public String getConfiguredUrl() {
        return configuredUrl != null ? configuredUrl : BASE_URL;
    }

    /**
     * Create a new instance of the {@link DataRobotAIClient} from an
     * {@code API_KEY} environment variable
     * 
     */
    public DataRobotAIClient() {
        this(System.getenv("API_KEY"));
    }

    /**
     * Create a new instance of the {@link DataRobotAIClient}
     * 
     * @param key The API Key from your account
     */
    public DataRobotAIClient(String key) {
        this(key, BASE_URL, new NetHttpTransport());
    }

    /**
     * Create a new instance of the {@link DataRobotAIClient}
     * 
     * @param key      The API Key from your account
     * @param endpoint The URL this client will connect tos
     */
    public DataRobotAIClient(String key, String endpoint) {
        this(key, endpoint, new NetHttpTransport());
    }

    /**
     * Internal provided for testing use only
     */
    public DataRobotAIClient(String key, String endpoint, HttpTransport httpTransport) throws IllegalArgumentException {
        this.key = key;

        if (endpoint == null || endpoint.isEmpty())
            throw new IllegalArgumentException(
                    "No value was provided for the endpoint. If you do not know the value, use the ctor with the api key only");
        if (!endpoint.endsWith("/")) {
            endpoint = endpoint + "/";
        }

        configuredUrl = endpoint;
        apiConnection = new ApiConnection(endpoint, key, httpTransport, this);

        data = new DatasetClient(this);
        status = new StatusClient(this);
        learning = new LearningSessionClient(this);
        prediction = new PredictionClient(this);
        ais = new AIClient(this);
    }

    /**
     * Internal
     *
     * Returns the HTTP connection used for all calls in the client.
     * 
     * @return The HTTP connection used for all calls in the client.
     */
    @Override
    public ApiConnection getConnection() {
        return this.apiConnection;
    }

    /**
     * Internal
     */
    public Action<HttpRequest, HttpResponse> getHttpMessageTransformer() {
        return httpMessageTransformer;
    }

    /**
     * Internal
     */
    public void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer) {
        this.httpMessageTransformer = httpMessageTransformer;
        UpdateChildClientTransformers();
    }

    private void UpdateChildClientTransformers() {
        if (data.getHttpMessageTransformer() == null) {
            data.setHttpMessageTransformer(httpMessageTransformer);
        }
        if (status.getHttpMessageTransformer() == null) {
            status.setHttpMessageTransformer(httpMessageTransformer);
        }
        if (learning.getHttpMessageTransformer() == null) {
            learning.setHttpMessageTransformer(httpMessageTransformer);
        }
        if (prediction.getHttpMessageTransformer() == null) {
            learning.setHttpMessageTransformer(httpMessageTransformer);
        }
        if (ais.getHttpMessageTransformer() == null) {
            ais.setHttpMessageTransformer(httpMessageTransformer);
        }
    }

    /**
     * {@link Dataset IDatasetClient} based API operations
     */
    @Override
    public IDatasetClient data() {
        return this.data;
    }

    /**
     * {@link Status IStatusClient} based API operations
     */
    @Override
    public IStatusClient status() {
        return this.status;
    }

    /**
     * {@link Prediction IPredictionClient} based API operations
     */
    @Override
    public IPredictionClient predictions() {
        return this.prediction;
    }

    /**
     * {@link AI IAIClient} based API operations
     */
    @Override
    public IAIClient ais() {
        return this.ais;
    }

    /**
     * {@link ILearningSessionClient} based API operations
     */
    @Override
    public ILearningSessionClient learning() {
        return this.learning;
    }

    /**
     * Internal
     */
    @Override
    public Ping ping() throws ClientException {
        return this.apiConnection.get(Ping.class, "/ping", null, httpMessageTransformer);
    }

    /**
     * Create an AI that will learn from past data to predict on new data. Note a
     * {@link AI} can be sometimes referred to as an AI.
     * 
     * @param name The name of the AI
     * 
     * @throws ClientException      when 4xx or 5xx response is received from
     *                              server, or errors in parsing the response.
     * @throws InterruptedException when a thread is waiting, sleeping, or otherwise
     *                              occupied, and the thread is interrupted, either
     *                              before or during the activity.
     */
    public AI createAI(String name) throws ClientException, InterruptedException {
        Argument.IsNotNullOrEmpty(name, "name");
        return this.ais.createAI(name);
    }

    @Override
    public String toString() {
        return "DataRobotAIClient";
    }

}
