package com.datarobot.impl;

import com.datarobot.IDataRobotAIClient;
import com.datarobot.IDatasetClient;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;
import com.datarobot.model.*;
import com.datarobot.util.Action;
import com.datarobot.model.PagingParams;

import java.io.*;
import java.net.URI;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * The {@link DatasetClient} object provides access to the dataset endpoints of
 * the DataRobot AI API. This object is not meant to be used directly but
 * through the {@code DataRobotAIClient.data()} method.
 */
public class DatasetClient implements IDatasetClient, IDoGetCallback<Dataset> {
    private IDataRobotAIClient client;
    private Action<HttpRequest, HttpResponse> httpMessageTransformer = null;
    private final ExecutorService pool = Executors.newFixedThreadPool(10);

    /**
     * {@link Dataset IDatasetClient} based API operations. Users will not need to
     * instantiate this object directly. It can be accessed through
     * {@code DataRobotAIClient.data()}.
     * 
     * @param client {@link DataRobotAIClient}
     */
    public DatasetClient(IDataRobotAIClient client) {
        this.client = client;
    }

    /**
     * internal
     */
    public Action<HttpRequest, HttpResponse> getHttpMessageTransformer() {
        return httpMessageTransformer;
    }

    /**
     * internal
     */
    public void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer) {
        this.httpMessageTransformer = httpMessageTransformer;
    }

    /**
     * Upload a new dataset. See the Dataset page in the documentation for size
     * limitations.
     * 
     * @param sourceFile The path of the local file to upload
     * 
     * @return The new {@link Dataset}
     * 
     * @throws ClientException       when 4xx or 5xx response is received from
     *                               server, or errors in parsing the response.
     * @throws FileNotFoundException when a file with the specified pathname does
     *                               not exist, or if the file does exist but is
     *                               inaccessible for some reason.
     * @throws InterruptedException  when a thread is waiting, sleeping, or
     *                               otherwise occupied, and the thread is
     *                               interrupted, either before or during the
     *                               activity.
     */
    @Override
    public Dataset importFile(String sourceFile) throws ClientException, FileNotFoundException, InterruptedException {
        Argument.IsNotNull(sourceFile, "sourceFile");
        StatusTask<Dataset> task = importFileTask(sourceFile);
        return task.getResult();
    }

    /**
     * Import a dataset from a url. See the Dataset page in the documentation for
     * size limitations.
     * 
     * @param url The url to a publicly available file
     * 
     * @return The new {@link Dataset}
     * 
     * @throws ClientException      when 4xx or 5xx response is received from
     *                              server, or errors in parsing the response.
     * @throws InterruptedException when a thread is waiting, sleeping, or otherwise
     *                              occupied, and the thread is interrupted, either
     *                              before or during the activity.
     */
    @Override
    public Dataset importUrl(String url) throws ClientException, InterruptedException {
        Argument.IsNotNull(url, "URL");
        StatusTask<Dataset> task = importURLTask(URI.create(url));
        return task.getResult();
    }

    // Pending for now
    // @Override
    // public DatasetData importStream(String fileName, InputStream stream) throws
    // ClientException, InterruptedException {
    // Argument.IsNotNull(fileName, "fileName");
    // Argument.IsNotNull(stream, "stream");
    // StatusTask<DatasetData> task = importStreamTask(fileName, stream);
    // return task.getResult();
    // }

    private StatusTask<Dataset> importFileTask(String sourceFile) throws ClientException, FileNotFoundException {
        File initialFile = new File(sourceFile);
        InputStream inputStream = new FileInputStream(initialFile);
        Path p = Paths.get(sourceFile);

        try {
            return importStreamTask(p.getFileName().toString(), inputStream);
        } finally {
            try {
                inputStream.close();
            } catch (IOException ioe) {
                /* blackhole */ }
        }
    }

    private StatusTask<Dataset> importURLTask(URI uri) throws ClientException {
        // brief hack for now until I decide what to do with the IDatasetSource
        // interface
        DatasetUrlSource body = new DatasetUrlSource("Hack", uri);

        DatasetImportResponse urlResponse = client.getConnection().post(DatasetImportResponse.class,
                "datasets/urlImports/", null, body, this.httpMessageTransformer);
        return new StatusTask<>(this.client, urlResponse, this);
    }

    private StatusTask<Dataset> importStreamTask(String fileName, InputStream stream) throws ClientException {
        // brief hack for now until I decide what to do with the IDatasetSource
        // interface
        DatasetStreamSource streamSource = new DatasetStreamSource(fileName, stream);
        DatasetImportResponse fileResponse = client.getConnection().postFile(DatasetImportResponse.class,
                "datasets/fileImports/", null, streamSource.getName(), streamSource.getStream(),
                streamSource.getContentType(), this.httpMessageTransformer);

        return new StatusTask<>(this.client, fileResponse, this);
    }

    /**
     * Start the process of uploading a dataset from a local file. This method can
     * be used to avoid blocking during large file imports. See the Dataset page in
     * the documentation for size limitations.
     * 
     * @param fileName The path of the local file to upload
     * 
     * @return {@link Future}{@link Dataset}
     * 
     * @throws FileNotFoundException when a file with the specified pathname does
     *                               not exist, or if the file does exist but is
     *                               inaccessible for some reason.
     * @throws ClientException       when 4xx or 5xx response is received from
     *                               server, or errors in parsing the response.
     */
    @Override
    public Future<Dataset> startImportFile(final String fileName) throws ClientException, FileNotFoundException {
        final StatusTask<Dataset> task = importFileTask(fileName);
        return pool.submit(new Callable<Dataset>() {
            @Override
            public Dataset call() throws ClientException, InterruptedException {
                return (Dataset) task.getResult();
            }
        });
    }

    /**
     * Start the process of uploading a dataset from a local file. This method can
     * be used to avoid blocking during large file imports. See the Dataset page in
     * the documentation for size limitations.
     * 
     * @param url The path of the url to be imported
     * 
     * @return {@link Future}{@link Dataset}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public Future<Dataset> startimportUrl(final String url) throws ClientException {
        final StatusTask<Dataset> task = importURLTask(URI.create(url));
        return pool.submit(new Callable<Dataset>() {
            @Override
            public Dataset call() throws ClientException, InterruptedException {
                return task.getResult();
            }
        });
    }

    /**
     * Retrieve a list of uploaded datasets associated with this account
     * 
     * @return {@link DatasetList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public DatasetList list() throws ClientException {
        return this.list(new PagingParams(0, 0));
    }

    /**
     * Retrieve a list of uploaded datasets associated with this account
     * 
     * @param params The {@link PagingParams} object for this list
     * 
     * @return {@link DatasetList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public DatasetList list(PagingParams params) throws ClientException {
        Map<String, Object> parameters = null;

        if (params != null) {
            parameters = params.toParameters();
        }
        DatasetList list = client.getConnection().get(DatasetList.class, "datasets/", parameters,
                httpMessageTransformer);
        return setClient(list);
    }

    /**
     * Retrieve a list of uploaded datasets associated with this AI
     * 
     * @param aiId Datasets for this AI will only be returned
     * 
     * @return {@link DatasetList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    public DatasetList list(String aiId) throws ClientException {
        Argument.IsNotNullOrEmpty(aiId, aiId);
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("aiId", aiId);

        DatasetList list = client.getConnection().get(DatasetList.class, "datasets/", parameters,
                this.httpMessageTransformer);
        return setClient(list);
    }

    private DatasetList setClient(DatasetList list) {
        for (Dataset dataset : list.getItems()) {
            dataset.setClient(this.client);
        }
        return list;
    }

    /**
     * Retrieve a dataset
     * 
     * @param datasetId The ID of the dataset to retrieve
     * 
     * @return The queried {@link Dataset}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public Dataset get(String datasetId) throws ClientException {
        Argument.IsNotNullOrEmpty(datasetId, "datasetId");

        return client.getConnection().get(Dataset.class, "datasets/" + datasetId + "/", null,
                this.httpMessageTransformer);
    }

    /**
     * Delete a dataset
     * 
     * @param datasetId The ID of the dataset to delete
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public void delete(String datasetId) throws ClientException {
        Argument.IsNotNullOrEmpty(datasetId, "datasetId");

        client.getConnection().delete("/datasets/" + datasetId + "/", null, this.httpMessageTransformer);
    }

    @Override
    public String toString() {
        return "DatasetClient";
    }
}
