package com.datarobot.impl;

import com.datarobot.ILearningSessionClient;
import com.datarobot.IDataRobotAIClient;
import com.datarobot.IPredictionClient;
import com.datarobot.model.*;
import com.datarobot.util.Action;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;

import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * The {@link LearningSessionClient} object provides access to the learning
 * endpoints of the DataRobot AI API. This object is not meant to be used
 * directly but through the {@code IDataRobotAIClient#learning()} method.
 */
public class LearningSessionClient implements ILearningSessionClient, IDoGetCallback<LearningSession> {
    private IDataRobotAIClient client;
    private Action<HttpRequest, HttpResponse> httpMessageTransformer = null;
    private final ExecutorService pool = Executors.newFixedThreadPool(10);

    /**
     * {@link ILearningSessionClient} based API operations. Users will not need to
     * instantiate this object directly. It can be accessed through
     * {@code DataRobotAIClient.learning()}.
     * 
     * @param client {@link DataRobotAIClient}
     */
    public LearningSessionClient(IDataRobotAIClient client) {
        this.client = client;
    }

    /**
     * internal
     */
    @Override
    public Action<HttpRequest, HttpResponse> getHttpMessageTransformer() {
        return httpMessageTransformer;
    }

    /**
     * internal
     */
    @Override
    public void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer) {
        this.httpMessageTransformer = httpMessageTransformer;
    }

    /**
     * Retrieve a {@link LearningSession} by id
     * 
     * @param id The ID of the learning session to retrieve
     * 
     * @return The queried {@link LearningSession}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    public LearningSession get(String id) throws ClientException {
        Argument.IsNotNullOrEmpty(id, "id");

        return client.getConnection().get(LearningSession.class, "learningSessions/" + id + "/", null,
                this.httpMessageTransformer);
    }

    /**
     * Retrieve a list of learning sessions associated with this account
     * 
     * @return {@link LearningSessionList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public LearningSessionList list() throws ClientException {
        return this.list(null);
    }

    /**
     * Retrieve a list of learning sessions associated with this account
     * 
     * @param params The {@link PagingParams} object for this list
     * 
     * @return {@link LearningSessionList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public LearningSessionList list(PagingParams params) throws ClientException {
        Map<String, Object> parameters = null;

        if (params != null) {
            parameters = params.toParameters();
        }

        LearningSessionList list = client.getConnection().get(LearningSessionList.class, "learningSessions/",
                parameters, httpMessageTransformer);
        for (LearningSession ls : list.getItems()) {
            ls.setClient(this.client);
        }
        return list;
    }

    /**
     * Delete a learning session
     * 
     * @param id The ID of the learning session to delete
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public void delete(String id) throws ClientException {
        Argument.IsNotNullOrEmpty(id, "id");
        client.getConnection().delete("learningSessions/" + id + "/", null, this.httpMessageTransformer);
    }

    /**
     * Create a learning session and wait for it to be able to predict
     * 
     * @param datasetId The ID of the dataset to learn on
     * @param target    The name of the feature from the dataset to learn how to
     *                  predict
     * 
     * @return The new {@link LearningSession}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public StatusTask<LearningSession> learn(String datasetId, String target) throws ClientException {
        Argument.IsNotNullOrEmpty(datasetId, "datasetId");
        Argument.IsNotNullOrEmpty(target, "target");
        return learnTask(datasetId, target);
    }

    /**
     * Set up a learning session without waiting for it to be ready. To block until
     * this process finishes, use {@link ILearningSessionClient#learn} instead or
     * call {@code .get()} on the return {@code Future<LearningSession>}.
     * 
     * @param datasetId The ID of the dataset to learn on
     * @param target    The name of the feature from the dataset to learn how to
     *                  predict
     * 
     * @return The new {@link Future}{@link LearningSession}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public Future<LearningSession> startLearn(String datasetId, String target) throws ClientException {
        Argument.IsNotNullOrEmpty(datasetId, "datasetId");
        Argument.IsNotNullOrEmpty(target, "target");
        final StatusTask<LearningSession> task = learnTask(datasetId, target);
        return pool.submit(new Callable<LearningSession>() {
            @Override
            public LearningSession call() throws ClientException, InterruptedException {
                return (LearningSession) task.getResult();
            }
        });
    }

    private StatusTask<LearningSession> learnTask(String datasetId, String target) throws ClientException {
        LearnParams params = new LearnParams(datasetId, target);
        LearningSessionLearnResponse learnResponse = client.getConnection().post(LearningSessionLearnResponse.class,
                "learningSessions/", null, params, this.httpMessageTransformer);

        return new StatusTask<>(this.client, learnResponse, this);

    }

    /**
     * Retrieve features association with a {@link LearningSession}. Values for
     * these features must be provided when making a prediction.
     * 
     * @param learningSessionId The ID of the learning session
     * 
     * @return {@link LearningSessionFeatures}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public LearningSessionFeatures getFeatures(String learningSessionId) throws ClientException {
        Argument.IsNotNullOrEmpty(learningSessionId, "learningSessionId");

        return client.getConnection().get(LearningSessionFeatures.class,
                "learningSessions/" + learningSessionId + "/features/", null, httpMessageTransformer);
    }

    /**
     * Get the deployment associated with this learning session. The resulting
     * deployment can be used to predict via
     * {@link IPredictionClient#deploymentPredict} although
     * {@link LearningSession#predict} can also be called directly on the
     * {@link LearningSession} object.
     * 
     * @param learningSessionId The ID of the learning session
     * 
     * @return {@link Deployment}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public Deployment getDeployment(String learningSessionId) throws ClientException {
        Argument.IsNotNullOrEmpty(learningSessionId, "learningSessionId");
        String url = "learningSessions/" + learningSessionId + "/deployment/";
        return client.getConnection().get(Deployment.class, url, null, this.httpMessageTransformer);
    }

    /**
     * Get performance information for the given learning session. For details on
     * how to interpret this information see the learning session documentation.
     * 
     * @param learningSessionId The ID of the learning session
     * 
     * @return {@link Evaluation}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public Evaluation getEvaluation(String learningSessionId) throws ClientException {
        Argument.IsNotNullOrEmpty(learningSessionId, "learningSessionId");
        LearningSession ls = get(learningSessionId);
        return ls.getEvaluation();
    }

    @Override
    public String toString() {
        return "LearningSessionClient";
    }
}
