package com.datarobot.impl;

import com.datarobot.IDataRobotAIClient;
import com.datarobot.IStatusClient;
import com.datarobot.model.PagingParams;
import com.datarobot.model.Status;
import com.datarobot.model.StatusList;
import com.datarobot.util.Action;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;

import java.util.Map;

/**
 * The {@link StatusClient} object provides access to the status endpoints of
 * the DataRobot AI API. This object is not meant to be used directly but
 * through the {@code DataRobotAIClient.status()} method.
 */
public class StatusClient implements IStatusClient {
    private IDataRobotAIClient client;
    private Action<HttpRequest, HttpResponse> httpMessageTransformer = null;

    /**
     * {@link Status IStatusClient} based API operations. Users will not need to
     * instantiate this object directly. It can be accessed through
     * {@code DataRobotAIClient.status()}.
     * 
     * @param client {@link DataRobotAIClient}
     */
    public StatusClient(IDataRobotAIClient client) {
        this.client = client;
    }

    /**
     * internal
     */
    @Override
    public Action<HttpRequest, HttpResponse> getHttpMessageTransformer() {
        return httpMessageTransformer;
    }

    /**
     * internal
     */
    @Override
    public void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer) {
        this.httpMessageTransformer = httpMessageTransformer;
    }

    /**
     * A list with status information on all {@link StatusTask} currently running
     * for this account
     * 
     * @return {@link StatusList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public StatusList list() throws ClientException {
        return this.list(null);
    }

    /**
     * A list with status information on all {@link StatusTask} currently running
     * for this account
     * 
     * @param params The {@link PagingParams} object for this list
     * 
     * @return {@link StatusList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public StatusList list(PagingParams params) throws ClientException {
        Map<String, Object> parameters = null;

        if (params != null) {
            parameters = params.toParameters();
        }
        return client.getConnection().get(StatusList.class, "status/", parameters, httpMessageTransformer);
    }

    /**
     * Get a status object by ID
     * 
     * @param statusId The ID of the {@link Status}
     * 
     * @return The queried {@link Status} object
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or
     *                         errors in parsing the response.
     */
    @Override
    public Status get(String statusId) throws ClientException {
        Argument.IsNotNullOrEmpty(statusId, "statusId");
        return client.getConnection().get(Status.class, "status/" + statusId + "/", null, httpMessageTransformer);
    }

    @Override
    public String toString() {
        return "StatusClient";
    }
}
