package com.datarobot.impl;

import com.datarobot.IDataRobotAIClient;
import com.datarobot.model.*;
import com.datarobot.util.Action;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A reference to a {@code Task} on the DataRobot server.
 * 
 * Client code that uses the DataRobot AI API package generally should not
 * construct these objects directly, they should be instantiated by AI API
 * Client methods.
 * 
 * This object may be out of sync with the DataRobot sever, for example, if
 * multiple processes or users have permission to modify or delete it on the
 * server.
 * 
 */
public class StatusTask<T> {
    private final int DEFAULT_TIMEOUT = 30 * 60; // 30 minutes
    private final int MAX_WAIT = 5 * 60; // 5 minutes
    private Pattern urlIdPattern = Pattern.compile("/([a-f\\d]{24})/$");

    private IDataRobotAIClient client;
    private IDoGetCallback<T> callback;
    private boolean complete = false;
    private boolean successful = false;
    private boolean didTimeOut = false;
    private T result = null;
    private String error;
    private String status;
    private String message;
    private ILocationResponse hasLocation;
    private final ExecutorService pool = Executors.newFixedThreadPool(10);
    private static final Logger logger = LogManager.getLogger(StatusTask.class);
    private Action<HttpRequest, HttpResponse> httpMessageTransformer = null;

    /**
     * Check if a task is successful
     * 
     * @return Boolean {@code True} when the operation was successful, {@code False}
     *         when the API indicated there was an error with the operation, or if
     *         the operation is not yet complete
     */
    public boolean isSuccessful() {
        return this.successful;
    }

    /**
     * Check if a task is successful
     * 
     * @return Boolean {@code True} when the operation was successful, {@code False}
     *         when the API indicated there was an error with the operation, or if
     *         the operation is not yet complete
     */
    public boolean getSuccess() {
        return this.successful;
    }

    /**
     * Returns the {@code error} of this {@link StatusTask}
     * 
     * @return the error that occurred
     */
    public String getError() {
        return this.error;
    }

    /**
     * Returns the current {@code status} of this {@link StatusTask}
     * 
     * @return the current task status
     */
    public String getStatus() {
        return this.status;
    }

    /**
     * Returns the current {@code message} of this {@link StatusTask}
     * 
     * @return the current message
     */
    public String getMessage() {
        return this.message;
    }

    /**
     * Indication if this {@link StatusTask} timed out before completion.
     * 
     * @return Boolean {@code True} if the operation took longer than the timeout
     *         given to {@code getResult} or if the default timeout was exceeded
     */
    public boolean timeOut() {
        return this.didTimeOut;
    }

    /**
     * internal
     */
    public Action<HttpRequest, HttpResponse> getHttpMessageTransformer() {
        return httpMessageTransformer;
    }

    /**
     * internal
     */
    public void setHttpMessageTransformer(Action<HttpRequest, HttpResponse> httpMessageTransformer) {
        this.httpMessageTransformer = httpMessageTransformer;
    }

    /**
     * internal
     */
    public StatusTask(IDataRobotAIClient client, ILocationResponse hasLocation, IDoGetCallback<T> callback) {
        this.client = client;
        this.hasLocation = hasLocation;
        this.callback = callback;

        if (hasLocation.getStatusCode() == 303) {
            this.complete = true;
            this.error = "";
            this.successful = true;
            this.status = "COMPLETE";
            this.message = "";
        } else if (hasLocation.getStatusCode() == 202 || hasLocation.getStatusCode() == 204) {
            this.complete = false;
            this.error = "";
            this.successful = true;
            this.status = "CREATED";
            this.message = "";
        }
    }

    /**
     * Start the process of returning the output of this {@link StatusTask} once the
     * operation completes. Use {@code getResult()} on the return of this object to
     * retrieve the result.
     * 
     * @param timeout the time in seconds to wait for this {@link StatusTask} to
     *                complete
     * 
     * 
     * @return the result of this {@link Future}{@link StatusTask}
     * 
     */
    public Future<T> getResultAsync(final int timeout) {
        return pool.submit(new Callable<T>() {
            @Override
            public T call() throws ClientException, InterruptedException {
                return getResult(timeout);
            }
        });
    }

    /**
     * Start the process of returning the output of this {@link StatusTask} once the
     * operation completes. Use {@code getResult()} on the return of this object to
     * retrieve the result.
     * 
     * @return the result of this {@link Future}{@link StatusTask}
     * 
     */
    public Future<T> getResultAsync() {
        return pool.submit(new Callable<T>() {
            @Override
            public T call() throws ClientException, InterruptedException {
                return getResult(DEFAULT_TIMEOUT);
            }
        });
    }

    /**
     * Wait until the operation completes, and return its output
     * 
     * @return the result of this {@link StatusTask}
     * 
     * @throws ClientException      when 4xx or 5xx response is received from
     *                              server, or errors in parsing the response.
     * @throws InterruptedException when a thread is waiting, sleeping, or otherwise
     *                              occupied, and the thread is interrupted, either
     *                              before or during the activity.
     */
    public T getResult() throws ClientException, InterruptedException {
        return this.getResult(DEFAULT_TIMEOUT);
    }

    /**
     * Wait until the operation completes, and return its output
     * 
     * @param timeout the time in seconds to wait for this {@link StatusTask} to
     *                complete
     * 
     * @return the result of this {@link StatusTask}
     * 
     * @throws ClientException      when 4xx or 5xx response is received from
     *                              server, or errors in parsing the response.
     * @throws InterruptedException when a thread is waiting, sleeping, or otherwise
     *                              occupied, and the thread is interrupted, either
     *                              before or during the activity.
     */
    public T getResult(int timeout) throws ClientException, InterruptedException {
        if ((timeout == 0) && !this.complete) {
            this.checkComplete();
        } else {
            int current_timeout = DEFAULT_TIMEOUT;

            if (timeout != -1) {
                current_timeout = timeout;
            }

            int elapsed = 0;
            double i = 0;

            // exponential backoff w/ jitter loop
            while (true) {
                if (elapsed >= current_timeout) {
                    didTimeOut = true;
                    throw new ClientException("Timeout occurred.");
                }

                int wait = new Random().nextInt(Math.min(MAX_WAIT, (int) Math.pow(1.5, i)));
                logger.debug("Waiting " + wait + " second(s)...");
                Thread.sleep(wait * 1000);

                if (this.checkComplete()) {
                    break;
                }

                i = i + 1;
                elapsed = elapsed + wait;
            }
        }
        if (this.result != null) {
            return this.result;
        }

        throw new ClientException("Task did not complete successfully.");
    }

    private boolean checkComplete() throws ClientException {
        if (this.complete) {
            return true;
        }
        // Check the status.
        Status statusObj = this.client.status().get(this.hasLocation.getStatusId());

        if (statusObj.getStatus().equals("COMPLETED")) {
            // less ugly hack - I guess stripUrlId has a purpose after all :P
            this.result = callback.get(stripUrlId((statusObj.getResult()).toString()));
            return true;
        } else {
            // Not done. Set Status and message and loop.
            this.status = statusObj.getStatus();
            this.message = statusObj.getMessage();

            if (new HashSet<>(Arrays.asList("ERROR", "ABORT")).contains(statusObj.getStatus())) {
                this.complete = true;
                this.successful = false;
                this.error = statusObj.getMessage();
                logger.error(this.error);
                return true;
            }
            logger.info(this.status + ": " + this.message);
            return false;
        }
    }

    private String stripUrlId(String location) throws ClientException {
        Matcher matchId = urlIdPattern.matcher(location);
        if (!matchId.find()) {
            throw new ClientException("Invalid ID in redirect header.");
        }
        return matchId.group(1);
    }

    @Override
    public String toString() {
        return "StatusTask [client=" + client + ", complete=" + complete + ", successful=" + successful
                + ", didTimeOut=" + didTimeOut + ", status=" + status + "]";
    }
}
