package com.datarobot.model;

import java.io.FileNotFoundException;
import java.io.Serializable;
import java.util.Map;
import org.joda.time.DateTime;

import com.datarobot.model.Output;
import com.datarobot.IDataRobotAIClient;
import com.datarobot.impl.ClientException;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * A container for multiple learning sessions, dataset, and other entities in
 * DataRobot AI.
 * 
 * An AI groups together different entities necessary to solve a business
 * problem, rather than being strictly focused on delivering a single model
 * without any additional context.
 * 
 * <p>
 * Note that this object is also a reference to a {@code AI} on the DataRobot
 * server.
 * 
 * Client code that uses the DataRobot AI API package generally should not
 * construct these objects directly, they should be instantiated by AI API
 * Client methods.
 * 
 * This object may be out of sync with the DataRobot sever, for example, if
 * multiple processes or users have permission to modify or delete it on the
 * server.
 * 
 */
public class AI implements Serializable, INeedClient {

	private final static long serialVersionUID = 4003768784013717195L;
	private String outputListUrl;
	private String datasetListUrl;
	private String learningSessionListUrl;

	@JsonIgnore
	private IDataRobotAIClient client;

	@JsonProperty("name")
	private String name;

	@JsonProperty("outputCount")
	private int outputCount;

	@JsonProperty("datasetCount")
	private int datasetCount;

	@JsonProperty("learningSessionCount")
	private int learningSessionCount;

	@JsonProperty("id")
	private String id;

	@JsonProperty("links")
	private void unpack(Map<String, String> links) throws ClientException {
		this.outputListUrl = links.get("outputsList");
		this.datasetListUrl = links.get("datasetsList");
		this.learningSessionListUrl = links.get("learningSessionsList");
	}

	@JsonProperty("createdOn")
	private DateTime createdOn;

	/**
	 * Get a URL for the list of {@link Output} objects associated with this
	 * {@link AI}
	 * 
	 * @return URL location of the output list associated with this AI
	 */
	public String getOutputListUrl() {
		return this.outputListUrl;
	}

	/**
	 * Get a URL for the list of {@link Dataset} objects associated with this
	 * {@link AI}
	 * 
	 * @return URL location of the dataset list associated with this AI
	 */
	public String getDatasetUrl() {
		return this.datasetListUrl;
	}

	/**
	 * Get a URL for the list of {@link LearningSession} objects associated with
	 * this {@link AI}
	 * 
	 * @return URL location of the learning session list associated with this AI
	 */
	public String getLearningSessionUrlList() {
		return this.learningSessionListUrl;
	}

	/**
	 * Get the name of this {@link AI}
	 * 
	 * @return The name of this AI
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Get the number of {@link Output} objects associated with this {@link AI}
	 * 
	 * @return The number of AI outputs
	 */
	public int getOutputCount() {
		return this.outputCount;
	}

	/**
	 * Get the number of {@link Dataset} objects associated with this {@link AI}
	 * 
	 * @return The number of AI datasets
	 */
	public int getDatasetCount() {
		return this.datasetCount;
	}

	/**
	 * Get the number of {@link LearningSession} objects associated with this
	 * {@link AI}
	 * 
	 * @return The number of AI learning sessions
	 */
	public int getLearningSessionCount() {
		return this.learningSessionCount;
	}

	/**
	 * Get the unique identifier for this {@link AI}
	 * 
	 * @return The ID of this AI
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * Get the date this {@link AI} was created
	 * 
	 * @return The creation date of this AI
	 */
	public DateTime getCreatedDate() {
		return this.createdOn;
	}

	/**
	 * internal
	 */
	public void setOutputListUrl(String outputListUrl) {
		this.outputListUrl = outputListUrl;
	}

	/**
	 * internal
	 */
	public void setDatasetListUrl(String datasetListUrl) {
		this.datasetListUrl = datasetListUrl;
	}

	/**
	 * internal
	 */
	public void setLearningSessionListUrl(String learningSessionListUrl) {
		this.learningSessionListUrl = learningSessionListUrl;
	}

	/**
	 * internal
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * internal
	 */
	public void setOutputCount(int outputCount) {
		this.outputCount = outputCount;
	}

	/**
	 * internal
	 */
	public void setDatasetCount(int datasetCount) {
		this.datasetCount = datasetCount;
	}

	/**
	 * internal
	 */
	public void setLearningSessionCount(int learningSessionCount) {
		this.learningSessionCount = learningSessionCount;
	}

	/**
	 * internal
	 */
	public void setId(String id) {
		this.id = id;
	}

	// Need this method to work with INeedClient interface
	/**
	 * internal
	 */
	@Override
	public void setClient(IDataRobotAIClient client) {
		this.client = client;

	}

	/**
	 * Import a file to an AI. See the Dataset page in the documentation for size
	 * limitations.
	 * 
	 * @param filepath The path of a local file to upload
	 * 
	 * @return {@link Dataset}
	 * 
	 * @throws ClientException       when 4xx or 5xx response is received from
	 *                               server, or errors in parsing the response.
	 * @throws FileNotFoundException when a file with the specified pathname does
	 *                               not exist, or if the file does exist but is
	 *                               inaccessible for some reason.
	 * @throws InterruptedException  when a thread is waiting, sleeping, or
	 *                               otherwise occupied, and the thread is
	 *                               interrupted, either before or during the
	 *                               activity.
	 */
	public Dataset importFile(String filepath) throws FileNotFoundException, ClientException, InterruptedException {
		Dataset data = client.data().importFile(filepath);
		addDataset(data);
		return data;
	}

	/**
	 * Import a file to an AI via a public URL. See the Dataset page in the
	 * documentation for size limitations.
	 * 
	 * @param url The url to a publically available file
	 * 
	 * @return {@link Dataset}
	 * 
	 * @throws ClientException      when 4xx or 5xx response is received from
	 *                              server, or errors in parsing the response.
	 * @throws InterruptedException when a thread is waiting, sleeping, or otherwise
	 *                              occupied, and the thread is interrupted, either
	 *                              before or during the activity.
	 */
	public Dataset importUrl(String url) throws ClientException, InterruptedException {
		Dataset data = client.data().importUrl(url);
		addDataset(data);
		return data;
	}

	/**
	 * Add a {@link Dataset} to this {@link AI}
	 * 
	 * @param dataset The dataset which to add
	 * 
	 * @throws ClientException when 4xx or 5xx response is received from server, or
	 *                         errors in parsing the response.
	 */
	public void addDataset(Dataset dataset) throws ClientException {
		client.ais().addDataset(id, dataset.getId());
		sync();
	}

	/**
	 * Add a {@link LearningSession} to this {@link LearningSession}
	 * 
	 * @param learningSession The learning session which to add
	 * 
	 * @throws ClientException when 4xx or 5xx response is received from server, or
	 *                         errors in parsing the response.
	 */
	public void addLearningSession(LearningSession learningSession) throws ClientException {
		// TODO learningSession.getTarget should be getting the output name instead
		client.ais().addLearningSession(id, learningSession.getId(), learningSession.getTarget());
		sync();
	}

	// TODO learn via dataset Id here as well (Pattern. class)
	/**
	 * Create a {@link LearningSession} directly associated to this {@link AI}
	 * 
	 * @param target     The name of the selected target feature to learn
	 * @param sourcefile The data on which to learn via a path to a local file
	 * 
	 * @return The {@link AI} this learning session was created with
	 * 
	 * @throws ClientException       when 4xx or 5xx response is received from
	 *                               server, or errors in parsing the response.
	 * @throws InterruptedException  when a thread is waiting, sleeping, or
	 *                               otherwise occupied, and the thread is
	 *                               interrupted, either before or during the
	 *                               activity.
	 * @throws FileNotFoundException when a file with the specified pathname does
	 *                               not exist, or if the file does exist but is
	 *                               inaccessible for some reason.
	 */
	public AI learn(String target, String sourcefile)
			throws ClientException, InterruptedException, FileNotFoundException {
		// TODO - once datasetId is incorporated this needs to not default to importing
		// a new dataset everytime
		Dataset trainingData = importFile(sourcefile);
		LearningSession ls = client.learning().learn(trainingData.getId(), target).getResult();
		addLearningSession(ls);
		return this;
	}

	/**
	 * Retrieve AI predictions
	 * 
	 * @param target     The name of the selected target feature to predict
	 * @param sourceFile The data on which to make predictions via a path to a local
	 *                   file
	 * 
	 * @return {@link PredictionList}
	 * 
	 * @throws FileNotFoundException when a file with the specified pathname does
	 *                               not exist, or if the file does exist but is
	 *                               inaccessible for some reason.
	 * @throws ClientException       when 4xx or 5xx response is received from
	 *                               server, or errors in parsing the response.
	 */
	public PredictionList predict(String target, String sourceFile) throws FileNotFoundException, ClientException {
		return client.predictions().aiPredict(id, target, sourceFile);
	}

	/**
	 * Retrieve ai predictions
	 * 
	 * @param target     The name of the selected target feature to predict
	 * @param sourceFile The data on which to make predictions via a path to a local
	 *                   file
	 * 
	 * @return {@link PredictionList}
	 * 
	 * @throws FileNotFoundException when a file with the specified pathname does
	 *                               not exist, or if the file does exist but is
	 *                               inaccessible for some reason.
	 * @throws ClientException       when 4xx or 5xx response is received from
	 *                               server, or errors in parsing the response.
	 */
	public PredictionList infer(String target, String sourceFile) throws FileNotFoundException, ClientException {
		return client.predictions().aiPredict(id, target, sourceFile);
	}

	/**
	 * Retrieve an {@link Output} for this {@link AI}
	 * 
	 * @param outputTarget The target of the output
	 * 
	 * @return {@link Output}
	 * 
	 * @throws ClientException when 4xx or 5xx response is received from server, or
	 *                         errors in parsing the response.
	 */
	public Output getOutput(String outputTarget) throws ClientException {
		return client.ais().getOutput(this.id, outputTarget);
	}

	/**
	 * Retrieve a list of {@link Output} associated with this {@link AI}
	 * 
	 * @return {@link OutputList}
	 * 
	 * @throws ClientException when 4xx or 5xx response is received from server, or
	 *                         errors in parsing the response.
	 */
	public OutputList getOutputs() throws ClientException {
		return client.ais().getOutputs(this.id);
	}

	/**
	 * Add a {@link Output} to this {@link LearningSession}
	 * 
	 * @param learningSessionId The ID of the learning session this output is
	 *                          connected to
	 * @param outputName        The name of the output to add
	 * 
	 * @throws ClientException when 4xx or 5xx response is received from server, or
	 *                         errors in parsing the response.
	 */
	public void addOutput(String learningSessionId, String outputName) throws ClientException {
		client.ais().addOutput(this.id, learningSessionId, outputName);
		sync();
	}

	/**
	 * Retrieve a list of {@link LearningSession} objects associated with this
	 * {@link AI}
	 * 
	 * @return {@link LearningSessionList}
	 * 
	 * @throws ClientException when 4xx or 5xx response is received from server, or
	 *                         errors in parsing the response.
	 */
	public LearningSessionList getLearningSessions() throws ClientException {
		return client.ais().getLearningSessions(this.id);
	}

	/**
	 * Retrieve a list of {@link Dataset} objects associated with this {@link AI}
	 * 
	 * @return {@link DatasetList}
	 * 
	 * @throws ClientException when 4xx or 5xx response is received from server, or
	 *                         errors in parsing the response.
	 */
	public DatasetList getDatasets() throws ClientException {
		return client.data().list(this.id);
	}

	public void sync() throws ClientException {
		AI serverData = client.ais().get(id);
		this.outputCount = serverData.outputCount;
		this.datasetCount = serverData.datasetCount;
		this.learningSessionCount = serverData.learningSessionCount;
		this.name = serverData.name;
	}

	@Override
	public String toString() {
		return "AI [outputListUrl=" + outputListUrl + ", datasetListUrl=" + datasetListUrl + ", learningSessionListUrl="
				+ learningSessionListUrl + ", client=" + client + ", name=" + name + ", outputCount=" + outputCount
				+ ", datasetCount=" + datasetCount + ", learningSessionCount=" + learningSessionCount + ", id=" + id
				+ "]";
	}

}
