package com.datarobot.model;

import com.datarobot.IDataRobotAIClient;
import com.fasterxml.jackson.annotation.*;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.joda.time.DateTime;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * A reference to a {@code Dataset} on the DataRobot server.
 * 
 * Client code that uses the DataRobot AI API package generally should not construct these objects directly, they should be instantiated by AI API Client methods.
 * 
 * This object may be out of sync with the DataRobot sever, for example, if multiple processes or users have permission to modify or delete it on the server.
 * 
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({
        "id",
        "name",
        "created",
        "size",
        "rowCount",
        "columnCount",
})
public class Dataset implements Serializable, INeedClient {

	@JsonIgnore
    private IDataRobotAIClient client;

    @JsonProperty("id")
    private String id;
    @JsonProperty("datasetName")
    private String name;
    @JsonProperty("createdOn")
    private DateTime created;
    @JsonProperty("size")
    private int size;
    @JsonProperty("rowCount")
    private int rowCount;
    @JsonProperty("columnCount")
    private int columnCount;
    @JsonIgnore
    private Map<String, Object> additionalProperties = new HashMap<>();
    private final static long serialVersionUID = 7159522227188063616L;

    /**
     * internal
     */
    public void setClient(IDataRobotAIClient client) {
        this.client = client;
    }

    /**
     * Retrieve the unique identifier for this {@link Dataset}
     * 
     * @return The ID for this dataset
     */
    public String getId() {
        return id;
    }

    /**
     * Retrieve the name of this {@link Dataset}
     * 
     * @return The name of this dataset
     */
    public String getName() {
        return name;
    }


    /**
     * Retrieve the datetime this {@link Dataset} was created
     *
     * @return The creation datetime of this dataset
     */
    public DateTime getCreatedOn() {
        return created;
    }

    /**
     * Retrieve the size of this {@link Dataset}
     *
     * @return The size of the {@link Dataset}
     */
    public int getSize() {
        return size;
    }

    /**
     * Retrieve the number of rows of this {@link Dataset}
     *
     * @return The number of rows of the {@link Dataset}
     */
    public int getRowCount() {
        return rowCount;
    }


    /**
     * Retrieve the number of columns of this {@link Dataset}
     * 
     * @return The number of columns of the {@link Dataset}
     */
    public int getColumnCount() {
        return columnCount;
    }

    /**
     * internal
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
        return this.additionalProperties;
    }

    /**
     * internal
     */
    @JsonAnySetter
    public void setAdditionalProperty(String name, Object value) {
        this.additionalProperties.put(name, value);
    }

    /**
     * internal
     */
    @Override
    public int hashCode() {
        return new HashCodeBuilder()
                .append(name)
                .append(created)
                .append(id)
                .append(size)
                .append(rowCount)
                .append(columnCount)
                .append(additionalProperties)
                .toHashCode();
    }

    /**
     * internal
     */
    @Override
    public boolean equals(Object other) {
        if (other == this) {
            return true;
        }
        if (!(other instanceof Dataset)) {
            return false;
        }
        Dataset rhs = ((Dataset) other);
        return new EqualsBuilder().append(name, rhs.name)
                .append(created, rhs.created)
                .append(id, rhs.id)
                .append(size, rhs.size)
                .append(rowCount, rhs.rowCount)
                .append(columnCount, rhs.columnCount)
                .append(additionalProperties, rhs.additionalProperties)
                .isEquals();
    }
    
    @Override
	public String toString() {
		return "Dataset [client=" + client + ", id=" + id + ", name=" + name + ", created=" + created + "]";
	}
}
