package com.datarobot.model;

import com.datarobot.IDataRobotAIClient;
import com.datarobot.impl.ClientException;
import com.datarobot.impl.DataRobotAIClient;
import com.datarobot.model.Evaluation;
import com.fasterxml.jackson.annotation.*;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.joda.time.DateTime;

import java.io.FileNotFoundException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A reference to a {@code LearningSession} on the DataRobot server.
 * 
 * Client code that uses the DataRobot AI API package generally should not construct these objects directly, they should be instantiated by AI API Client methods.
 * 
 * This object may be out of sync with the DataRobot sever, for example, if multiple processes or users have permission to modify or delete it on the server.
 * 
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({
        "id",
        "datasetId",
        "name",
        "target",
        "created",
        "evaluation"
})
public class LearningSession implements Serializable, INeedClient {
    @JsonIgnore
    IDataRobotAIClient client;
    @JsonProperty("id")
    private String id;
    @JsonProperty("datasetId")
    private String datasetId;
    @JsonProperty("name")
    private String name;
    @JsonProperty("target")
    private String target;
    @JsonProperty("created")
    private DateTime created;
    @JsonProperty("modelCount")
    private int modelCount;
    @JsonProperty("evaluation")
    private Evaluation evaluation;
    @JsonProperty("links")
    private void unpack(Map<String, String> links) throws ClientException {
    	if (links.containsKey("result")){
    		this.result = (String)links.get("result");
    	}
    }
    private String result;
    @JsonIgnore
    private Map<String, Object> additionalProperties = new HashMap<>();
    private final static long serialVersionUID = -7584943958576758458L;
    @JsonIgnore
    private Deployment deployment;

    /**
     * internal
     */
    public void setClient(IDataRobotAIClient client) {
        this.client = client;
    }

    /**
     * Get the unique identifier of this {@link LearningSession}
     *
     * @return The ID of this learning session
     */
    public String getId() {
        return this.id;
    }
    
    /**
     * Get the unique identifier of the {@link Dataset} this {@link LearningSession} learned from 
     * 
     * @return The ID of the dataset connected to this learning session
     */
    public String getDatasetId() {
    	return this.datasetId;
    }

    /**
     * Get the name of this {@link LearningSession}
     *
     * @return The name of this learning session
     */
    public String getName() {
        return name;
    }

    /**
     * Get the target of this {@link LearningSession}
     *
     * @return The target of this learning session
     */
    public String getTarget() {
        return target;
    }

    /**
     * Retrieve the datetime this {@link LearningSession} was created
     * 
     * @return The creation datetime of this learning session
     */
    public DateTime getCreatedOn() {
        return created;
    }

    /**
     * Retrieve number of models trained as part of this learning session.
     * Note that these are not unique algorithms attempted, but encompass the combination of 
     * different algorithms trained against varying sample sizes from the training dataset.
     * 
     * @return The number of models trained as part of this learning session
     */
    public int getModelCount() {
        return modelCount;
    }
    
    /**
     * Retrieve performance information for the given learning session. 
     * For details on how to interpret this information, see the learning session documentation. 
     * 
     * @return The {@link Evaluation} of this learning session
     * @throws ClientException 
     */
    public Evaluation getEvaluation() throws ClientException {
    	return evaluation;
    }
    
    /**
     * Refresh this object with the server.
     * @throws ClientException 
     */
    public void refresh() throws ClientException {
    	LearningSession ls = this.client.learning().get(this.id);
    	this.evaluation = ls.getEvaluation();
    	this.modelCount = ls.getModelCount();
    }
    
    /**
     * internal 
     */
    public void setDatasetId(String datasetId) {
    	this.datasetId = datasetId;
    }
    
    /**
     * internal
     */
    public void setDeployment(Deployment deployment) {
    	this.deployment = deployment;
    }

    /**
     * internal
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
        return this.additionalProperties;
    }

    /**
     * internal
     */
    @JsonAnySetter
    public void setAdditionalProperty(String name, Object value) {
        this.additionalProperties.put(name, value);
    }

    /**
     * Make predictions on new data using this learning session
     * 
     * @param sourceFile The data on which to predict via a filepath on the local system
     * 
     * @return {@link PredictionList}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or errors in parsing the response.
     * @throws FileNotFoundException when a file with the specified pathname does not exist, or if the file does exist but is inaccessible for some reason.
     */
    public PredictionList predict(String sourceFile) throws ClientException, FileNotFoundException {
        if (this.client == null) {
            throw new ClientException("The instance of LearningSession must include a reference to the DataRobotAIClient in "
                    + "order to make fit a model. Please include the reference when constructing the object.");
        }
        Deployment deployment = client.learning().getDeployment(id);
        return client.predictions().deploymentPredict(deployment, sourceFile);
    }
    
    /**
     * Retrieve features associated with this learning session. Values for these features must be 
     * provided when making a {@code Prediction}
     * 
     * @return {@link LearningSessionFeatures}
     * 
     * @throws ClientException when 4xx or 5xx response is received from server, or errors in parsing the response.
     */
    public LearningSessionFeatures getFeatures() throws ClientException {
    	if (this.client == null) {
            throw new ClientException("The instance of LearningSession must include a reference to the DataRobotAIClient in "
                    + "order to retrieve features.  Please include the reference when constructing the object.");
        }
    	return client.learning().getFeatures(this.id);
    }

    /**
     * internal
     */
    @Override
    public int hashCode() {
        return new HashCodeBuilder()
                .append(id)
                .append(datasetId)
                .append(name)
                .append(target)
                .append(created)
                .append(additionalProperties)
                .toHashCode();
    }

    /**
     * internal
     */
    @Override
    public boolean equals(Object other) {
        if (other == this) {
            return true;
        }
        if ((other instanceof LearningSession) == false) {
            return false;
        }
        LearningSession rhs = ((LearningSession) other);
        return new EqualsBuilder()
                .append(id, rhs.id)
                .append(datasetId, rhs.datasetId)
                .append(name, rhs.name)
                .append(target, rhs.target)
                .append(created, rhs.created)
                .append(additionalProperties, rhs.additionalProperties)
                .isEquals();
    }

	@Override
	public String toString() {
		return "LearningSession [client=" + client + ", id=" + id + ", datasetId=" + datasetId + ", name=" + name
				+ ", target=" + target + ", created=" + created + ", modelCount=" + modelCount + ", evaluation="
				+ evaluation + "]";
	}
    
}


