package com.datarobot.model;

import java.io.Serializable;
import java.util.Map;

import org.joda.time.DateTime;

import com.datarobot.IDataRobotAIClient;
import com.datarobot.impl.ClientException;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * A reference to an {@code Output} on the DataRobot server.
 * 
 * Client code that uses the DataRobot AI API package generally should not construct these objects directly, they should be instantiated by AI API Client methods.
 * 
 * This object may be out of sync with the DataRobot sever, for example, if multiple processes or users have permission to modify or delete it on the server.
 * 
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class Output implements Serializable, INeedClient {

	private static final long serialVersionUID = -5533037101250361907L;
	private String datarobotKey;
	private String deploymentId;
	private String url;
	private String modelType;
	private Map<String, String> source;
	private String aiId;
	private Evaluation evaluation;
	
	@JsonIgnore
	private IDataRobotAIClient client;
	
	@JsonProperty("name")
	private String name;
	
	@JsonProperty("source")
	private void unpack_source(Map<String, String> source) throws ClientException {
		this.source = source;
		this.datarobotKey = (String) source.get("datarobot-key");
		this.deploymentId = (String) source.get("deploymentId");
		this.url = (String) source.get("url");
		this.modelType = (String) source.get("modelType");
    }
	
	@JsonProperty("evaluation")
	private void unpack_eval(Map<String, Object> eval) throws ClientException {
		this.evaluation = new Evaluation(
				((String) eval.get("summary")),
				((Double) eval.get("score")),
				((String) eval.get("label")));
	}
	
	@JsonProperty("target")
	private String target;
	
	@JsonProperty("createdOn")
	private DateTime createdOn;
	
	/**
	 * Get the datarobot-key required for predictions associated with this {@link Output}
	 * 
	 * @return The datarobot-key of this output
	 */
	public String getDataRobotKey() {
		return this.datarobotKey;
	}
	
	/**
	 * Get the unique identifier for the {@link Deployment} of this {@link Output}
	 * 
	 * @return The ID for the deployment associated with this output
	 */
	public String getDeploymentId() {
		return this.deploymentId;
	}
	
	/**
	 * Get the URL for the {@link Deployment} of this {@link Output}
	 * 
	 * @return The URL for the deployment associated with this output
	 */
	public String getUrl() {
		return this.url;
	}
	
	/**
	 * Get the source for this {@link Output}
	 * 
	 * @return The source of this output
	 */
	public Map<String, String> getSource(){
		return this.source;
	}
	
	/**
	 * Get the name of this {@link Output}
	 * 
	 * @return The name of this output
	 */
	public String getName() {
		return this.name;
	}
	
	/**
	 * Get the target for this {@link Output}
	 * 
	 * @return The target of this output
	 */
	public String getTarget() {
		return this.target;
	}	

	/**
	 * Get the model type for this {@link Output}
	 * 
	 * @return The model type for this output
	 */
	public String getModelType() {
		return this.modelType;
	}
	
	/**
	 * Get an {@link Evaluation} for this {@link Output}
	 * 
	 * @return An evaluation of the learning session for this output.
	 */
	public Evaluation getEvaluation() {
		return this.evaluation;
	}
	
	/**
	 * Get the date this {@link Output} was created
	 * 
	 * @return The creation date of this Output
	 */
	public DateTime getCreatedDate() {
		return this.createdOn;
	}
	
	
	/**
	 * Get a list of features for this {@link Output}
	 * 
	 * @return {@link OutputFeatures}
	 * @throws ClientException 
	 */
	public OutputFeatures getFeatures() throws ClientException {
		return this.client.ais().getOutputFeatures(this.aiId, this.target);
	}
	
	public void setAiId(String id) {
		this.aiId = id;
	}
	
	// carryover from INeedClient
	/**
	 * internal
	 */
	@Override
	public void setClient(IDataRobotAIClient client) {
		this.client = client;
		
	}
	
	@Override
	public String toString() {
		return "Output [datarobotKey=" + datarobotKey + ", deploymentId=" + deploymentId + ", url=" + url
				+ ", modelType=" + modelType + ", aiId=" + aiId + ", client=" + client + ", name=" + name + ", target="
				+ target + "]";
	}

}
