package com.datarobot.model;

import java.util.HashMap;
import java.util.Map;

/**
 * This class represents {@code Paging Parameters} that can be used to limit results from the return {@code List} object that could potentially
 * contain a large number of results. The results by default will use a {@code PagingParams.limit} of 50
 * items starting at the 0th item, using an {@code offset} of 0. To retrieve the next group, set the {@code offset}
 * to 50 to skip ahead 50 items.
 *
 **/
public class PagingParams {

	private Integer offset;
    private Integer limit;
    
    /**
     * A PagingParams object that represents the default settings ({@code offset} at 0, and {@code limit} at 50)
     */
    public static PagingParams Default = new PagingParams(0, 50);

    /**
     * Constructs
     * 
     * @param offset The number of result to skip. Defaults to 0. 
     * @param limit The maximum number of results to retrieve. Defaults to 50.
     */
    public PagingParams(Integer offset, Integer limit)
    {
        this.offset = offset;
        this.limit = limit;
    }

    /**
     * Get the offset for this {@link PagingParams} object
     *
     * @return The number of items to skip
     */
    public int getOffset() {
        return offset;
    }

    /**
     * Set the offset for this {@link PagingParams} object
     *
     * @param offset The number of items to skip
     */
    public void setOffset(int offset) {
        this.offset = offset;
    }


    /**
     * Get the limit for this {@link PagingParams} object
     * 
     * @return The number of items to retrieve 
     */
    public int getLimit() {
        return limit;
    }

    /**
     * Set the limit for this {@link PagingParams} object
     *
     * @param limit The number of items to retrieve 
     */
    public void setLimit(int limit) {
        this.limit = limit;
    }

    /**
     * internal
     * <p>
     * Converts this object to a {@code Map} used to build the query string parameters for a request
     *
     * @return A Map containing the parameters to be included as querystrings on the request.
     */
    public Map<String,Object> toParameters() {
        Map<String,Object> parameters = new HashMap<>();

        if (this.offset != null) {
            parameters.put("offset", this.getOffset());
        }

        if (this.limit != null) {
            parameters.put("limit", this.getLimit());
        } else {
            parameters.put("limit", this.Default.getLimit());
        }

        return parameters;
    }
    
    @Override
	public String toString() {
		return "PagingParams [offset=" + offset + ", limit=" + limit + "]";
	}

}