package com.datarobot.model;

import com.datarobot.impl.ClientException;
import com.fasterxml.jackson.annotation.*;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.joda.time.DateTime;

import java.io.Serializable;
import java.net.URI;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A reference to the {@code Status} object for a task on the DataRobot server.
 * 
 * Client code that uses the DataRobot AI API package generally should not construct these objects directly, they should be instantiated by AI API Client methods.
 * 
 * This object may be out of sync with the DataRobot sever, for example, if multiple processes or users have permission to modify or delete it on the server.
 * 
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({
        "statusId",
        "status",
        "code",
        "description",
        "created",
        "message",
        "statusType",
        "links"
})
public class Status extends ReturnsLocation implements Serializable, ILocationResponse  {

	@JsonIgnore
    private Pattern statusIdPattern = Pattern.compile("status/([a-f0-9-]{36})/$");
    @JsonIgnore
    private final static long serialVersionUID = 7155748365180452743L;

    @JsonProperty("statusId")
    private String statusId;
    @JsonProperty("status")
    private String status;
    @JsonProperty("code")
    private String code;
    @JsonProperty("description")
    private String description;
    @JsonProperty("created")
    private DateTime created;
    @JsonProperty("message")
    private String message;
    @JsonProperty("statusType")
    private String statusType;
    @JsonProperty("links")
    private void unpack(Map<String, String> links) throws ClientException {
    	if (links.containsKey("result")){
    		this.result = (String)links.get("result");
    	}
    }	
    private String result;
    @JsonIgnore
    private int statusCode;
    
    @JsonIgnore
    private Map<String, Object> additionalProperties = new HashMap<>();

    /**
     * Get the status code of this {@link Status} object
     * 
     * @return the status code
     */
    public int getStatusCode() {
        return statusCode;
    }

    // Repeat method of .getStatusId() further down - keeping for the time being as much of the code base is still using this
    // will remove completely during Status class refactoring
    /**
     * Get the ID of this {@link Status} object
     * 
     * @return the status ID
     */
    public String getId() {
        return this.getStatusId();
    }
    
    // Need this pointless getter to make this thing an official bean
    /**
     * Get the location of the result of this {@link Status} object
     * 
     * @return the URI of the result
     */
    public URI getResult() {
    	return URI.create(this.result);
    }
    
    /**
     * The ID of this status object.
     *
     * @return An ID used to refer to this status object.
     */
    @Override
    public String getStatusId() {
        return statusId;
    }

    /**
     * Get the current status of the process
     *
     * @return The current status indicating what step a process is currently in
     */
    public String getStatus() {
        return status;
    }

    /**
     * Get the current status code
     *
     * @return the status code
     */
    public String getCode() {
        return code;
    }

    /**
     * Get a description describing details of the current status
     *
     * @return a description of the current status
     */
    public String getDescription() {
        return description;
    }

    /**
     * Get the DateTime of when the status was created
     *
     * @return DateTime of when the status was created
     */
    public DateTime getCreatedOn() {
        return created;
    }

    /**
     * Get the message for the current status
     *
     * @return the message for the current status
     */
    public String getMessage() {
        return message;
    }

    /**
     * Get the status type of the current status
     * 
     * @return the status type
     */
    public String getStatusType() {
        return statusType;
    }
    
 // Once Automodels + Datasets are refactored out of ILocationResponse this method will go away
    /**
     * internal
     */
    @Override
    public URI getLocation() {
    	return this.getResult();
    }
    
 // Once Automodels + Datasets are refactored out of ILocationResponse this method will go away
    /**
     * internal
     */
    @Override
    public void setStatusId() {
    }
    
    /**
     * internal
     */
    public void setResult(String result) {
    	result = this.result;
    }
    
    /**
     * internal
     */
    @Override
    public String getObjectId() {
    	String msg = "This will never be called";
    	return msg;
    }
    
    /**
     * internal
     */
    @Override 
    public void setObjectId(String objectId) {
    	
    }
    
    /**
     * internal
     */
    public void setStatus(String status) {
        this.status = status;
    }
    
    /**
     * internal
     */
    public void setMessage(String message) {
        this.message = message;
    }
    
    /**
     * internal
     */
    public void setStatusCode(int statusCode) {
        this.statusCode = statusCode;
    }
    
    /**
     * internal
     */
    public void setStatusId(String statusId) {
        this.statusId = statusId;
    }

    /**
     * internal
     */
    @JsonAnySetter
    public void setAdditionalProperty(String name, Object value) {
        this.additionalProperties.put(name, value);
    }
    
    /**
     * internal
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
        return this.additionalProperties;
    }

    /**
     * internal
     */
    @Override
    public int hashCode() {
        return new HashCodeBuilder()
                .append(statusId)
                .append(status)
                .append(code)
                .append(description)
                .append(created)
                .append(message)
                .append(statusType)
                .append(additionalProperties).toHashCode();
    }

    /**
     * internal
     */
    @Override
    public boolean equals(Object other) {
        if (other == this) {
            return true;
        }
        if ((other instanceof Status) == false) {
            return false;
        }
        Status rhs = ((Status) other);
        return new EqualsBuilder()
                .append(statusId, rhs.statusId)
                .append(status, rhs.status)
                .append(code, rhs.code)
                .append(description, rhs.description)
                .append(created, rhs.created)
                .append(message, rhs.message)
                .append(statusType, rhs.statusType)
                .append(additionalProperties, rhs.additionalProperties)
                .isEquals();
    }

    @Override
    void parseIdFromUrl(String location) throws ClientException {
        this.statusId = stripStatusId(location);
    }

    private String stripStatusId(String location) throws ClientException {
        Matcher statusId = statusIdPattern.matcher(location);
        if (statusId.find()) {
            return statusId.group(1);
        }
        return null;
    }
    
    @Override
	public String toString() {
		return "Status [statusId=" + statusId + ", status=" + status + ", code=" + code + ", description=" + description
				+ ", created=" + created + ", message=" + message + ", statusType=" + statusType + ", result=" + result
				+ ", statusCode=" + statusCode + "]";
	}
}
