package com.datasonnet.jsonnet

/*-
 * Copyright 2019-2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import fastparse.IndexedParserInput

/**
  * An exception that can keep track of the Sjsonnet call-stack while it is
  * propagating upwards. This helps provide good error messages with line
  * numbers pointing towards user code.
  */
case class Error(msg: String,
                 stack: List[StackTraceElement],
                 underlying: Option[Throwable])
  extends Exception(msg, underlying.orNull){
  setStackTrace(stack.toArray.reverse)
  def addFrame(fileName: Path, wd: Path, offset: Int)(implicit ev: EvalErrorScope) = {
    val newFrame = ev.loadCachedSource(fileName) match{
      case None =>
        new StackTraceElement(
          "", "",
          fileName.relativeToString(wd) + " offset:",
          offset
        )
      case Some(resolved) =>
        val Array(line, col) =
          new IndexedParserInput(resolved).prettyIndex(offset).split(':')

        new StackTraceElement(
          "", "",
          fileName.relativeToString(wd) + ":" + line,
          col.toInt
        )
    }

    this.copy(stack = newFrame :: this.stack)
  }
}


object Error {
  def tryCatch[T](offset: Int)
                 (implicit fileScope: FileScope, evaluator: EvalErrorScope): PartialFunction[Throwable, Nothing] = {
    case e: Error => throw e
    case e: Error.Delegate =>
      throw new Error(e.msg, Nil, None)
        .addFrame(fileScope.currentFile, evaluator.wd, offset)
    case e: Throwable =>
      throw new Error("Internal Error", Nil, Some(e))
        .addFrame(fileScope.currentFile, evaluator.wd, offset)
  }
  def tryCatchWrap[T](offset: Int)
                     (implicit fileScope: FileScope, evaluator: EvalErrorScope): PartialFunction[Throwable, Nothing] = {
    case e: Error => throw e.addFrame(fileScope.currentFile, evaluator.wd, offset)
    case e: Error.Delegate =>
      throw new Error(e.msg, Nil, None)
        .addFrame(fileScope.currentFile, evaluator.wd, offset)
    case e: Throwable =>
      throw new Error("Internal Error", Nil, Some(e))
        .addFrame(fileScope.currentFile, evaluator.wd, offset)
  }
  def fail(msg: String, offset: Int)
          (implicit fileScope: FileScope, evaluator: EvalErrorScope) = {
    throw Error(msg, Nil, None).addFrame(fileScope.currentFile, evaluator.wd, offset)
  }

  def failIfNonEmpty(names: collection.BitSet,
                     outerOffset: Int,
                     formatMsg: (String, String) => String)
                    (implicit fileScope: FileScope, eval: EvalErrorScope) = if (names.nonEmpty){
    val plural = if (names.size > 1) "s" else ""
    val nameSnippet = names.map(fileScope.indexNames).mkString(", ")
    fail(formatMsg(plural, nameSnippet), outerOffset)
  }


  /**
    * An exception containing a message, which is expected to get caught by
    * the nearest enclosing try-catch and converted into an [[Error]]
    */
  case class Delegate(msg: String) extends Exception(msg)

}

/**
  * FileScope models the per-file context that is propagated throughout the
  * evaluation of a single Jsonnet file. Contains the current file path, as
  * well as the mapping of local variable names to local variable array indices
  * which is shared throughout each file.
  */
class FileScope(val currentFile: Path,
                val nameIndices: Map[String, Int],
                val sourceP: String = ""){
  // Only used for error messages, so in the common case
  // where nothing blows up this does not need to be allocated
  // ( And also for debugger support )
  lazy val indexNames = nameIndices.map(_.swap)
  lazy val source = sourceP

  def getNameByIndex(idx: Int): Option[String] = {
    val found = nameIndices.find(_._2 == idx)
    if (found.nonEmpty) {
      Some(found.get._1)
    } else {
      None
    }
  }
}

trait EvalErrorScope {
  def extVars: Map[String, ujson.Value]
  def loadCachedSource(p: Path): Option[String]
  def wd: Path
}
