/*
 * Decompiled with CFR 0.152.
 */
package com.dtsx.astra.sdk;

import com.dtsx.astra.sdk.AstraDB;
import com.dtsx.astra.sdk.db.AstraDBOpsClient;
import com.dtsx.astra.sdk.db.DbOpsClient;
import com.dtsx.astra.sdk.db.domain.CloudProviderType;
import com.dtsx.astra.sdk.db.domain.Database;
import com.dtsx.astra.sdk.db.domain.DatabaseCreationRequest;
import com.dtsx.astra.sdk.db.domain.DatabaseStatusType;
import com.dtsx.astra.sdk.db.exception.DatabaseNotFoundException;
import com.dtsx.astra.sdk.utils.ApiLocator;
import com.dtsx.astra.sdk.utils.Assert;
import com.dtsx.astra.sdk.utils.AstraEnvironment;
import com.dtsx.astra.sdk.utils.AstraRc;
import io.stargate.sdk.data.DataApiClient;
import io.stargate.sdk.utils.AnsiUtils;
import io.stargate.sdk.utils.Utils;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AstraDBAdmin {
    private static final Logger log = LoggerFactory.getLogger(AstraDBAdmin.class);
    public static final int CONNECT_TIMEOUT_SECONDS = 20;
    public static final CloudProviderType FREE_TIER_CLOUD = CloudProviderType.GCP;
    public static final String FREE_TIER_CLOUD_REGION = "us-east1";
    public static final String TOKEN_HEADER_PARAM = "X-Token";
    public static final String DEFAULT_KEYSPACE = "default_keyspace";
    final AstraDBOpsClient devopsDbClient;
    final AstraEnvironment env;
    final String token;
    final HttpClient httpClient;
    static String astraConfigToken;

    public AstraDBAdmin() {
        this(astraConfigToken);
    }

    public AstraDBAdmin(String token) {
        this(token, AstraEnvironment.PROD);
    }

    public AstraDBAdmin(String token, AstraEnvironment env) {
        this.env = env;
        this.token = token;
        this.devopsDbClient = new AstraDBOpsClient(token, env);
        this.httpClient = HttpClient.newBuilder().version(HttpClient.Version.HTTP_2).connectTimeout(Duration.ofSeconds(20L)).build();
    }

    public Stream<Database> findAllDatabases() {
        return this.devopsDbClient.findAllNonTerminated().filter(db -> db.getInfo().getDbType() != null);
    }

    public UUID createDatabase(@NonNull String name) {
        if (name == null) {
            throw new NullPointerException("name is marked non-null but is null");
        }
        return this.createDatabase(name, FREE_TIER_CLOUD, FREE_TIER_CLOUD_REGION);
    }

    public UUID createDatabase(@NonNull String name, @NonNull CloudProviderType cloud, @NonNull String cloudRegion) {
        if (name == null) {
            throw new NullPointerException("name is marked non-null but is null");
        }
        if (cloud == null) {
            throw new NullPointerException("cloud is marked non-null but is null");
        }
        if (cloudRegion == null) {
            throw new NullPointerException("cloudRegion is marked non-null but is null");
        }
        Optional<Database> optDb = this.findDatabaseByName(name).findFirst();
        if (optDb.isPresent()) {
            Database db = optDb.get();
            switch (db.getStatus()) {
                case ACTIVE: {
                    log.info("Database " + AnsiUtils.green((String)"{}") + " already exists and is ACTIVE.", (Object)name);
                    return UUID.fromString(db.getId());
                }
                case MAINTENANCE: 
                case INITIALIZING: 
                case PENDING: 
                case RESUMING: {
                    log.info("Database {} already exists and is in {} state, waiting for it to be ACTIVE", (Object)name, (Object)db.getStatus());
                    this.waitForDatabase(this.devopsDbClient.database(db.getId()));
                    return UUID.fromString(db.getId());
                }
                case HIBERNATED: {
                    log.info("Database {} is in {} state, resuming...", (Object)name, (Object)db.getStatus());
                    this.resumeDb(db);
                    this.waitForDatabase(this.devopsDbClient.database(db.getId()));
                    return UUID.fromString(db.getId());
                }
            }
            throw new IllegalStateException("Database already exist but cannot be activate");
        }
        UUID newDbId = UUID.fromString(this.devopsDbClient.create(DatabaseCreationRequest.builder().name(name).cloudProvider(cloud).cloudRegion(cloudRegion).keyspace(DEFAULT_KEYSPACE).withVector().build()));
        log.info("Database {} is starting (id={}): it will take about a minute please wait...", (Object)name, (Object)newDbId);
        this.waitForDatabase(this.devopsDbClient.database(newDbId.toString()));
        return newDbId;
    }

    public boolean deleteDatabaseByName(@NonNull String name) {
        if (name == null) {
            throw new NullPointerException("name is marked non-null but is null");
        }
        Optional<Database> opDb = this.findDatabaseByName(name).findFirst();
        opDb.ifPresent(db -> this.devopsDbClient.database(db.getId()).delete());
        return opDb.isPresent();
    }

    public boolean deleteDatabaseById(@NonNull UUID databaseId) {
        if (databaseId == null) {
            throw new NullPointerException("databaseId is marked non-null but is null");
        }
        if (this.findDatabaseById(databaseId).isPresent()) {
            this.devopsDbClient.database(databaseId.toString()).delete();
            return true;
        }
        return false;
    }

    public Stream<Database> findDatabaseByName(String name) {
        Assert.hasLength((String)name, (String)"Database name");
        return this.findAllDatabases().filter(db -> name.equals(db.getInfo().getName()));
    }

    public boolean isDatabaseExists(String name) {
        return this.findDatabaseByName(name).findFirst().isPresent();
    }

    public Optional<Database> findDatabaseById(@NonNull UUID id) {
        if (id == null) {
            throw new NullPointerException("id is marked non-null but is null");
        }
        Assert.notNull((Object)id, (String)"Database identifier should not be null");
        return this.devopsDbClient.findById(id.toString());
    }

    public AstraDB database(@NonNull String databaseName) {
        if (databaseName == null) {
            throw new NullPointerException("databaseName is marked non-null but is null");
        }
        List dbs = this.findDatabaseByName(databaseName).collect(Collectors.toList());
        if (dbs.isEmpty()) {
            throw new DatabaseNotFoundException(databaseName);
        }
        if (dbs.size() > 1) {
            throw new IllegalStateException("More than one database exists with the same name, use id.");
        }
        return this.database(UUID.fromString(((Database)dbs.get(0)).getId()));
    }

    public AstraDB database(UUID databaseId) {
        return new AstraDB(this.token, databaseId, null, this.env, DEFAULT_KEYSPACE);
    }

    private void waitForDatabase(DbOpsClient dbc) {
        long top = System.currentTimeMillis();
        while (DatabaseStatusType.ACTIVE != this.getStatus(dbc) && System.currentTimeMillis() - top < 180000L) {
            try {
                Thread.sleep(5000L);
                System.out.print("\u25a0");
                System.out.flush();
            }
            catch (InterruptedException e) {
                log.warn("Interrupted {}", (Object)e.getMessage());
                Thread.currentThread().interrupt();
            }
        }
        if (this.getStatus(dbc) != DatabaseStatusType.ACTIVE) {
            throw new IllegalStateException("Database is not in expected state after timeouts");
        }
    }

    private DatabaseStatusType getStatus(DbOpsClient dbc) {
        return ((Database)dbc.find().orElseThrow(() -> new DatabaseNotFoundException(dbc.getDatabaseId()))).getStatus();
    }

    private void resumeDb(Database db) {
        try {
            String endpoint = ApiLocator.getApiRestEndpoint((String)db.getId(), (String)db.getInfo().getRegion()) + "/v2/schemas/keyspace";
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(endpoint)).timeout(Duration.ofSeconds(20L)).header("Content-Type", "application/json").header(TOKEN_HEADER_PARAM, this.token).GET().build();
            HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() == 500) {
                throw new IllegalStateException("Cannot resume database, please check your account");
            }
        }
        catch (InterruptedException e) {
            log.warn("Interrupted {}", (Object)e.getMessage());
            Thread.currentThread().interrupt();
        }
        catch (Exception e) {
            log.warn("Resuming request might have failed, please check {}}", (Object)e.getMessage());
        }
    }

    public DataApiClient getDataApiClient(@NonNull String databaseName) {
        if (databaseName == null) {
            throw new NullPointerException("databaseName is marked non-null but is null");
        }
        return this.database(databaseName).getApiClient();
    }

    public DataApiClient getDataApiClient(@NonNull UUID databaseId) {
        if (databaseId == null) {
            throw new NullPointerException("databaseId is marked non-null but is null");
        }
        return this.database(databaseId).getApiClient();
    }

    public AstraDBOpsClient getDevopsApiClient() {
        return this.devopsDbClient;
    }

    public String getToken() {
        return this.token;
    }

    static {
        new AstraRc().getSectionKey("default", "ASTRA_DB_APPLICATION_TOKEN").ifPresent(s -> {
            astraConfigToken = s;
        });
        Utils.readEnvVariable((String)"ASTRA_DB_APPLICATION_TOKEN").ifPresent(s -> {
            astraConfigToken = s;
        });
    }
}

