/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.dse.graph.api;

import com.datastax.driver.dse.graph.GraphStatement;
import org.apache.tinkerpop.gremlin.process.traversal.dsl.graph.GraphTraversal;

/**
 * A class allowing to execute multiple DSE Graph mutation operations inside the same transaction.
 * For more information, see {@link com.datastax.dse.graph.api.DseGraph#batch}.
 */
public abstract class TraversalBatch implements Iterable<GraphTraversal> {

  /**
   * Add one traversal to this {@link TraversalBatch} instance (thread-safe).
   *
   * @param graphTraversal the mutation traversal to add to this {@link TraversalBatch} instance.
   */
  public abstract void add(GraphTraversal graphTraversal);

  /**
   * Add multiple traversals to this {@link TraversalBatch} instance (thread-safe).
   *
   * @param newTraversals the mutation traversals to add to this {@link TraversalBatch} instance.
   */
  public abstract void addAll(Iterable<GraphTraversal> newTraversals);

  /**
   * Return the number of operations currently contained in the {@link TraversalBatch}.
   *
   * <p>Calling this regularly is not great for performance. Also, this number may not be accurate
   * in heavily concurrent situations.
   *
   * @return the number of operations currently contained in the {@link TraversalBatch}.
   */
  public abstract int size();

  /**
   * Convert the {@link com.datastax.dse.graph.api.TraversalBatch} into a {@link
   * com.datastax.driver.dse.graph.GraphStatement} to execute on a {@link
   * com.datastax.driver.dse.DseSession}.
   *
   * @return an executable statement instance.
   */
  public abstract GraphStatement asGraphStatement();
}
