/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.dse.graph.internal;

import static com.datastax.dse.graph.internal.utils.StringUtils.getOptimalStringAlignmentDistance;

import com.datastax.dse.graph.api.predicates.Search;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang.StringUtils;

/**
 * List of predicates for geolocation usage with DseGraph and Search indexes. Should not be accessed
 * directly but through the {@link Search} static methods.
 */
public enum SearchPredicate implements DsePredicate {
  /** Whether the text contains a given term as a token in the text (case insensitive). */
  token {
    @Override
    public boolean test(Object value, Object condition) {
      preEvaluate(condition);
      return value != null && evaluate(value.toString(), (String) condition);
    }

    boolean evaluate(String value, String terms) {
      Set<String> tokens = Sets.newHashSet(tokenize(value.toLowerCase()));
      terms = terms.trim();
      List<String> tokenTerms = tokenize(terms.toLowerCase());
      if (!terms.isEmpty() && tokenTerms.isEmpty()) return false;
      for (String term : tokenTerms) {
        if (!tokens.contains(term)) return false;
      }
      return true;
    }

    @Override
    public boolean isValidCondition(Object condition) {
      return condition != null && StringUtils.isNotBlank((String) condition);
    }

    @Override
    public String toString() {
      return "token";
    }
  },

  /** Whether the text contains a token that starts with a given term (case insensitive). */
  tokenPrefix {
    @Override
    public boolean test(Object value, Object condition) {
      preEvaluate(condition);
      return value != null && evaluate(value.toString(), (String) condition);
    }

    boolean evaluate(String value, String prefix) {
      for (String token : tokenize(value.toLowerCase())) {
        if (token.startsWith(prefix.toLowerCase().trim())) return true;
      }
      return false;
    }

    @Override
    public boolean isValidCondition(Object condition) {
      return condition != null;
    }

    @Override
    public String toString() {
      return "tokenPrefix";
    }
  },

  /** Whether the text contains a token that matches a regular expression (case insensitive). */
  tokenRegex {
    @Override
    public boolean test(Object value, Object condition) {
      preEvaluate(condition);
      return value != null && evaluate(value.toString(), (String) condition);
    }

    boolean evaluate(String value, String regex) {
      Pattern compiled = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
      for (String token : tokenize(value.toLowerCase())) {
        if (compiled.matcher(token).matches()) return true;
      }
      return false;
    }

    @Override
    public boolean isValidCondition(Object condition) {
      return condition != null && StringUtils.isNotBlank((String) condition);
    }

    @Override
    public String toString() {
      return "tokenRegex";
    }
  },

  /**
   * Whether some token in the text is within a given edit distance from the given term (case
   * insensitive).
   */
  tokenFuzzy {
    @Override
    public boolean test(Object value, Object condition) {
      preEvaluate(condition);
      if (value == null) return false;

      EditDistance fuzzyCondition = (EditDistance) condition;

      for (String token : tokenize(value.toString().toLowerCase())) {
        if (getOptimalStringAlignmentDistance(token, fuzzyCondition.query.toLowerCase())
            <= fuzzyCondition.distance) {
          return true;
        }
      }

      return false;
    }

    @Override
    public boolean isValidCondition(Object condition) {
      return condition != null;
    }

    @Override
    public String toString() {
      return "tokenFuzzy";
    }
  },

  /** Whether the text starts with a given prefix (case sensitive). */
  prefix {
    @Override
    public boolean test(Object value, Object condition) {
      preEvaluate(condition);
      return value != null && value.toString().startsWith(((String) condition).trim());
    }

    @Override
    public boolean isValidCondition(Object condition) {
      return condition != null;
    }

    @Override
    public String toString() {
      return "prefix";
    }
  },

  /** Whether the text matches a regular expression (case sensitive). */
  regex {
    @Override
    public boolean test(Object value, Object condition) {
      preEvaluate(condition);
      return value != null
          && Pattern.compile((String) condition, Pattern.DOTALL)
              .matcher(value.toString())
              .matches();
    }

    @Override
    public boolean isValidCondition(Object condition) {
      return condition != null && StringUtils.isNotBlank((String) condition);
    }

    @Override
    public String toString() {
      return "regex";
    }
  },

  /** Whether the text is within a given edit distance from the given term (case sensitive). */
  fuzzy {
    @Override
    public boolean test(Object value, Object condition) {
      preEvaluate(condition);
      if (value == null) return false;
      EditDistance fuzzyCondition = (EditDistance) condition;
      return getOptimalStringAlignmentDistance(value.toString(), fuzzyCondition.query)
          <= fuzzyCondition.distance;
    }

    @Override
    public boolean isValidCondition(Object condition) {
      return condition != null;
    }

    @Override
    public String toString() {
      return "fuzzy";
    }
  },

  /**
   * Whether tokenized text contains a given phrase, optionally within a given proximity (case
   * insensitive).
   */
  phrase {
    @Override
    public boolean test(Object value, Object condition) {
      preEvaluate(condition);
      if (value == null) return false;

      EditDistance phraseCondition = (EditDistance) condition;

      List<String> valueTokens = tokenize(value.toString().toLowerCase());
      List<String> phraseTokens = tokenize(phraseCondition.query.toLowerCase());

      int valuePosition = 0;
      int phrasePosition = 0;
      int distance = 0;

      // Look for matches while phrase/value tokens and distance budget remain
      while (phrasePosition < phraseTokens.size()
          && valuePosition < valueTokens.size()
          && distance <= phraseCondition.distance) {

        if (phraseTokens.get(phrasePosition).equals(valueTokens.get(valuePosition))) {
          // Early return-true when we've matched the whole phrase (within the specified distance)
          if (phrasePosition == phraseTokens.size() - 1) return true;
          phrasePosition++;
        } else if (0 < phrasePosition) {
          // We've previously found at least one matching token in the input string,
          // but the current token does not match the phrase.  Increment distance.
          distance++;
        }

        valuePosition++;
      }

      return false;
    }

    @Override
    public boolean isValidCondition(Object condition) {
      return condition != null;
    }

    @Override
    public String toString() {
      return "phrase";
    }
  };

  // Match anything that is not either:
  // 1) a unicode letter, regardless of subcategory (same as Character.isLetter), or
  // 2) a unicode decimal digit number (same as Character.isDigit)
  private static final Pattern TOKEN_SPLIT_PATTERN = Pattern.compile("[^\\p{L}\\p{Nd}]");

  static List<String> tokenize(String str) {
    String[] rawTokens = TOKEN_SPLIT_PATTERN.split(str); // could contain empty strings
    return Stream.of(rawTokens).filter(t -> 0 < t.length()).collect(Collectors.toList());
  }
}
