/*
 * Copyright DataStax, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.datastax.oss.driver.internal.mapper.processor;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.JavaFile;

/** A code generator that produces exactly one source file. */
public abstract class SingleFileCodeGenerator implements CodeGenerator {

  /** A generic warning that can get added to the javadoc of any generated type. */
  public static final String JAVADOC_GENERATED_WARNING =
      "Generated by the DataStax driver mapper, do not edit directly.\n";

  public static final String JAVADOC_PARAGRAPH_SEPARATOR = "\n\n<p>";

  protected final ProcessorContext context;

  protected SingleFileCodeGenerator(ProcessorContext context) {
    this.context = context;
  }

  @Override
  public void generate() {
    ClassName typeName = getPrincipalTypeName();
    String fileName = typeName.simpleName();
    if (!typeName.packageName().isEmpty()) {
      fileName = typeName.packageName() + '.' + fileName;
    }
    context.getFiler().write(fileName, getContents());
  }

  /** Name of the principal type being declared in this file. */
  protected abstract ClassName getPrincipalTypeName();

  protected abstract JavaFile.Builder getContents();
}
