/*
 * Copyright DataStax, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.datastax.oss.driver.internal.mapper.processor;

import com.datastax.oss.driver.api.mapper.annotations.Dao;
import com.datastax.oss.driver.api.mapper.annotations.Entity;
import com.datastax.oss.driver.api.mapper.annotations.Mapper;
import com.squareup.javapoet.ClassName;
import javax.lang.model.element.TypeElement;

/** Centralizes the naming conventions for types or members generated by the processor. */
public class GeneratedNames {

  /**
   * A suffix that gets appended to the name of every generated class (to make it obvious that it
   * should not be referenced from client code, e.g. in IDE completion).
   */
  private static final String SUFFIX = "__MapperGenerated";

  /** The entity helpers' private constants holding generic type definitions. */
  public static final String GENERIC_TYPE_CONSTANT = "GENERIC_TYPE";

  /** The helper class generated for an {@link Entity}-annotated class. */
  public static ClassName entityHelper(TypeElement entityClass) {
    return entityHelper(ClassName.get(entityClass));
  }

  /** Variant for {@link #entityHelper(TypeElement)} when only the type name is known. */
  public static ClassName entityHelper(ClassName entityClassName) {
    return peerClass(entityClassName, "Helper" + SUFFIX);
  }

  /** The builder for a {@link Mapper}-annotated interface. */
  public static ClassName mapperBuilder(TypeElement mapperInterface) {
    String custom = mapperInterface.getAnnotation(Mapper.class).builderName();
    if (custom.isEmpty()) {
      // Note that the builder is referenced in client code, so generate a "normal" name (no
      // underscores).
      return peerClass(mapperInterface, "Builder");
    } else {
      int i = custom.lastIndexOf('.');
      return ClassName.get(custom.substring(0, i), custom.substring(i + 1));
    }
  }

  /** The implementation of a {@link Mapper}-annotated interface. */
  public static ClassName mapperImplementation(TypeElement mapperInterface) {
    return peerClass(mapperInterface, "Impl" + SUFFIX);
  }

  /** The implementation of a {@link Dao}-annotated interface. */
  public static ClassName daoImplementation(TypeElement daoInterface) {
    return peerClass(daoInterface, "Impl" + SUFFIX);
  }

  // Generates a non-nested peer class. If the base class is nested, the names of all enclosing
  // classes are prepended to ensure uniqueness. For example:
  // com.datastax.Foo.Bar.Baz => com.datastax.Foo_Bar_BazImpl
  private static ClassName peerClass(ClassName base, String suffix) {
    ClassName topLevel = base;
    StringBuilder prefix = new StringBuilder();
    while (topLevel.enclosingClassName() != null) {
      topLevel = topLevel.enclosingClassName();
      prefix.insert(0, '_').insert(0, topLevel.simpleName());
    }
    return topLevel.peerClass(prefix.toString() + base.simpleName() + suffix);
  }

  private static ClassName peerClass(TypeElement element, String suffix) {
    return peerClass(ClassName.get(element), suffix);
  }
}
