/*
 * Decompiled with CFR 0.152.
 */
package com.datorama.oss.timbermill.pipe;

import com.datorama.oss.timbermill.pipe.EventOutputPipe;
import com.datorama.oss.timbermill.pipe.SizedBoundEventsQueue;
import com.datorama.oss.timbermill.pipe.TimbermillServerOutputPipeBuilder;
import com.datorama.oss.timbermill.unit.Event;
import com.datorama.oss.timbermill.unit.EventsWrapper;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPOutputStream;
import org.apache.http.HttpHost;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimbermillServerOutputPipe
implements EventOutputPipe {
    private static final int HTTP_TIMEOUT = 10000;
    private static final int MAX_RETRY = 5;
    private static final Logger LOG = LoggerFactory.getLogger(TimbermillServerOutputPipe.class);
    private static volatile boolean keepRunning = true;
    private URL timbermillServerUrl;
    private SizedBoundEventsQueue buffer;
    private ExecutorService executorService;
    private boolean sendGzippedRequest;

    private TimbermillServerOutputPipe() {
    }

    TimbermillServerOutputPipe(TimbermillServerOutputPipeBuilder builder) {
        keepRunning = true;
        if (builder.timbermillServerUrl == null) {
            throw new RuntimeException("Must enclose the Timbermill server URL");
        }
        try {
            HttpHost httpHost = HttpHost.create((String)builder.timbermillServerUrl);
            this.timbermillServerUrl = new URL(String.valueOf(httpHost.toURI()) + "/events");
        }
        catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
        this.buffer = new SizedBoundEventsQueue(builder.maxBufferSize, builder.maxSecondsBeforeBatchTimeout);
        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("timbermill-sender-%d").build();
        this.executorService = Executors.newFixedThreadPool(builder.numOfThreads, namedThreadFactory);
        this.executeEventsSenders(builder.maxEventsBatchSize, builder.numOfThreads);
        this.sendGzippedRequest = builder.sendGzippedRequest;
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            LOG.info("Shutting down timbermill-senders executor service");
            keepRunning = false;
            this.executorService.shutdown();
            try {
                if (!this.executorService.awaitTermination(5000L, TimeUnit.MILLISECONDS)) {
                    LOG.info("Events buffer size was {} on shutdown", (Object)this.buffer.size());
                    this.executorService.shutdownNow();
                }
            }
            catch (InterruptedException interruptedException) {
                this.executorService.shutdownNow();
            }
        }));
    }

    private void executeEventsSenders(int maxEventsBatchSize, int numOfThreads) {
        Runnable getAndSendEventsTask = () -> {
            LOG.info("Starting send events thread");
            do {
                try {
                    List<Event> eventsToSend = this.buffer.getEventsOfSize(maxEventsBatchSize);
                    if (eventsToSend.isEmpty()) continue;
                    EventsWrapper eventsWrapper = new EventsWrapper(eventsToSend);
                    this.sendEvents(eventsWrapper);
                }
                catch (Exception e) {
                    LOG.error("Error sending events to Timbermill server", (Throwable)e);
                }
            } while (keepRunning);
        };
        int i = 0;
        while (i < numOfThreads) {
            this.executorService.execute(getAndSendEventsTask);
            ++i;
        }
    }

    public void close() {
        LOG.info("Gracefully shutting down Timbermill output pipe.");
        keepRunning = false;
        LOG.info("Timbermill server was output pipe.");
        this.executorService.shutdown();
        while (!this.executorService.isTerminated()) {
            try {
                Thread.sleep(2000L);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    private void sendEvents(EventsWrapper eventsWrapper) throws IOException {
        byte[] eventsWrapperBytes = this.getEventsWrapperBytes(eventsWrapper);
        int tryNum = 1;
        while (tryNum <= 5) {
            try {
                HttpURLConnection httpCon = this.getHttpURLConnection();
                this.sendEventsOverConnection(httpCon, eventsWrapperBytes);
                int responseCode = httpCon.getResponseCode();
                if (responseCode == 200) {
                    LOG.debug("{} events were sent to Timbermill server", (Object)eventsWrapper.getEvents().size());
                    return;
                }
                LOG.warn("Request #" + tryNum + " to Timbermill return status {}, Attempt: {}/{} Message: {}", new Object[]{responseCode, tryNum, 5, httpCon.getResponseMessage()});
            }
            catch (Exception e) {
                LOG.warn("Request #" + tryNum + " to Timbermill failed, Attempt: " + tryNum + "/" + 5, (Throwable)e);
            }
            try {
                Thread.sleep((long)(Math.pow(2.0, tryNum) * 1000.0));
            }
            catch (InterruptedException interruptedException) {}
            ++tryNum;
        }
        LOG.error("Can't send events to Timbermill, failed {} attempts.\n Failed request: {} ", (Object)5, (Object)new String(eventsWrapperBytes));
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void sendEventsOverConnection(HttpURLConnection httpCon, byte[] eventsWrapperBytes) throws IOException {
        Throwable throwable = null;
        Object var4_5 = null;
        try {
            OutputStream httpConOS = httpCon.getOutputStream();
            try {
                try (OutputStream os = this.sendGzippedRequest ? new GZIPOutputStream(httpConOS, eventsWrapperBytes.length) : httpConOS;){
                    os.write(eventsWrapperBytes);
                }
                if (httpConOS == null) return;
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                if (httpConOS == null) throw throwable;
                httpConOS.close();
                throw throwable;
            }
            httpConOS.close();
            return;
        }
        catch (Throwable throwable3) {
            if (throwable == null) {
                throwable = throwable3;
                throw throwable;
            } else {
                if (throwable == throwable3) throw throwable;
                throwable.addSuppressed(throwable3);
            }
            throw throwable;
        }
    }

    private byte[] getEventsWrapperBytes(EventsWrapper eventsWrapper) throws JsonProcessingException {
        ObjectMapper om = new ObjectMapper();
        return om.writeValueAsBytes((Object)eventsWrapper);
    }

    private HttpURLConnection getHttpURLConnection() throws IOException {
        HttpURLConnection httpURLConnection = (HttpURLConnection)this.timbermillServerUrl.openConnection();
        httpURLConnection.setRequestMethod("POST");
        httpURLConnection.setRequestProperty("content-type", "application/json");
        httpURLConnection.setDoOutput(true);
        httpURLConnection.setConnectTimeout(10000);
        httpURLConnection.setReadTimeout(10000);
        if (this.sendGzippedRequest) {
            httpURLConnection.setRequestProperty("Content-Encoding", "gzip");
        }
        return httpURLConnection;
    }

    public void send(Event e) {
        if (!this.buffer.offer(e)) {
            LOG.warn("Event {} was removed from the queue due to insufficient space", (Object)e.getTaskId());
        }
    }

    public int getCurrentBufferSize() {
        return this.buffer.size();
    }
}

