/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image.internal;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.imageio.spi.IIORegistry;
import javax.imageio.spi.ImageInputStreamSpi;
import javax.imageio.spi.ImageOutputStreamSpi;
import javax.imageio.spi.ImageReaderSpi;
import javax.imageio.spi.ImageWriterSpi;
import javax.imageio.spi.ServiceRegistry;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleListener;

import com.day.image.ImageSupport;

/**
 * The <code>Activator</code> is used to unregister the GIF Image Writer
 * provided by this bundle from the system's list of Image Writers
 */
public class Activator implements BundleActivator, BundleListener {

    // the name of the folder containing the service provider registrations
    private static final String SERVICES_FOLDER = "META-INF/services/";

    // map of dynamically registered ImageIO plugins indexed by the
    // source bundle ID
    private Map<Long, List<Object>> perBundleProvider = new HashMap<Long, List<Object>>();

    public void start(BundleContext context) {
        // make sure the all custom image io classes are registered
        ImageSupport.registerImageIOSpi();

        // register ImageIO providers of active bundles
        for (Bundle bundle : context.getBundles()) {
            if (bundle.getState() == Bundle.ACTIVE) {
                registerImageIOProvider(bundle);
            }
        }

        // start listening for bundle activity
        context.addBundleListener(this);
    }

    public void stop(BundleContext context) {
        // stop listening for bundle activity
        context.removeBundleListener(this);

        // unregister ImageIO provider from the bundles we have been managing
        IIORegistry reg = IIORegistry.getDefaultInstance();
        for (List<Object> providers : perBundleProvider.values()) {
            unregisterImageIOProvider(reg, providers);
        }

        // ensure the GIF Image Writer is unregistered when stopped
        ImageSupport.deregisterImageIOSpi();
    }

    // ---------- BundleListener

    /**
     * Check for ImageIO plugins of started bundles and ensure ImageIO of
     * stopped bundles are unregistered.
     */
    public void bundleChanged(BundleEvent event) {
        switch (event.getType()) {
            case BundleEvent.STARTED:
                // register provider
                registerImageIOProvider(event.getBundle());
                break;
            case BundleEvent.STOPPED:
                // unregister provider
                unregisterImageIOProvider(event.getBundle());
                break;
        }
    }

    /**
     * Registers any ImageIO plugins provided by the given bundle. If there are
     * any the plugins are registered with the default ImageIO Registry and also
     * noted in the {@link #perBundleProvider} map for later unregistration.
     */
    private void registerImageIOProvider(final Bundle bundle) {

        // the classe loader (proxy) to load the plugins from the bundle
        final BundleProxyClassLoader loader = new BundleProxyClassLoader(bundle);

        // check for known SPIs
        final List<Object> providers = new ArrayList<Object>();
        loadProviders(providers, loader, ImageInputStreamSpi.class);
        loadProviders(providers, loader, ImageOutputStreamSpi.class);
        loadProviders(providers, loader, ImageReaderSpi.class);
        loadProviders(providers, loader, ImageWriterSpi.class);

        // if there are any, install them
        if (!providers.isEmpty()) {
            final IIORegistry reg = IIORegistry.getDefaultInstance();
            registerImageIOProvider(reg, providers);
            perBundleProvider.put(bundle.getBundleId(), providers);
        }
    }

    /**
     * Unregisters the plugins registered from the given bundle. If the given
     * bundles does not have any plugins registered, this method does nothing.
     */
    private void unregisterImageIOProvider(final Bundle bundle) {
        final IIORegistry reg = IIORegistry.getDefaultInstance();

        List<Object> providers = perBundleProvider.remove(bundle.getBundleId());
        if (providers != null) {
            unregisterImageIOProvider(reg, providers);
        }
    }

    /**
     * Registers the plugins from the <code>providers</code> list with the
     * ImageIO Registry <code>reg</code>.
     */
    private void registerImageIOProvider(IIORegistry reg, List<Object> providers) {
        reg.registerServiceProviders(providers.iterator());
    }

    /**
     * Unregisters the plugins from the <code>providers</code> list from the
     * ImageIO Registry <code>reg</code>.
     */
    private void unregisterImageIOProvider(IIORegistry reg,
            List<Object> providers) {

        for (Object provider : providers) {
            reg.deregisterServiceProvider(provider);
        }
    }

    /**
     * Loads the ImageIO plugin SPIs for the given SPI type through the
     * classloader attached to the bundle.
     *
     * @param <T> The ImageIO plugin SPI type to look up
     * @param providers The list of SPI objects to which the found are added
     * @param bundleLoader The ClassLoader to use for finding new SPI instances
     * @param spiClass The ImageIO plugin class object
     */
    private <T> void loadProviders(List<Object> providers,
            BundleProxyClassLoader bundleLoader, Class<T> spiClass) {
        final String entryName = SERVICES_FOLDER + spiClass.getName();
        if (bundleLoader.hasResource(entryName)) {
            final ClassLoader old = Thread.currentThread().getContextClassLoader();
            try {
                Thread.currentThread().setContextClassLoader(bundleLoader);
                Iterator<T> spi = ServiceRegistry.lookupProviders(spiClass, bundleLoader);
                while (spi.hasNext()) {
                    try {
                        providers.add(spi.next());
                    } catch (Throwable t) {
                        // hmm, huston we have a problem ???
                        System.err.println(t);
                        t.printStackTrace(System.err);
                    }
                }
            } finally {
                Thread.currentThread().setContextClassLoader(old);
            }
        }
    }

    /**
     * The <code>BundleProxyClassLoader</code> is a class loader which delegates
     * all calls to load classes and access resources to a single underlying
     * bundle without any further question.
     */
    private static class BundleProxyClassLoader extends ClassLoader {

        private final Bundle bundle;

        public BundleProxyClassLoader(final Bundle bundle) {
            this.bundle = bundle;
        }

        @Override
        protected synchronized Class<?> loadClass(String name, boolean resolve)
                throws ClassNotFoundException {
            return bundle.loadClass(name);
        }

        @Override
        public URL getResource(String name) {
            return bundle.getResource(name);
        }

        @SuppressWarnings("unchecked")
        @Override
        public Enumeration<URL> getResources(String name) throws IOException {
            return bundle.getResources(name);
        }

        boolean hasResource(String name) {
            return bundle.getResource(name) != null;
        }
    }
}
