/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image.internal.font;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.jcr.observation.EventIterator;

/**
 * The <code>AbstractFontCache</code> class is the abstract base class to
 * implement a font data cache. This class extends the <code>java.util.HashMap</code>
 * class implementing the entry values as weak references, such that they might
 * be garbage collected.
 * <p>
 * Implementations have to implemnt the {@link #pageModified} method to handle
 * observation events in the configured font path.
 *
 * @version $Revision: 20173 $, $Date: 2006-05-03 16:47:57 +0200 (Mit, 03 Mai 2006) $
 * @author fmeschbe
 * @since degu
 * @audience core
 */
public abstract class AbstractFontCache<T> {

    /** The queue being filled with references of collected values */
    private final ReferenceQueue<T> reaped = new ReferenceQueue<T>();

    private final Map<String, ValueRef<T>>contents = new HashMap<String, ValueRef<T>>();

    /**
     * Deregisters from the observation service if registered and removes all
     * entries from the font cache.
     * <p>
     * Implementations of this class may overwrite this method but must make
     * sure call this base method to really destroy the instance.
     */
    public void destroy() {
        clear();
    }

    //---------- ModificationListener ------------------------------------------

    /**
     * Called when a <code>PageModification</code> takes place
     * @param modification page modification that took place
     */
    public abstract void onEvent(EventIterator events);

    //---------- HashMap overwrites for weak references ------------------------

    /**
     * Returns the value to which the specified key is mapped in this identity
     * hash map, or <tt>null</tt> if the map contains no mapping for this key.
     * A return value of <tt>null</tt> does not <i>necessarily</i> indicate
     * that the map contains no mapping for the key; it is also possible that
     * the map explicitly maps the key to <tt>null</tt>. The
     * <tt>containsKey</tt> method may be used to distinguish these two cases.
     * <p>
     * This method might also return <code>null</code> in case the referent
     * of the weak reference value has already been collected. In such a situation
     * the <code>containsKey</code> method still returns <code>true</code>.
     *
     * @param   key the key whose associated value is to be returned.
     * @return  the value to which this map maps the specified key, or
     *          <tt>null</tt> if the map contains no mapping for this key.
     * @see #put(Object, Object)
     */
    public T get(Object key) {
        ValueRef<T> vr = contents.get(key);
        return vr != null ? vr.get() : null;
    }

    /**
     * Associates the specified value with the specified key in this map.
     * If the map previously contained a mapping for this key, the old
     * value is replaced.
     * <p>
     * If the map already contained a mapping the value is considered to be a
     * <code>Reference</code> instance, which is cleared to help the garbage
     * collector.
     * <p>
     * This implementation does not store the key value directly but wraps the
     * key in a <code>Reference</code> object for the key value to be available
     * to garbage collection.
     * <p>
     * Before storing the new value, the map is cleared from entries, whose
     * referents have already been collected.
     * <p>
     * If the map already contained a value with this key, the referent of that
     * reference value is returned and cleared from the reference. If the
     * referent has already been collected the return value is <code>null</code>.
     * In the other case that the referent has not yet been collected, the
     * referent is again hard referenced, which prevents it from being collected.
     *
     * @param key key with which the specified value is to be associated.
     * @param value value to be associated with the specified key.
     * @return previous value associated with specified key, or <tt>null</tt>
     *	       if there was no mapping for key.  A <tt>null</tt> return can
     *	       also indicate that the HashMap previously associated
     *	       <tt>null</tt> with the specified key.
     */
    public T put(String key, T value) {
        reap();
        ValueRef<T> vr = new ValueRef<T>(value, key, reaped);
        return dereference(contents.put(key, vr));
    }

    /**
     * Copies all of the mappings from the specified map to this map
     * These mappings will replace any mappings that
     * this map had for any of the keys currently in the specified map.
     * <p>
     * Before storing the new values, the map is cleared from entries, whose
     * referents have already been collected.
     *
     * @param t mappings to be stored in this map.
     * @throws NullPointerException if the specified map is null.
     */
    public void putAll(Map<String, T> baseMap) {
        reap();
        for (Entry<String, T> entry : baseMap.entrySet()) {
            put(entry.getKey(), entry.getValue());
        }
    }

    /**
     * Removes the mapping for this key from this map if present.
     * <p>
     * If the map contained a mapping the value is considered to be a
     * <code>Reference</code> instance, which is cleared to help the garbage
     * collector. The return value is not the refernce object but the referent
     * which have been contained in the reference. If the referent has already
     * been collected, <code>null</code> is returned anyway.
     * <p>
     * Before removing the indicated entry, the map is cleared from entries,
     * whose referents have already been collected.
     *
     * @param  key key whose mapping is to be removed from the map.
     * @return previous value associated with specified key, or <tt>null</tt>
     *	       if there was no mapping for key.  A <tt>null</tt> return can
     *	       also indicate that the map previously associated <tt>null</tt>
     *	       with the specified key.
     */
    public Object remove(Object key) {
        reap();
        return dereference(contents.remove(key));
    }

    /**
     * Removes all mappings from this map.
     * <p>
     * Before clearing the map it is cleared from entries, whose referents have
     * already been collected.
     */
    public void clear() {
        reap();
        contents.clear();
    }

    //--------- Object overwrite -----------------------------------------------

    /**
     * Returns true if the other object is the same as this object. This is
     * actually the same implementation as the <code>Object.equals</code> but
     * overwrites the implementation of the base class which equals to
     * objects of the same type if the entries are equal. If both maps are empty
     * they are equal ...
     *
     * @param o The object to compare to.
     *
     * @return <code>true</code> if the other object is the same as this.
     */
    public boolean equals(Object o) {
        return this == o;
    }

    /**
     * Returns a hash code for this instance. Since we overwrite the
     * {@link #equals} we should accordingly adapt the hashcode.
     *
     * @return The hashcode of this instance.
     */
    public int hashCode() {
        return getClass().getName().hashCode();
    }

    /**
     * Returns a string representation of this instance consisting of the class
     * name and the result of the base class <code>toString</code> method.
     *
     * @return The string representation of this instance.
     */
    public String toString() {
        return getClass().getName() + super.toString();
    }

    //--------- reaping --------------------------------------------------------

    /**
     * Frees the map from all values having been collected. All references whose
     * referents have been collected are placed on the {@link #reaped} queue,
     * from where they are accessible.
     */
    @SuppressWarnings("unchecked")
    public void reap() {
        ValueRef<T> ref;
        while ((ref = (ValueRef<T>) reaped.poll()) != null) {
            contents.remove(ref.key);
            ref.key = null; // help gc
        }
    }

    // --------- internal ------------------------------------------------------

    /**
     * If the object is a <code>Reference</code>, the referent is extracted,
     * cleared from the reference and returned. Else the object itself is simply
     * returned.
     *
     * @param obj The <code>Object</code> to dereference if it is a
     *      <code>Reference</code>
     *
     * @return The object if it is not a <code>Reference</code> or the referent
     *      of the reference if not yet collected.
     */
    private T dereference(ValueRef<T> ref) {
        if (ref != null) {
            T result = ref.get();
            ref.clear();
            return result;
        }

        return null;
    }

    //---------- internal class ------------------------------------------------

    /**
     * The <code>ValueRef</code> class encapsulates a value in a
     * <code>WeakReference</code> which will be collected by an garbage
     * collection run. This helper class is used by the
     * {@link AbstractFontCache#reap()} method to clean up collected map entries.
     */
    private static class ValueRef<T> extends WeakReference<T> {
        private Object key;
        ValueRef(T val, Object key, ReferenceQueue<T> q) {
            super(val, q);
            this.key = key;
        }

        /**
         * Returns the hash code of this value references referent. If the
         * referent has alredy been cleared out, the hash code of this is
         * returned.
         *
         * @return The hash code of the referent or if the referent is
         *      <code>null</code> the hash code of this is returned.
         */
        public int hashCode() {
            /**
             * Compiler says :
             *      get() is inherited from java.lang.ref.Reference and hides
             *      method in outer class com.day.cq.image.AbstractFontCache.
             *      An explicit 'this' qualifier must be used to select the
             *      desired instance.
             */
            Object r = this.get();
            return r != null ? r.hashCode() : super.hashCode();
        }

        /**
         * Returns <code>true</code> if the referent equals the given object.
         * If the referent has already been cleared, <code>false</code> is
         * returned.
         *
         * @param obj The <code>Object</code> to compare to this
         *
         * @return <code>true</code> if <code>obj</code> and the referent object
         *      are the same or if the referent equals the object. If the
         *      referent has already been cleared, false is returned.
         */
        public boolean equals(Object obj) {
            /**
             * Compiler says :
             *      get() is inherited from java.lang.ref.Reference and hides
             *      method in outer class com.day.cq.image.AbstractFontCache.
             *      An explicit 'this' qualifier must be used to select the
             *      desired instance.
             */
            Object r = this.get();
            if (r == obj) {
                return true;
            } else if (r == null) {
                return false;
            } else {
                return r.equals(obj);
            }
        }
    }

}
