/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image.internal.font;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.felix.scr.annotations.Activate;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Service;
import org.osgi.framework.BundleContext;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/** Provides disk files to be used by the the Font.createFont()
 *  method, to avoid using the variant that uses an InputStream and
 *  creates multiple identical files in the File.createTempFile() folder.
 *
 *  This class derives filenames from a digest of the InputStreams
 *  used to create fonts, and reuses the same file multiple times for
 *  streams that have the same digest.
 */
@Component
@Service(FontFileProvider.class)
public class FontFileProviderImpl implements FontFileProvider {
    private final Logger log = LoggerFactory.getLogger(getClass());

    private static final String DIGEST = "SHA-1";
    private static final String FILENAME_PREFIX = FontFileProviderImpl.class.getSimpleName() + "_";
    private static final String FILE_EXT = ".font";
    private static final char[] HEX = "0123456789abcdef".toCharArray();
    private BundleContext bundleContext;
    private AtomicInteger tmpFileCounter = new AtomicInteger();

    @Activate
    protected void activate(ComponentContext ctx) {
        bundleContext = ctx.getBundleContext();

        // To avoid keeping unused files around,
        // enumerate files from our bundle's private data area,
        // and try to delete all that match our filename pattern.
        final File f = ctx.getBundleContext().getDataFile("FOO");
        deleteMatchingFiles(f.getParentFile(), FILENAME_PREFIX);
    }

    /** Try to delete all files from specified folder that have
     *  names starting with namePrefix. Failed deletions are
     *  just logged at the info level.
     */
    void deleteMatchingFiles(File folder, String namePrefix) {
        if(!folder.isDirectory()) {
            log.debug("deleteMatchingFiles: {} is not a folder", folder.getAbsolutePath());
            return;
        }
        final String [] names = folder.list();
        int deleted = 0;
        if(names != null) {
            for(String name : names) {
                if(name.startsWith(namePrefix)) {
                    final File f = new File(folder, name);
                    f.delete();
                    if(f.exists()) {
                        log.info(
                                "Failed to delete font file {}, might be in use. Will try again next time",
                                f.getAbsolutePath());
                    } else {
                        log.debug("Deleted possibly unused font file {}", f.getAbsolutePath());
                        deleted++;
                    }
                }
            }
        }

        if(deleted > 0) {
            log.info("{} possibly unused font files deleted from {}", deleted, folder.getAbsolutePath());
        } else {
            log.info("No unused font files found under {}", folder.getAbsolutePath());
        }
    }

    /** Return a disk file for the supplied InputStream,
     *  using the stream's digest to build the filename and reusing
     *  existing files when available.
     */
    public File getFileForStream(InputStream is) throws IOException {
        // Create a temp file and copy is to it, while computing its md5
        File result = null;
        File tmp = bundleContext.getDataFile("TMP_" + tmpFileCounter.incrementAndGet() + "_" + System.currentTimeMillis() + ".tmp");
        try {
            MessageDigest digest = null;
            try {
                digest = MessageDigest.getInstance(DIGEST);
            } catch(NoSuchAlgorithmException nse) {
                throw (IOException)new IOException("NoSuchAlgorithmException with digest=" + DIGEST).initCause(nse);
            }
            final OutputStream os = new DigestOutputStream(new FileOutputStream(tmp), digest);
            final byte [] buffer = new byte[8192];
            int n=0;
            while( (n=(is.read(buffer, 0, buffer.length))) > 0) {
                os.write(buffer,0, n);
            }
            os.flush();
            os.close();
            result = bundleContext.getDataFile(digestToFilename(digest.digest()));
            boolean fileExists = false;
            synchronized (this) {
                fileExists = result.canRead();
                if(!fileExists) {
                    tmp.renameTo(result);
                    if(!result.canRead()) {
                        throw new IOException("Error renaming " + tmp.getAbsolutePath() + " to " + result.getAbsolutePath());
                    }
                    tmp = null;
                }
            }

            if(fileExists) {
                log.debug("{} font file already provided, using it as is", result.getAbsolutePath());
            } else {
                result.deleteOnExit();
                log.info("Font file created for new content (will be deleted on exit): {}", result.getAbsolutePath());
            }

        } finally {
            if(tmp != null) {
                log.debug("Deleting unused temp file {}", tmp.getAbsolutePath());
                tmp.delete();
                if(tmp.exists()) {
                    log.warn("Failed to delete temporary file {}", tmp.getAbsolutePath());
                }
            }
            is.close();
        }

        return result;
    }

    /** Convert digest to filename, with our name prefix in front
     *  and using our extension.
     */
    private String digestToFilename(byte [] digest) {
        final StringBuilder sb = new StringBuilder(FILENAME_PREFIX);
        for(byte b : digest) {
            sb.append(HEX[(b >> 4) & 0x0f]);
            sb.append(HEX[b & 0x0f]);
        }
        sb.append(FILE_EXT);
        return sb.toString();
    }
}
