/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image.internal.font;

import java.awt.GraphicsEnvironment;
import java.util.ArrayList;
import java.util.Dictionary;
import java.util.List;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.commons.osgi.PropertiesUtil;
import org.osgi.framework.ServiceException;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.image.font.AbstractFont;
import com.day.image.font.FontListEntry;
import com.day.image.internal.font.resource.ResourceFontProvider;

/**
 * The <code>FontHelper</code> class implements a service to set up the
 * graphics environment. The main setup functionality is to setup the way the
 * Java runtime finds TrueType fonts and to specify the list of directories the
 * {@link FonterFont} class uses to load fonts from.
 * <p>
 * The configuration expected is defined in the
 * <code>resources/dtd/delivery/graphics.dtd</code> file.
 * <p>
 * <strong>{@link FonterFont} path</strong>
 * <p>
 * The path for the {@link FonterFont} class is specified in the
 * <code>fontpath</code> element, where each entry defines a ContentBus handle
 * to include in the path. This handle may contain globbing characters, in which
 * case the handles are resolved with the system ticket's {@link HandleExpander}
 * giving a list of handles to add to the path. This globbing evaluation only
 * takes place at service start time. Therefore only handles existing at that
 * time will be added to the path.
 * <p>
 * <strong>Java runtime font path</strong>
 * <p>
 * The font path for the Java runtime libraries is defined in the
 * <code>awtfontpath</code> element. Each entry specifies a file system path
 * to be added to the AWT font path. Relative names are taken relative to the
 * Communique 3 context. Entries may contain globbing characters, in which case
 * the names are expanded through the file system. Only paths existing at
 * service start time are added to the font path.
 * <p>
 * If the <code>appendJREFonts</code> attribute of the
 * <code>awtfontpath</code> is set to
 * <code>true</code> <em>${java.home}/lib/fonts</em> is added to the font
 * path.
 * <p>
 * Please see the notes on the Java runtime font handling below !
 * <p>
 * <strong>Please consider the following notes on the Java runtime font support</strong>
 * <p>
 * The font path handling is done in the platform specific
 * <code>GraphicsEnvironment</code> implementation. For Java platforms based
 * on Sun's Java implementation (obvioulsy Sun's implementation itself as well
 * as HP's implementation for HP-UX), this is handled in the 'internal'
 * <code>sun.java2d.SunGraphicsEnvironment</code> class.
 * <p>
 * The font support in the Java runtime library loads the list of fonts
 * available once during the uptime of a single Java VM. This means, that fonts
 * added after using the graphics environment - and specifically font support -
 * the first time, will not be recognized until after a JVM restart. Sad but
 * true :-)
 * <p>
 * <em><strong>
 * NOTE: The handling of the platform AWT fontpath is highly implementation
 * dependent as nothing is documented except for the use of the font.properties
 * which itself is platform dependent, though documented. The current
 * implementation has been tested and works on the following platforms and
 * implementations : Sun Java 1.3.1 on Solaris, Windows, and Linux; Sun Java
 * 1.4 on Windows and Linux; HP Java 1.3.1_06 on HP-UX. Additionally, it can
 * be assumed, that this should work on Sun Java 1.4 on Solaris, though this
 * has not been tested for now.
 * </strong></em>
 */
@Component(
        name = "com.day.cq.image.internal.font.FontHelper",
        metatype = true,
        label = "Day Commons GFX Font Helper",
        description = "Configures the GFX Font Helper which helps managing "
            + "the font path and caching of font information.")
public class FontHelper {

    /**
     * The instanced configured by the OSGi Declarative Services mechanism.
     */
    private static FontHelper INSTANCE;

    /** default log */
    private final Logger log = LoggerFactory.getLogger(getClass());

    @Property(
            cardinality = 2147483647,
            label = "Font Path",
            description = "The list of Resource Tree locations providing "
                + "fonts. Multiple entries may be listed. Each entry must "
                + "be an absolute path; that is empty entries or entries "
                + "not starting with a slash (/) character are ignored. "
                + "The location of each entry must be existing. This may "
                + "cause the creation of JCR Nodes at configured entry paths.")
    private static final String FONTPATH = "fontpath";

    /**
     * Default oversampling factor to apply when the
     * {@link AbstractFont#TTOVERSAMPLING} flag is set when drawing text (value
     * is 16).
     */
    private static final int DEFAULT_OVERSAMPLING_FACTOR = 16;

    @Property(
            intValue = DEFAULT_OVERSAMPLING_FACTOR,
            label = "Oversampling Factor",
            description = "The factor to apply internally to the font when "
                + "drawing text with the TTOVERSAMPLING flag set. Default "
                + "value is 16 which is also the value used by the Fonter.exe "
                + "tool to prepare FonterFonts.")
    private static final String OVERSAMPLING_FACTOR = "oversamplingFactor";

    @Reference
    private ResourceResolverFactory resourceResolverFactory;

    @Reference
    private FontFileProvider fontFileProvider;

    private ResourceResolver resourceResolver;

    /** The font path */
    private String[] fontPath = new String[0];

    /**
     * The list of font providers configured and valid
     */
    private FontProvider[] fontProviders = new FontProvider[0];

    // ---------- SCR integration

    /**
     * Configures and starts this service.
     */
    protected void activate(ComponentContext context) {

        try {
            resourceResolver = resourceResolverFactory.getAdministrativeResourceResolver(null);
        } catch (LoginException re) {
            log.error(
                "activate: Cannot get administrative session to access fonts !",
                re);
        }

        Dictionary<?, ?> props = context.getProperties();

        // initialize font stuff
        initFontPath(PropertiesUtil.toStringArray(props.get(FONTPATH)));
        PlatformFont.setOversamplingFactor(PropertiesUtil.toInteger(
            props.get(OVERSAMPLING_FACTOR), DEFAULT_OVERSAMPLING_FACTOR));

        INSTANCE = this;

        // register the font providers
        if (resourceResolver != null) {
            addFontProvider(new ResourceFontProvider());
        }
    }

    /**
     * Stops and deconfigures the service.
     */
    protected void deactivate(ComponentContext context) {
        for (FontProvider fontProvider : fontProviders) {
            fontProvider.destroy();
        }

        if (INSTANCE == this) {
            INSTANCE = null;
        }

        if (resourceResolver != null) {
            resourceResolver.close();
            resourceResolver = null;
        }

        // remove references
        fontProviders = new FontProvider[0];
        fontPath = new String[0];
    }

    protected void addFontProvider(FontProvider provider) {
        provider.init(resourceResolver, getFontPath(), fontFileProvider);
        if (fontProviders.length == 0) {
            fontProviders = new FontProvider[] { provider };
        } else {
            FontProvider[] newfp = new FontProvider[fontProviders.length + 1];
            System.arraycopy(fontProviders, 0, newfp, 0, fontProviders.length);
            newfp[fontProviders.length] = provider;
            fontProviders = newfp;
        }
    }

    protected void removeFontProvider(FontProvider provider) {
        for (int i = 0; i < fontProviders.length; i++) {
            if (fontProviders[i] == provider) {
                provider.destroy();

                if (fontProviders.length == 1) {
                    fontProviders = new FontProvider[0];
                } else {
                    FontProvider[] newfp = new FontProvider[fontProviders.length - 1];
                    if (i > 0) {
                        System.arraycopy(fontProviders, 0, newfp, 0, i);
                    }
                    if (i < fontProviders.length - 1) {
                        System.arraycopy(fontProviders, i + 1, newfp, i,
                            fontProviders.length - i);
                    }
                }
            }
        }
    }

    // ---------- utility methods

    /**
     * Returns a FontHelper instance. If the FontHelper has been
     * activated in the OSGi framework, the activated and configured instance is
     * returned. Otherwise an unconfigured throw-away instance is returned.
     * <p>
     * The instance returned must be kept in a long-term field because future
     * calls to this method may return different instances.
     */
    public static FontHelper getInstance() {
        if (INSTANCE == null) {
            return new FontHelper();
        }
        return INSTANCE;
    }

    /**
     * Returns the ContentBus font path configured in the <code>fontpath</code>
     * element of the configuration file. The entries in the list are guaranteed
     * to not end with a slash.
     *
     * @return The ContentBus font path.
     * @throws ServiceException if the <code>FontHelper</code> is not
     *             started.
     */
    String[] getFontPath() {
        return fontPath;
    }

    /**
     * Returns the list of the {@link FontProvider} implementations configured
     * in the <code>fontproviders</code> element of the configuration file.
     *
     * @return The {@link FontProvider} implementations
     * @throws ServiceException if the <code>FontHelper</code> is not
     *             started.
     */
    FontProvider[] getFontProviders() {
        return fontProviders;
    }

    /**
     * Returns the list of available fonts on the platform. For each font found
     * its family name (e.g. Helvetica), point size (??) and style (e.g.
     * Font.BOLD) is returned. Note: We only get the font family names and
     * define all fonts to be available for all styles, as Java2D will create
     * missing styles on the fly.
     *
     * @return List of fonts on the platform
     */
    /**
     * Returns the list of available fonts known to all registered font
     * providers and the <code>com.day.image.Font</code> class.
     *
     * @return List of fonts on the platform
     */
    public List<FontListEntry> getFontList() {
        List<FontListEntry> fontList = new ArrayList<FontListEntry>();

        // font provider entries
        for (FontProvider fontProvider : getFontProviders()) {
            fontList.addAll(fontProvider.getFontList());
        }

        // Java platform fonts
        GraphicsEnvironment env = GraphicsEnvironment.getLocalGraphicsEnvironment();
        for (String platFont : env.getAvailableFontFamilyNames()) {
            fontList.add(new FontListEntry("[Java Platform]", platFont, 0, 0xff));
        }

        return fontList;
    }

    public AbstractFont getFont(String faceName, int size, int style) {

        // we should select one of the registered font types and fall back to
        // the com.day.image.Font class.
        AbstractFont fnt = null;
        for (int i = 0; i < fontProviders.length; i++) {
            log.debug("Font: Asking provider {}", fontProviders[i]);
            fnt = fontProviders[i].getFont(faceName, size, style);
            if (fnt != null) {
                log.debug("Font: Using font {}", fnt);
                break;
            }
        }

        // did we find a provider's font ?
        if (fnt == null) {
            log.debug("Font: No provider has font, fallback to base class");
            fnt = new PlatformFont(faceName, size, style);
        }

        return fnt;
    }

    // ---------- internal

    /**
     * Initializes the system properties specifying the system font path to use
     * and whether to use platform specific fonts.
     */
    private void initFontPath(String[] fontPathConfig) {

        if (fontPathConfig != null && fontPathConfig.length > 0) {

            // validate the font patch config -- non empty and absolute entries
            List<String> fontPathList = new ArrayList<String>();
            for (String fontPathEntry : fontPathConfig) {
                if (fontPathEntry == null || fontPathEntry.length() == 0) {
                    log.info("initFontPath: Ignoring empty font path entry");
                } else if (fontPathEntry.charAt(0) != '/') {
                    log.info(
                        "initFontPath: Ignoring non-absolute font path entry {}",
                        fontPathEntry);
                } else {
                    fontPathList.add(fontPathEntry);
                }
            }

            // get the path entries
            fontPath = fontPathList.toArray(new String[fontPathList.size()]);

            // log the settings
            if (log.isDebugEnabled()) {
                log.debug("Communique font settings:");
                for (String fontPathEntry : fontPath) {
                    log.debug("  Path entry {}", fontPathEntry);
                }
            }

        } else {

            log.debug("initFontPath: Font path not configured");
            fontPath = new String[0];

        }
    }
}