/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image.internal.font;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Stroke;
import java.awt.font.FontRenderContext;
import java.awt.font.GlyphMetrics;
import java.awt.font.GlyphVector;
import java.awt.font.LineMetrics;
import java.awt.font.TextAttribute;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BandCombineOp;
import java.awt.image.WritableRaster;
import java.text.BreakIterator;
import java.text.CharacterIterator;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import com.day.image.Layer;
import com.day.image.font.AbstractFont;


/**
 * The <code>Font</code> class is the Java reimplementation of the former ECMA
 * host object <code>Font</code>. In addition to the original Font object
 * this class also contains the drawText() method, to keep it with the
 * getTextExtent() and as it is customary that an object draws itself somewhere.
 * <p>
 * This class internally encapsulates a <code>java.awt.Font</code> object
 * which is retrieved by the {@link #findFont} method. Extensions of this class
 * may override the {@link #findFont} method to provide other mechanisms to load
 * a <code>java.awt.Font</code> object than by calling the
 * <code>java.awt.Font</code> constructor.
 * <p>
 * <em>Note:</em> The <code>findFont</code> method is specified to allow
 * returning <code>null</code> instead of a valid <code>java.awt.Font</code>
 * object. In this case the extension class MUST provide overwritten
 * {@link #getTextExtent}, {@link #drawText}, and {@link #getHeight} methods.
 * These methods in this class depend on the {@link #findFont} method to return
 * a valid <code>java.awt.Font</code> object.
 * <p>
 * Examples of extensions of this class and how to handle overwritings :
 * <ol>
 * <li>Loading TrueType fonts from the network - Simply overwrite the
 * {@link #findFont} method to provide another way to generate the
 * <code>java.awt.Font</code> object using the
 * <code>java.awt.Font.createFont</code> method.
 * <li>Supporting new font formates - Overwrite the {@link #findFont} method to
 * return <code>null</code> and overwrite the {@link #getTextExtent},
 * {@link #drawText}, and {@link #getHeight} to provide font format specific
 * implementations.
 * </ol>
 *
 * @version $Revision: 39767 $, $Date: 2008-08-20 15:37:26 +0200 (Mi, 20 Aug
 *          2008) $
 * @author fmeschbe
 * @since coati
 * @audience wad
 */
public class PlatformFont extends AbstractFont {

    // ---------- private constants

    /** The Graphics object is needed to get at the font metrices. */
    private static final Graphics DUMMY_GRAPHICS = new Layer(1, 1, null).getG2();

    // ---------- fields

    /**
     * The oversampling factor used when drawing text with the
     * {@link #TTOVERSAMPLING} flag set (default value is 16).
     *
     * @see #TTOVERSAMPLING
     * @see #getOversamplingFactor()
     * @see #setOversamplingFactor(int)
     * @see #drawText(Layer, int, int, int, int, String, Paint, Stroke, int,
     *      double, int)
     */
    private static int oversamplingFactor = 16;

    /** Keep a reference to the orginal font family name */
    private final String family;

    /** This is the size mapped according to the {@link #TTFONTERSCALE} flag */
    private final int size;

    /** Keep a reference to the original font style requested */
    private final int style;

    /** This is the font we are using */
    private final java.awt.Font font;

    /** The metrics of the font */
    private final FontMetrics metrics;

    /**
     * Ascender value of the font. Depending on the {@link #TTFONTERLINESPACING}
     * flag this value is the normal ascender of the font or the max. ascender.
     */
    private final double ascent;

    /**
     * Descender value of the font. Depending on the
     * {@link #TTFONTERLINESPACING} flag this value is the normal descender of
     * the font or the max. descender.
     */
    private final double descent;

    /**
     * Line height of the font. This is the sum of {@link #ascent},
     * {@link #descent} and the font leading.
     */
    private final double height;

    // ---------- constructors

    /**
     * Creates and new instance for the given font facename, size and style. The
     * style flags are masked to the known style flags of the
     * <code>java.awt.Font</code> class.
     *
     * @param faceName Name of the font as known to the user. This generally is
     *            not the same as the name of the font file. In fact this MUST
     *            be the name of the Font family. That is "Times New Roman" and
     *            not "Times New Roman Bold" is expected.
     * @param size Size in points for the font to open
     * @param style Style flags for the new font
     * @see "The <code>java.awt.Font</code> javadoc page for details of this
     *      call."
     */
    public PlatformFont(String faceName, int size, int style, Font awtFont) {

        // scale the size first (before accessing the font, see bug #19611)
        size = scaleFontSize(size, style);

        // load the font from the platform if not provided
        if (awtFont == null) {
            Float posture = (style & ITALIC) != 0
                    ? TextAttribute.POSTURE_OBLIQUE
                    : TextAttribute.POSTURE_REGULAR;
            Float weight = (style & BOLD) != 0
                    ? TextAttribute.WEIGHT_BOLD
                    : TextAttribute.WEIGHT_REGULAR;
            HashMap<TextAttribute, Object> map = new HashMap<TextAttribute, Object>();
            map.put(TextAttribute.FAMILY, faceName);
            map.put(TextAttribute.SIZE, new Float(size)); // auto-box is Integer
            map.put(TextAttribute.POSTURE, posture);
            map.put(TextAttribute.WEIGHT, weight);

            awtFont = new java.awt.Font(map);
        }

        // assign the basic font descriptor fields
        this.family = faceName;
        this.size = size;
        this.style = style;
        this.font = awtFont;

        // To get at the font metrics, we need a graphics object
        this.metrics = DUMMY_GRAPHICS.getFontMetrics(font);

        // line metrics calculation
        if ((style & TTFONTERLINESPACING) != 0) {
            ascent = metrics.getMaxAscent();
            descent = metrics.getMaxDescent();
            height = ascent + descent + metrics.getLeading();
        } else {
            ascent = metrics.getAscent();
            descent = metrics.getDescent();
            height = metrics.getHeight();
        }
    }

    /**
     * Creates and new instance for the given font facename, size and style. The
     * style flags are masked to the known style flags of the
     * <code>java.awt.Font</code> class.
     *
     * @param faceName Name of the font as known to the user. This generally is
     *            not the same as the name of the font file. In fact this MUST
     *            be the name of the Font family. That is "Times New Roman" and
     *            not "Times New Roman Bold" is expected.
     * @param size Size in points for the font to open
     * @param style Style flags for the new font
     * @see "The <code>java.awt.Font</code> javadoc page for details of this
     *      call."
     */
    public PlatformFont(String faceName, int size, int style) {
        this(faceName, size, style, null);
    }

    /**
     * Open the font with the given facename, size and style flags. The style
     * flags are masked to the known style flags of the
     * <code>java.awt.Font</code> class. This constructor call opens the font
     * with the default font style <code>PLAIN</code>.
     *
     * @param faceName Name of the font as known to the user. This generally is
     *            not the same as the name of the font file.
     * @param size Size in points for the font to open
     * @see "The <code>java.awt.Font</code> javadoc page for details of this
     *      call."
     */
    public PlatformFont(String faceName, int size) {
        this(faceName, size, PlatformFont.PLAIN);
    }

    // ---------- methods

    @Override
    public Font getAwtFont() {
        return font;
    }

    /**
     * Calculate the bounding box of the text, if it would be rendered with the
     * rendering attributes given. The calculation will be done as if the text
     * would be rendered in the current font object.
     * <p>
     * If the width of the text box is set to some value other than zero, the
     * text is broken to fit lines of the given length. The line breaking
     * algorithm breaking algorithm breaks on whitespace (blank, tab), carriage
     * return and linefeed (CR/LF) in any combination as well as on other
     * characters such as hyphens.
     * <p>
     * If the text contains carriage return and/or linefeed characters, the text
     * is broken into several lines, regardless of whether the text would be
     * broken because of the text box width setting.
     *
     * @param x left edge of the text box, required
     * @param y top edge of the text box, required
     * @param width maximum width of the textbox. If 0 (or negative) the width
     *            of the bounding box is dependent of the rendering attributes
     * @param height maximum height of the textbox. If 0 (or negative) the width
     *            of the bounding box is dependent of the rendering attributes
     * @param text the text string to calculate the bounding box for
     * @param align alignment, rotation and TrueType attributes for the text.
     *            Use {@link #ALIGN_LEFT} as default value.
     * @param cs extra intercharacter spacing. Use 0 as default value to not add
     *            additional space.
     * @param ls extra line spacing. Use 0 as default value to not add
     *            additional space.
     * @return returns a rectangle representing the bounding box, if the text
     *         would be rendered in this font using the given additional
     *         rendering attributes.
     */
    public Rectangle2D getTextExtent(int x, int y, int width, int height,
            String text, int align, double cs, int ls) {

        // Don't do anything, if don't have anything...
        if (text == null || text.length() == 0) {
            return new Rectangle(x, y, 0, 0);
        }

        TextBlock tb = new TextBlock(this, text, width, height, align, cs, ls);
        return tb.getAlignedBoundingBox(x, y);
    }

    /**
     * Render the given text string in the given font to the <code>Layer</code>
     * using the attributes given. Use the default values given for unspecified
     * values.
     * <p>
     * If the width of the text box is set to some value other than zero, the
     * text is broken to fit lines of the given length. The line breaking
     * algorithm breaking algorithm breaks on whitespace (blank, tab), carriage
     * return and linefeed (CR/LF) in any combination as well as on other
     * characters such as hyphens.
     * <p>
     * If the text contains carriage return and/or linefeed characters, the text
     * is broken into several lines, regardless of whether the text would be
     * broken because of the text box width setting.
     *
     * @param layer the layer to draw the text into
     * @param x left edge of the text box, required
     * @param y top edge of the text box, required
     * @param width maximum width of the textbox. If 0 (or negative) the width
     *            of the bounding box is dependent of the rendering attributes
     * @param height maximum height of the textbox. If 0 (or negative) the width
     *            of the bounding box is dependent of the rendering attributes
     * @param text the text string to calculate the bounding box for
     * @param paint The <code>Paint</code> to use for the text drawing.
     * @param stroke The <code>Stroke</code> to use for the text drawing.
     * @param align alignment, rotation and TrueType attributes for the text.
     *            Use {@link Font#ALIGN_LEFT} as default value.
     * @param cs extra intercharacter spacing. Use 0 as default value to not add
     *            additional space.
     * @param ls extra line spacing. Use 0 as default value to not add
     *            additional space.
     * @return the number of text lines drawn to the layer
     */
    public int drawText(Layer layer, int x, int y, int width, int height,
            String text, Paint paint, Stroke stroke, int align, double cs,
            int ls) {

        // Don't do anything, if don't have anything...
        if (text == null || text.length() == 0) {

            /** FIXME really throw on empty text ? */

            throw new IllegalArgumentException("text must not be empty");
        }

        // Create the textblock
        TextBlock tb = new TextBlock(this, text, width, height, align, cs, ls);

        // return the lines drawn into the layer
        return tb.draw(layer, x, y, paint, stroke);
    }

    /**
     * Returns the calculated font height in pixels.
     *
     * @return the calculated font height in pixels.
     */
    public double getHeight() {
        return height;
    }

    /**
     * Returns the ascent of this <code>Font</code> object, which is the
     * maximal value any glyph of this font ascends above the base line.
     *
     * @return The ascent of this <code>Font</code>.
     */
    public double getAscent() {
        return ascent;
    }

    /**
     * Returns the descent of this <code>Font</code> object, which is the
     * maximal value any glyph of this font descends below the base line.
     *
     * @return The ascent of this <code>Font</code>.
     */
    public double getDescent() {
        return descent;
    }

    /**
     * Checks if this Font has a glyph for the specified character.
     * <p>
     * If this class is overwritten without setting the AWT font field, this
     * method always returns false. Overwriting classes should overwrite this
     * method providing correct implementation.
     *
     * @param c a unicode character code
     * @return <code>true</code> if this Font can display the character;
     *         <code>false</code> otherwise.
     */
    public boolean canDisplay(char c) {
        return getAwtFont().canDisplay(c);
    }

    /**
     * Indicates whether or not this Font can display a specified String. For
     * strings with Unicode encoding, it is important to know if a particular
     * font can display the string. This method returns an offset into the
     * String str which is the first character this Font cannot display without
     * using the missing glyph code. If the Font can display all characters, -1
     * is returned.
     * <p>
     * If this class is overwritten without setting the AWT font field, this
     * method always returns 0. Overwriting classes should overwrite this method
     * providing correct implementation.
     *
     * @param str a String object
     * @return an offset into str that points to the first character in str that
     *         this Font cannot display; or -1 if this Font can display all
     *         characters in str.
     */
    public int canDisplayUpTo(String str) {
        return getAwtFont().canDisplayUpTo(str);
    }

    /**
     * Indicates whether or not this Font can display the characters in the
     * specified text starting at start and ending at limit. This method is a
     * convenience overload.
     * <p>
     * If this class is overwritten without setting the AWT font field, this
     * method always returns 0. Overwriting classes should overwrite this method
     * providing correct implementation.
     *
     * @param text the specified array of characters
     * @param start the specified starting offset into the specified array of
     *            characters
     * @param limit the specified ending offset into the specified array of
     *            characters
     * @return an offset into text that points to the first character in text
     *         that this Font cannot display; or -1 if this Font can display all
     *         characters in text.
     */
    public int canDisplayUpTo(char[] text, int start, int limit) {
        return getAwtFont().canDisplayUpTo(text, start, limit);
    }

    /**
     * Indicates whether or not this Font can display the specified String. For
     * strings with Unicode encoding, it is important to know if a particular
     * font can display the string. This method returns an offset into the
     * String str which is the first character this Font cannot display without
     * using the missing glyph code . If this Font can display all characters,
     * -1 is returned.
     * <p>
     * If this class is overwritten without setting the AWT font field, this
     * method always returns 0. Overwriting classes should overwrite this method
     * providing correct implementation.
     *
     * @param iter a CharacterIterator object
     * @param start the specified starting offset into the specified array of
     *            characters
     * @param limit the specified ending offset into the specified array of
     *            characters
     * @return an offset into the String object that can be displayed by this
     *         Font.
     */
    public int canDisplayUpTo(CharacterIterator iter, int start, int limit) {
        return getAwtFont().canDisplayUpTo(iter, start, limit);
    }

    // ---------- Object overwrite

    /**
     * Returns a <code>String</code> representation of this object containing
     * the font name, the size and style flags.
     *
     * @return A <code>String</code> representation of this object.
     */
    public String toString() {
        return "Font[family=" + family + ", size=" + size + ", style=" + style
            + "]";
    }

    // ---------- TTFONTERSCALE support

    /**
     * Returns the scaled font size according to the
     * {@link AbstractFont#TTFONTERSCALE} style flag. If the flag is _not_ set,
     * the size is scaled to 4/3 of the requested size. Otherwise, if the flag
     * is set, the size is returned unmodified.
     *
     * @param size The requested size
     * @param style The font style flag with the optional
     *            <code>TTFONTERSCALE</code> flag set.
     * @return The optionally scaled size
     */
    public static int scaleFontSize(int size, int style) {
        if (((style & TTFONTERSCALE) == 0)) {
            return size * 4 / 3;
        }

        return size;
    }

    // ---------- Oversampling helper

    /**
     * Returns the factor applied internally to draw text in oversampled mode
     * when the {@link #TTOVERSAMPLING} flag is set on the
     * {@link #drawText(Layer, int, int, int, int, String, Paint, Stroke, int, double, int)}
     * method.
     */
    public static int getOversamplingFactor() {
        return oversamplingFactor;
    }

    /**
     * Sets the factor applied internally to draw text in oversampled mode when
     * the {@link #TTOVERSAMPLING} flag is set on the
     * {@link #drawText(Layer, int, int, int, int, String, Paint, Stroke, int, double, int)}
     * method.
     * <p>
     * While this method does not specify an upper bound for the validity of the
     * factor, you should keep in mind that (a) this factor is used to multiply
     * <code>int</code> values and (b) using bigger values cause more memory
     * being required and more time being spent scaling the result down.
     * <p>
     * By default the factor is set to <code>16</code>. Your mileage may vary
     * and you may test what best suits your needs, but it is expected that
     * values beyond 32 are more of a performance hit than they give higher
     * quality results.
     *
     * @param oversamplingFactor The new oversampling factor to use for
     *            oversampled text drawing. This value must be greater than
     *            zero, otherwise this method has no effect and the current
     *            setting is not replaced.
     */
    public static void setOversamplingFactor(int oversamplingFactor) {
        if (oversamplingFactor > 0) {
            PlatformFont.oversamplingFactor = oversamplingFactor;
        }
    }

    /**
     * Returns the given dimension scaled up by the oversampling factor.
     */
    private static double scaleUp(double value) {
        return oversamplingFactor * value;
    }

    /**
     * Returns the given dimension scaled down by the oversampling factor.
     */
    private static double scaleDown(double value) {
        // this does no rounding at all
        return value / oversamplingFactor;
    }

    /**
     * Returns the given dimension scaled up by the oversampling factor.
     */
    private static int scaleUp(int value) {
        return oversamplingFactor * value;
    }

    /**
     * Returns the given dimension scaled down by the oversampling factor. The
     * result is rounded up to the neirest integer.
     */
    private static int scaleDown(int value) {
        return (value + (oversamplingFactor / 2)) / oversamplingFactor;
    }

    // ---------- internal class

    /**
     * The <code>TextBlock</code> class is used to break up the text in words
     * and lines as well as to calculate the bounding box of the text, wrap
     * words as needed and position the graphical character representations
     *
     * @version $Revision: 39767 $, $Date: 2008-08-20 15:37:26 +0200 (Mi, 20 Aug
     *          2008) $
     * @author fmeschbe
     * @since coati
     * @audience core
     */
    private class TextBlock {

        /** The text to be drawn */
        private final String text;

        /**
         * The width of the text block of Integer.MAX_VALUE if undefined.
         * Horizontal alignment and wordwrapping only takes place if this value
         * is defined.
         */
        private final double width;

        /** The minimal height of the text box. */
        private final double height;

        /** The alignment flags as defined in the {@link Font} class */
        private final int flags;

        /** The additional spacing to insert between characters */
        private final double cs;

        /** The linespacing to use or the {@link Font#height} is undefined */
        private final double ls;

        /** The Java font object cached here */
        private final java.awt.Font awtFont;

        /** The anti-aliasing settings to draw the text */
        private final FontRenderContext frc;

        /** The offset from the baseline of the underline or strikethrough line */
        private final float strikeOffset;

        /**
         * The <code>Stroke</code> object used to draw the underline or
         * strikethrough line.
         */
        private final Stroke strikeStroke;

        /** The list of text lines, filled by {@link #breakText()} */
        private final TextLine[] lines;

        /** The raw bounding box of the text, filled by {@link #breakText()} */
        private Rectangle2D boundingBox;

        /**
         * Creates a new <code>TextBlock</code> object to draw the text using
         * the given attributes. If the text to be drawn takes more place, than
         * indicated by the width, the text will be broken on word boundaries
         * defined by space characters (<code>'&nbsp;'</code>). Also
         * horizontal alignment according to the alignment flags only takes
         * place if the width is defined.
         * <p>
         * If the text contains linefeeds (<code>\n</code>) lines will be
         * broken.
         *
         * @param font The font to use to draw the text
         * @param text The text to draw
         * @param width The intended with of the text. Lines longer than this
         *            width will be broken and aligned according to the
         *            alignment flags. If the width is negative or 0, no word
         *            wrapping and alignment takes place.
         * @param height The minimal height of the text box to draw. This value
         *            is used for vertical alignment.
         * @param flags The text drawing flags.
         * @param cs Additional character spacing.
         * @param ls Line spacing to use in 1/16th of a pixel. That is to define
         *            a line spacing of one pixel set this parameter to 16. If
         *            negative or null, the default font line spacing is used.
         */
        TextBlock(PlatformFont font, String text, double width, double height,
                int flags, double cs, double ls) {

            // fix ls setting
            ls = (ls > 0) ? ls / 16.0 : font.height;

            // get the correct font, optionally derive in size
            if ((flags & PlatformFont.TTOVERSAMPLING) != 0) {
                float size = scaleUp(font.getAwtFont().getSize());
                this.awtFont = font.getAwtFont().deriveFont(size);

                // scale additional spacing
                height = scaleUp(height);
                width = scaleUp(width);
                cs = scaleUp(cs);
                ls = scaleUp(ls);
            } else {
                this.awtFont = font.getAwtFont();
            }

            // primary values setting
            this.text = text;
            this.flags = flags;
            this.width = width;
            this.height = height;
            this.cs = cs;
            this.ls = ls;

            // defines the font render context
            boolean aalias = (flags & PlatformFont.TTANTIALIASED) != 0;
            this.frc = new FontRenderContext(null, aalias, false /* true */);

            // breaking the text
            this.lines = breakText();
            this.boundingBox = null;

            // underline and linethrough stuff
            LineMetrics lm = awtFont.getLineMetrics(" ", frc);
            if ((flags & PlatformFont.DRAW_STRIKEOUT) != 0
                || (font.style & PlatformFont.STRIKEOUT) != 0) {

                this.strikeOffset = lm.getStrikethroughOffset();

                // define the strike only if not outline
                this.strikeStroke = ((flags & PlatformFont.DRAW_OUTLINE) == 0)
                        ? new BasicStroke(lm.getStrikethroughThickness())
                        : null;

            } else if ((flags & PlatformFont.DRAW_UNDERLINE) != 0
                || (font.style & PlatformFont.UNDERLINE) != 0) {
                this.strikeOffset = lm.getUnderlineOffset();

                // define the strike only if not outline
                this.strikeStroke = ((flags & PlatformFont.DRAW_OUTLINE) == 0)
                        ? new BasicStroke(lm.getUnderlineThickness())
                        : null;

            } else {
                this.strikeOffset = Float.NaN;
                this.strikeStroke = null;
            }

        }

        /**
         * Returns the raw bounding box of the text. This bounding box the
         * minimal rectangular are used by the text, if it would be rendered
         * without any horizontal and/or vertical alignment at the origin.
         * <p>
         * This bounding box may be scaled by the
         * {@link Font#getOversamplingFactor() oversamplingFactor} if
         * oversampled text drawing is desired.
         *
         * @return the raw bounding box of the text.
         */
        Rectangle2D getRawBoundingBox() {
            assertBoundingBox();

            return new Rectangle2D.Double(boundingBox.getX(),
                boundingBox.getY(), boundingBox.getWidth(),
                boundingBox.getHeight());
        }

        /**
         * Returns the bounding box of the text defining the minimal rectangular
         * are used if the text would be drawn with vertical alignment applied
         * and drawn at the given position.
         * <p>
         * This bounding box may be scaled by the
         * {@link Font#getOversamplingFactor() oversamplingFactor} if
         * oversampled text drawing is desired.
         *
         * @param x The left edge of the text box area.
         * @param y The top edge of the text box area.
         */
        private Rectangle2D getAlignedBoundingBoxInternal(int x, int y) {
            assertBoundingBox();

            // we now have the overall bounding box aligned on the baseline of
            // the first line
            double vertDisp = 0;
            switch (flags & PlatformFont.ALIGN_VBASE) {
                case PlatformFont.ALIGN_TOP:
                    vertDisp = boundingBox.getMinY();
                    break;

                case PlatformFont.ALIGN_BASE:
                    // no displacement needed
                    break;

                case PlatformFont.ALIGN_BOTTOM:
                    vertDisp = boundingBox.getMaxY();
                    break;

                default:
                    break;
            }

            // displace by the alignment dependant displacement
            return new Rectangle2D.Double(boundingBox.getX() + x,
                boundingBox.getY() + y - vertDisp, boundingBox.getWidth(),
                boundingBox.getHeight());
        }

        /**
         * Returns the bounding box of the text defining the minimal rectangular
         * are used if the text would be drawn with vertical alignment applied
         * and drawn at the given position.
         *
         * @param x The left edge of the text box area.
         * @param y The top edge of the text box area.
         */
        Rectangle2D getAlignedBoundingBox(int x, int y) {
            Rectangle2D abox = getAlignedBoundingBoxInternal(0, 0);

            double l = abox.getX();
            double t = abox.getY();
            double w = abox.getWidth();
            double h = abox.getHeight();

            // resize if oversampling
            if ((flags & TTOVERSAMPLING) != 0) {
                l = scaleDown(l);
                t = scaleDown(t);
                w = scaleDown(w);
                h = scaleDown(h);
            }

            // redefine
            abox.setRect(l + x, t + y, w, h);

            // return the correct box
            return abox;
        }

        /**
         * Draws the text at the indicated position into the destination layer.
         * The paint parameter defines the paint to use for drawing the text and
         * the optional underline or strikethrough line. The stroke defines the
         * line stroke to use if the text should be drawn in outlined
         * characters. This stroke will then also be used for the optional
         * underline or strikethrough line. If the text should be drawn
         * normally, that is not as an outline, the stroke has no effect.
         *
         * @param layer The destination layer to draw the text into. If the text
         *            takes more space than the layer provides, the layer is
         *            enlarged so as to take the complete text. The text is not
         *            cropped to this layer.
         * @param x The left edge of the text box area.
         * @param y The top edge of the text box area.
         * @param paint The paint, for example a color or a
         *            <code>GradientPaint</code>, to use for drawing the text
         * @param stroke The stroke, for example a <code>BasicStroke</code>,
         *            to use for drawing the text.
         * @return The number of lines really drawn.
         */
        int draw(Layer layer, int x, int y, Paint paint, Stroke stroke) {

            // calculate bounding box, abort if empty
            assertBoundingBox();
            if (boundingBox.getWidth() <= 0 || boundingBox.getHeight() <= 0) {
                // nothing to be drawn, return immediately
                // note : might be negative due to Java2D glyph vectors
                return 0;
            }

            int linesDrawn = 0;
            Rectangle2D bbox = boundingBox;
            Rectangle2D abox = getAlignedBoundingBoxInternal(0, 0);

            // Calculate the size requirements of the text
            int textWidth = (int) Math.ceil(Math.max(width, bbox.getWidth()));
            int textHeight = (int) Math.ceil(Math.max(height, bbox.getHeight()));
            Layer drawer = new Layer(textWidth, textHeight, null);
            drawer.setX((int) abox.getX());
            drawer.setY((int) abox.getY());

            // antialiasing
            Object aaSetting = null;
            if ((flags & PlatformFont.TTANTIALIASED) != 0) {
                aaSetting = drawer.getRenderingHint(RenderingHints.KEY_ANTIALIASING);
                drawer.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                    RenderingHints.VALUE_ANTIALIAS_ON);
            }

            // Let us draw in the given color
            if (paint == null
                || (paint instanceof Color && ((Color) paint).getAlpha() == 255)) {
                /**
                 * Sun Java 1.3 has problems doing proper antialiasing according
                 * to the color if the color is not opaque black. Antialiasing
                 * is ok for not fully opaque colors with an alpha < 255, even
                 * an alpha of 254 works perfectly ! To fix, for fully opaque
                 * colors, we draw the text in black and later recolor the
                 * result according to the desired color.
                 */
                drawer.setPaint(Color.black);
            } else {
                /**
                 * No special treatments if the paint is not a color or if the
                 * color is not fully opaque color where alpha < 255.
                 */
                drawer.setPaint(paint);
                paint = null; // set to null to prevent recoloring
            }

            // and of course the stroke
            if (stroke != null) drawer.setStroke(stroke);

            // we need the graphics to draw the glyphs
            Graphics2D drawerG2 = drawer.getG2();

            double penY = -bbox.getY();

            for (int i = 0; i < lines.length; i++) {
                TextLine textLine = lines[i];
                String line = textLine.getLine();
                GlyphVector gv = textLine.getGlyphVector();

                double penX = -bbox.getX();
                double lw = textLine.getWidth(cs);

                // the border, to the right for ALIGN_LEFT
                double border = textWidth - lw;

                // right alignment
                if ((flags & PlatformFont.ALIGN_RIGHT) != 0) {

                    // place the border to the left
                    penX = border - gv.getGlyphMetrics(0).getLSB();
                    fixCharSpacing(gv, cs);

                    // center alignment
                } else if ((flags & PlatformFont.ALIGN_CENTER) != 0) {

                    // place half of the border to the left
                    penX = border / 2 - gv.getGlyphMetrics(0).getLSB();
                    fixCharSpacing(gv, cs);

                    // full justification
                } else if ((flags & PlatformFont.ALIGN_FULL) != 0) {

                    // word breaker
                    BreakIterator bi = BreakIterator.getLineInstance();
                    bi.setText(line);

                    // get the words
                    List<String> words = new ArrayList<String>();
                    int start = bi.first();
                    for (int end = bi.next(); end != BreakIterator.DONE; start = end, end = bi.next()) {

                        if (!Character.isWhitespace(line.charAt(start))) {
                            int wordEnd = fixLineEnd(line, end);

                            // the glyph vector for the word
                            words.add(line.substring(start, wordEnd));
                        }
                    }

                    // no words in the line
                    if (words.size() == 0) {
                        // null op

                        // one word - distribute characters by changing
                        // charSpacing
                    } else if (words.size() == 1) {

                        alignCharacters(gv, border);

                        // multiple words - distribute words by changing
                        // wordSpacing
                    } else {

                        // this draws the text, set gv to null to not draw again
                        alignWords(drawerG2, words, penX, penY, textWidth);
                        gv = null;

                    }

                    // for full justification, the line width is the text width
                    lw = textWidth;

                } else {

                    // default left alignment, place border to the right,
                    fixCharSpacing(gv, cs);

                }

                // draw glyph vector
                if (gv != null) {
                    drawGlyphVector(drawerG2, gv, (float) penX, (float) penY);
                }

                // strikethrough or underline
                if (!Float.isNaN(strikeOffset)) {

                    Stroke old = null;
                    if (strikeStroke != null) {
                        old = drawer.getStroke();
                        drawer.setStroke(strikeStroke);
                    }

                    double sy = penY + strikeOffset;
                    drawer.draw(new Line2D.Double(penX, sy, penX + lw, sy));

                    if (old != null) {
                        drawer.setStroke(old);
                    }

                }

                // increment the line counter
                linesDrawn++;

                // Advance by the line spacing
                penY += ls;
            }

            int deg = 0;
            if ((flags & ROT90) != 0) {
                deg = 90;
            } else if ((flags & ROT180) != 0) {
                deg = 180;
            } else if ((flags & ROT270) != 0) {
                deg = 270;
            }

            if (deg != 0) {
                drawer.rotate(deg);
                // we adjust position, if it should be centered
                if ((flags & ALIGN_CENTER) != 0 && (flags & ROTODD) != 0) {
                    drawer.setX(drawer.getX() + drawer.getHeight() / 2
                        - drawer.getWidth() / 2);
                }
            }

            // if oversampled, scale down
            if ((flags & PlatformFont.TTOVERSAMPLING) != 0) {
                // only get current setting if not yet retrieved
                if (aaSetting == null) {
                    aaSetting = drawer.getRenderingHint(RenderingHints.KEY_ANTIALIASING);
                }

                // force antialising for down scaling
                drawer.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                    RenderingHints.VALUE_ANTIALIAS_ON);

                // resize to correct size
                drawer.resize(scaleDown(drawer.getWidth()),
                    scaleDown(drawer.getHeight()));

                // translate the text according to the oversampling factor
                drawer.setX(scaleDown(drawer.getX()));
                drawer.setY(scaleDown(drawer.getY()));
            }

            // switch back the anti-aliasing setting
            if (aaSetting != null) {
                drawer.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                    aaSetting);
            }

            /**
             * Because of the antialiasing problems of Sun Java 1.3 with fully
             * opaque non-black colors, we have to recolor the result, which is
             * drawn in black. The paint is set to null above if the text has
             * already been drawn in the correct color.
             */
            if (paint != null) {
                // it must be a Color
                Color col = (Color) paint;
                float[][] elems = new float[][] { { 0, 0, 0, 0, col.getRed() },
                    { 0, 0, 0, 0, col.getGreen() },
                    { 0, 0, 0, 0, col.getBlue() }, { 0, 0, 0, 1, 0 } };
                WritableRaster imageRaster = drawer.getImage().getRaster();
                new BandCombineOp(elems, null).filter(imageRaster, imageRaster);
            }

            // translate the layer to the desired location
            drawer.setX(drawer.getX() + x);
            drawer.setY(drawer.getY() + y);

            // Merge the text layer onto the image layer
            layer.merge(drawer);

            return linesDrawn;
        }

        /**
         * Draws the <code>GlyphVector</code> at the given offset into the
         * layer according to the flags specifying whether only the outline
         * should be drawn.
         */
        private void drawGlyphVector(Graphics2D g2, GlyphVector gv, float xPos,
                float yPos) {

            if ((flags & PlatformFont.DRAW_OUTLINE) != 0) {
                g2.draw(gv.getOutline(xPos, yPos));
            } else {
                g2.drawGlyphVector(gv, xPos, yPos);
            }
        }

        /**
         * Rearranges the glyphs in the vector according to the optional
         * additional character spacing and the given line offset.
         *
         * @param gv The <code>GlyphVector</code> containing the glyphs whose
         *            positions are to be adapted.
         */
        private void fixCharSpacing(GlyphVector gv, double cs) {
            // adjust glyphs now
            double xOf = 0;
            for (int i = 0, ng = gv.getNumGlyphs(); i < ng; i++, xOf += cs) {
                Point2D pos = gv.getGlyphPosition(i);
                pos.setLocation(pos.getX() + xOf, pos.getY());
                gv.setGlyphPosition(i, pos);
            }
        }

        /**
         * Rearranges the glyphs consituting one single word for full jsutified
         * alignment by evenly distributing spacing around the characters.
         *
         * @param gv The <code>GlyphVector</code> containing the glyphs of the
         *            word to be evenly spaced.
         * @param space The white space available to distribute between the
         *            characters of the word.
         */
        private void alignCharacters(GlyphVector gv, double space) {
            fixCharSpacing(gv, cs + space / (gv.getNumGlyphs() - 1));
        }

        /**
         * Draws the words contained in the <code>words</code> list into the
         * {@link Layer} drawer distributing the words evenly on the line having
         * the given size.
         * <p>
         * Note that due to rounding errors, the text may not be drawn exactly
         * as expected. That is, some pixels might get lost.
         *
         * @param drawerG2 The <code>Graphics2D</code> into which to draw the
         *            text
         * @param words The <code>List</code> of words to be written
         * @param penX The x coordinate of the starting point to draw
         * @param penY The y coordinate of the base line of the starting point
         * @param textWidth The horizontal space to distribute the words in.
         */
        private void alignWords(Graphics2D drawerG2, List<String> words, double penX,
                double penY, int textWidth) {

            int nw = words.size();
            int lsbs[] = new int[nw]; // left side bearings of the words
            int widths[] = new int[nw]; // total width of each word
            int sumWidth = 0; // sum of the width of all words

            // first round: GlyphVectors for the words and charspacing
            GlyphVector[] gvs = new GlyphVector[nw];
            for (int i = 0; i < nw; i++) {

                // the glyph vector for the word
                String word = words.get(i);
                GlyphVector gv = awtFont.createGlyphVector(frc, word);

                // adapt positions for character spacing
                fixCharSpacing(gv, cs);

                // collect info on the glyphvector
                lsbs[i] = (int) Math.ceil(gv.getGlyphMetrics(0).getLSB());
                widths[i] = (int) Math.ceil(gv.getVisualBounds().getWidth());
                sumWidth += widths[i];

                // add the glyphvector to the array
                gvs[i] = gv;
            }

            int blanking = (int) Math.ceil(penX) + lsbs[0];
            int space = (textWidth - blanking - sumWidth) / (nw - 1);

            int xOfs = blanking;
            for (int i = 0; i < nw; i++) {

                // offset last word from right border
                if (i == nw - 1) {
                    xOfs = textWidth - widths[i];
                }

                // draw glyph vector [ shift xOfs by negative left side bearing
                // ]
                drawGlyphVector(drawerG2, gvs[i], xOfs - lsbs[i], (float) penY);

                // calculate next offset
                xOfs += widths[i] + space;
            }
        }

        /**
         * Breaks the text into lines according to the width specification of
         * the text block and returns a list of text strings.
         *
         * @return The <code>List</code> of text strings into which the text
         *         is broken.
         */
        private TextLine[] breakText() {

            // the result list returned at the end
            List<String> list = new LinkedList<String>();

            // get the master glyphvector
            GlyphVector gv = awtFont.createGlyphVector(frc, text);
            gv.performDefaultLayout();

            // get the iterator to break the text for the line
            BreakIterator bi = BreakIterator.getLineInstance();
            bi.setText(text);

            double maxWidth = (width > 0) ? width : Double.MAX_VALUE;
            int lineStartPos = 0; // start of the current line
            int lastBreakOp = 0; // last break oportunity
            int nextLineStart = bi.next(); // the next (first) break
            // oporutinity

            while (nextLineStart != BreakIterator.DONE) {

                // the total width of the line so far
                int lineEnd = fixLineEnd(text, nextLineStart);
                double pw = TextLine.getWidth(gv, lineStartPos, lineEnd, cs);

                // if we are longer than the width given or hava CR/LF
                if (pw > maxWidth) {

                    // we have earlier break oportuinities
                    if (lastBreakOp > lineStartPos) {

                        // break on the last opportunity
                        list.add(text.substring(lineStartPos, fixLineEnd(text,
                            lastBreakOp)));

                        // reset line start
                        lineStartPos = lastBreakOp;

                    } else {

                        // have to break the word !!!!
                        int i = lineStartPos;
                        do {
                            i++;
                            pw = TextLine.getWidth(gv, lineStartPos, i, cs);
                        } while (pw <= maxWidth);
                        // invariant : i denotes a string, once character too
                        // long

                        lineEnd = i - 1;

                        // if not a single character can be placed in the line
                        // breaking the text aborts and the rest of the string
                        // is ignored. see bug #11132.
                        if (lineEnd == lineStartPos) {
                            // prevent appending rest of data ("last chunk")
                            lineStartPos = Integer.MAX_VALUE;

                            // leave the loop and terminate breaking
                            break;
                        }

                        list.add(text.substring(lineStartPos, fixLineEnd(text,
                            lineEnd)));

                        lineStartPos = lineEnd;
                    }

                } else {

                    // check for force line break
                    char lastOnLine = text.charAt(nextLineStart - 1);
                    if (lastOnLine == '\r' || lastOnLine == '\n') {
                        if (lineEnd > lineStartPos) {
                            list.add(text.substring(lineStartPos, lineEnd));
                        } else {
                            list.add("");
                        }
                        lineStartPos = nextLineStart;
                    }

                    // keep an eye on last oportunity and try next
                    lastBreakOp = nextLineStart;
                    nextLineStart = bi.next();
                }
            }

            // last chunk
            int lastNonBlank = fixLineEnd(text, text.length());
            if (lineStartPos <= lastNonBlank) {
                list.add(text.substring(lineStartPos, lastNonBlank));
            }

            // BUG 22593: prevent issues later when rendering by ensuring the
            // lines list contains at least one entry, which is an empty string
            if (list.isEmpty()) {
                list.add("");
            }

            // convert to a TextLine array and return
            TextLine[] lines = new TextLine[list.size()];
            int i = 0;
            for (Iterator<String> it = list.iterator(); it.hasNext();) {
                String s = it.next();
                gv = awtFont.createGlyphVector(frc, s);
                lines[i++] = new TextLine(s, gv);
            }
            return lines;
        }

        /**
         * Asserts the calculated bounding box of the textblock. If the bounding
         * box is not yet calculated, this method simply returns, else the
         * bounding box based on the text broken into lines is calculated.
         * <p>
         * This bounding box may be scaled by the
         * {@link Font#getOversamplingFactor() oversamplingFactor} if
         * oversampled text drawing is desired.
         */
        private void assertBoundingBox() {
            if (boundingBox == null) {
                double yOffset = 0;
                for (int i = 0; i < lines.length; i++) {

                    // get the lines bounding box, fixed by the cs
                    Rectangle2D vb = lines[i].getGlyphVector().getVisualBounds();
                    vb.setRect(vb.getX(), vb.getY() + yOffset,
                        lines[i].getWidth(cs), vb.getHeight());

                    // assign or get union
                    boundingBox = (boundingBox == null)
                            ? vb
                            : boundingBox.createUnion(vb);

                    yOffset += ls;
                }
            }
        }

        /**
         * Checks the tentative end of the text. The input index
         * <code>lineEnd</code> designates the position of the first character
         * of the next text. The preceding character is assumed to be the last
         * valid character of this text. If the text ends with whitespace, the
         * index of the first whitespace character is returned, else the same
         * value is returned.
         *
         * @param text The <code>String</code> containing the complete text to
         *            be wrapped.
         * @param lineEnd The index of the first character of the next text.
         * @return The index of the first whitespace character at the end of
         *         this text or <code>lineEnd</code> if this text does not end
         *         with whitespace characters.
         */
        int fixLineEnd(String text, int lineEnd) {

            // find the last non-whitespace character preceding text[lineEnd]
            do {
                lineEnd--;
            } while (lineEnd >= 0
                && Character.isWhitespace(text.charAt(lineEnd)));
            // invariant lineEnd is the index of the last non-whitespace
            // character

            // return the first character following the last non-whitespace
            return lineEnd + 1;
        }

    }

    /**
     * The <code>TextLine</code> class is used to represent one line of text
     * as a <code>String</code> and as a <code>GlyphVector</code>. The text
     * and glyph vector are defined at construction time, while the width of the
     * glyphvector is calculated on the fly if needed.
     *
     * @version $Revision: 39767 $, $Date: 2008-08-20 15:37:26 +0200 (Mi, 20 Aug
     *          2008) $
     * @author fmeschbe
     * @since coati
     * @audience core
     */
    private static class TextLine {

        /** The text line <code>String</code> */
        private final String line;

        /** The text line <code>GlyphVector</code> */
        private GlyphVector glyphVector;

        /** The calculated width of the <code>GlyphVector</code> */
        private double width;

        /** The character spacing used to calculate the width */
        private double cs;

        /**
         * Creates an instance of a <code>TextLine</code> for the given text
         * <code>String</code> and <code>GlyphVector</code>.
         *
         * @param line The text <code>String</code>.
         * @param glyphVector The <code>GlyphVector</code> constructed from
         *            the text <code>String</code>.
         */
        TextLine(String line, GlyphVector glyphVector) {
            this.line = line;
            this.glyphVector = glyphVector;
            this.width = Double.NaN;
            this.cs = Double.NaN;
        }

        /**
         * Returns the text line <code>String</code>.
         *
         * @return the text line <code>String</code>.
         */
        public String getLine() {
            return line;
        }

        /**
         * Returns the text line <code>GlyphVector</code>.
         *
         * @return the text line <code>GlyphVector</code>.
         */
        GlyphVector getGlyphVector() {
            return glyphVector;
        }

        /**
         * Returns the width of the <code>GlyphVector</code> taking the
         * character spacing into account. The character spacing may be negative
         * to place the characters closer to each other.
         * <p>
         * This method caches the calcuated width of the
         * <code>GlyphVector</code> and only recalculates if the method is
         * called with a different character spacing value than was used for the
         * cached value.
         *
         * @param cs The character spacing value
         * @return The width of the <code>GlyphVector</code>.
         */
        double getWidth(double cs) {
            if (Double.isNaN(width) || this.cs != cs) {
                this.cs = cs;
                this.width = getWidth(glyphVector, 0,
                    glyphVector.getNumGlyphs(), cs);
            }
            return width;
        }

        /**
         * Returns the width of the section of the glyphvector taking into
         * account hangovers at the beginning and the end of the vector as well
         * as additional character spacing.
         *
         * @param gv The <code>GlyphVector</code> to be calculated
         * @param first The index of the first glyph in the section
         * @param end The index of the first glyph not in the section anymore.
         *            That is the section is the glyphs in the internvall
         *            <code>[first, end[</code>.
         * @param cs The additional character spacing to use to calculate the
         *            width. This value may be negative to place the characters
         *            closer to each other.
         * @return The width of the section of the glyphvector indicated.
         */
        static double getWidth(GlyphVector gv, int first, int end, double cs) {

            // guard against empty strings
            if (end <= first) {
                return 0;
            }

            // the width from the first to the last+1 character
            double w = gv.getGlyphPosition(end).getX()
                - gv.getGlyphPosition(first).getX();

            // additional character spacing
            int numGlyphs = end - first - 1;
            double sumcs = numGlyphs * cs;

            GlyphMetrics glyphMetricsLeft= gv.getGlyphMetrics(first);
            GlyphMetrics glyphMetricsRight = gv.getGlyphMetrics(end - 1);

            // the total width of the line so far
            // GRANITE-2340: glyphs of width zero have an LSB of -gv.getGlyphPosition(...).getX()
            return (glyphMetricsLeft.getBounds2D().getWidth() == 0 ? 0 : -glyphMetricsLeft.getLSB()) + w + sumcs
                - (glyphMetricsRight.getBounds2D().getWidth() == 0 ? 0 : glyphMetricsRight.getRSB());
        }
    }

}