/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image.internal.font.resource;

import java.awt.Font;
import java.awt.FontFormatException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.observation.Event;
import javax.jcr.observation.EventIterator;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import sun.security.krb5.internal.Ticket;

import com.day.image.font.AbstractFont;
import com.day.image.font.FontListEntry;
import com.day.image.internal.font.AbstractFontCache;
import com.day.image.internal.font.FontFileProvider;
import com.day.image.internal.font.PlatformFont;

/**
 * The <code>ResourceFontCache</code> class maintains a list of known font
 * files in each ContentBus location known to the class. This list is read when
 * the class is instantiated and kept up to date during the life time of system
 * with the help of the {@link com.day.cq.contentbus.service.ObservationService}.
 * This way, it is possible to add TrueType fonts during the run time of the
 * Communiqu� system and use these fonts immediately.
 * <p>
 * Additionally all fonts loaded during the life time are kept in an internal
 * cache, which is implemented such that the garbage collector might remove
 * fonts if memory is needed. If the fonts are later needed again, they will
 * simply be reloaded.
 *
 * @version $Revision: 20173 $, $Date: 2006-05-03 16:47:57 +0200 (Mit, 03 Mai
 *          2006) $
 * @author fmeschbe
 * @since degu
 * @audience core
 */
public class ResourceFontCache extends AbstractFontCache<Font> {

    private static final String CONTENT_DATA = "/jcr:data/jcr:content";

    private static final String NODE_TYPE_DEFAULT = "nt:unstructured";

    /**
     * Name of the resource property used to store the font family name in the
     * font map (value is "family").
     */
    private static final String PROP_FAMILY = "family";

    /**
     * Name of the resource property used to store the PostScript font name in
     * the font map (value is "psName").
     */
    private static final String PROP_PSNAME = "psName";

    /**
     * Name of the resource property used to store the absolute resource path to
     * the font in the resource tree (value is "family").
     */
    private static final String PROP_FONT_PATH = "fontPath";

    /**
     * Name of the resource property used to store the absolute path to the font
     * file in the filesystem for use with the platform (value is
     * "fontFilePath").
     */
    private static final String PROP_FONT_FILE_PATH = "fontFilePath";

    /** default log */
    private final Logger log = LoggerFactory.getLogger(getClass());

    /** The label of the font map page */
    private static final String FONT_MAP_RESOURCE_NAME = ".fontlist";

    /** The {@link Ticket} to maintain the TrueType font mapping */
    private final ResourceResolver systemTicket;

    /** Used to provide font files from input streams */
    private FontFileProvider fontFileProvider;

    /**
     * The {@link HandlePatternList} list containing the font path pattern
     * handles.
     */
    private final String[] fontPath;

    /**
     * The listener package used for the registration and by the modification
     * handler method to decide how to handle MOVE modifications.
     *
     * @see #setupCache
     */
    // private ContentPackage obsPackage;
    /**
     * The mapping of font names (key) to TrueType font handles (value).
     *
     * @see #setupCache
     */
    private Map<String, String> fontMap;

    /**
     * Creates an instance of the <code>ContentBusCache</code> class and
     * synchronizes the font mapping pages in all locations indicated by the
     * <code>fontPath</code> parameter. Also register the instance as the
     * modification listener for all pages ending with <em>.ttf</em> and
     * <em>.TTF</em> in the font path locations.
     *
     * @param systemTicket The {@link Ticket} used to access the ContentBus.
     * @param fontPath The {@link HandlePatternList} containing the handle globs
     *            defining the font path
     */
    ResourceFontCache(ResourceResolver systemTicket, String[] fontPath, FontFileProvider ffp) {
        // initialize the hash map
        super();

        // set the ticket
        this.systemTicket = systemTicket;
        this.fontPath = fontPath;
        this.fontFileProvider = ffp;

        // setup the cache
        setupCache();

        // register with the fontPath to get info on new/removed paths
        // this.fontPath.addListener(this);
    }

    // ---------- AbstractFontCache overwrites

    /**
     * Deregisters from the font path and the observation service if registered
     * and removes all entries from the font cache.
     */
    public void destroy() {
        // remove this from the handler pattern list
        // fontPath.removeListener(this);

        // destroy the base class
        super.destroy();
    }

    // --------- HandlePatternList.Listener

    /**
     * Resets the font map and re-registers with the observation service to get
     * notified on new/removed fonts in the new places.
     */
    public void handleListChanged() {
        setupCache();
    }

    // --------- local extensions

    /**
     * Creates a list of {@link Font.FontListEntry} elements representing the
     * fonts known in the font mapping.
     *
     * @return The list of known fonts.
     */
    synchronized List<FontListEntry> getFontList() {
        List<FontListEntry> fontList = new ArrayList<FontListEntry>();

        for (String name : fontMap.keySet()) {
            // the FonterFont style name, size always 0
            int zeroInd = name.indexOf('0'); // index of pseudo-size

            int style;
            if (zeroInd > 0) {
                // get the style flag from the name (after the zero)
                style = AbstractFont.stringToStyle(name.substring(zeroInd + 1));

                // get the name part
                name = name.substring(0, zeroInd);

                // if the style is 0 (PLAIN), everything is possible with this
                // entry
                if (style == PlatformFont.PLAIN) {
                    style = 0xff;
                }

            } else {
                // default if no size is all styles
                // note that this is an illegal state in the map, but handle
                // anyway
                style = 0xff;
            }

            // add a new entry to the list
            fontList.add(new FontListEntry("ResourceFont", name, 0, style));
        }
        return fontList;
    }

    /**
     * Returns the font handle to which the font name maps or <code>null</code>
     * if the font name does not correspond to a known font page.
     *
     * @param fontName The name of the font or which to return the font page
     *            handle.
     * @return The font page handle, if the font name has a known mapping, or
     *         <code>null</code> if no such mapping exists.
     */
    synchronized String getFontHandle(String fontName) {
        synchronized (fontMap) {
            return fontMap.get(fontName);
        }
    }

    // ---------- AbstractFontCache abstracts

    @Override
    public void onEvent(EventIterator events) {
        while (events.hasNext()) {
            Event event = events.nextEvent();

            String path;
            try {
                path = event.getPath();
            } catch (RepositoryException re) {
                log.info("onEvent: Cannot get path for event " + event, re);
                continue;
            }

            if (path.endsWith(CONTENT_DATA)) {

                // need the page containing the font
                String fontHandle = path.substring(0, path.length()
                    - CONTENT_DATA.length());
                String mapPageHandle = ResourceUtil.getParent(fontHandle);
                Resource fontMapPage = getFontMapPage(mapPageHandle);
                if (fontMapPage == null) {
                    log.debug(
                        "pageModified: No fontmap support for folder {}. "
                            + "Ignoring modification of {}", mapPageHandle,
                        fontHandle);
                    continue;
                }

                if (event.getType() == Event.PROPERTY_ADDED) {
                    addFont(fontMapPage, fontHandle);
                } else if (event.getType() == Event.PROPERTY_CHANGED) {
                    // simply remove and add again a modified font
                    removeFont(fontMapPage, fontHandle);
                    addFont(fontMapPage, fontHandle);
                } else if (event.getType() == Event.PROPERTY_REMOVED) {
                    removeFont(fontMapPage, fontHandle);
                }
            }
        }
    }

    // ---------- Font Map handling

    /**
     * Synchronizes the font mapping pages and loads the mappings into table
     * returned at the end. Missing font map pages are created, new fonts are
     * added to the relative font map page and missing fonts (presumably
     * deleted) are removed from the relative font map page.
     *
     * @param fontPath The list of folders containing TrueType fonts to
     *            consider.
     * @return The table containing all the known mappings from the folders.
     *         This <code>Map</code> uses the font names as the keys and the
     *         font page handles as the values.
     */
    private Map<String, String> syncFontMappings(String[] fontPath) {
        Map<String, String> fontMap = new HashMap<String, String>();
        for (String fontHandle : fontPath) {
            log.debug("syncFontMappings: Testing path entry {}", fontHandle);

            Resource fontMapPage = getFontMapPage(fontHandle);
            if (fontMapPage == null) {
                log.info("syncFontMappings: Missing .fontlist in {}."
                    + " Ignoring", fontHandle);
                continue;
            }
            Map<String, String> fileFontMap = getFileFontMap(fontMapPage);

            log.debug("syncFontMappings: Loaded {} entries from fontmap",
                String.valueOf(fileFontMap.size()));

            // check font files
            Resource parent = systemTicket.getResource(fontHandle);
            Iterator<Resource> pi = ResourceUtil.listChildren(parent);
            while (pi.hasNext()) {
                // ignore pages not having extens .ttf or .TTF
                Resource page = pi.next();
                String handle = page.getPath();

                log.debug("syncFontMappings: Testing page {}", handle);

                if (!handle.endsWith(".ttf") && !handle.endsWith(".TTF")) {
                    log.debug("syncFontMappings: {} not a TrueType page",
                        handle);
                    continue;
                }

                String hint = "Old";
                String fontName = fileFontMap.remove(handle);
                if (fontName == null) {
                    log.debug("syncFontMappings: Checking possible new entry");
                    fontName = addFont(fontMapPage, handle);
                    hint = "New";
                }
                if (fontName != null) {
                    log.debug("syncFontMappings: {} entry {} added", hint,
                        handle);
                    fontMap.put(fontName, handle);
                }
            }

            // font list has removed fonts, need to write the copy
            if (!fileFontMap.isEmpty()) {
                log.debug("syncFontMappings: Some fonts removed");
                for (String fontFileName : fileFontMap.values()) {
                    log.debug("syncFontMappings: Removing {} from font map",
                        fontFileName);
                    removeFontMapEntry(fontMapPage, fontFileName);
                }
                fileFontMap.clear();
            }

            // commit additions
            log.debug("syncFontMappings: Commit changes to the font map");
        }
        return fontMap;
    }

    /**
     * Adds the given handle pointing to a TrueType font page to the mapping of
     * the font map page. If any error occurrs extracting the font information
     * from the font page or if updating the mapping page fails, the mapping is
     * not updated and false is returned.
     * <p>
     * The handle assumed to have the <em>.ttf</em> or <em>.TTF</em>
     * extension but the page might still not contain a valid TrueType font
     * file. In this case no mapping is added of course.
     * <p>
     * This method starts a transaction on the map page if needed. It is the
     * duty of the caller of this method to commit or rollback this transaction.
     *
     * @param fontMapPage The {@link Page} containing the font mappings to which
     *            the font handle should be added.
     * @param fontHandle The handle of the (potential) TrueType font page.
     * @return The name of the font just added or <code>null</code> if adding
     *         the font fails.
     */
    private String addFont(Resource fontMapPage, String fontHandle) {

        log.debug("addFont: Checking page {}", fontHandle);
        File fontFile;
        try {
            Resource fontResource = systemTicket.getResource(fontHandle);
            final InputStream is = fontResource.adaptTo(InputStream.class);
            if(is == null) {
                throw new IOException("Resource does not adapt to InputStream:" + fontPath);
            }
            fontFile = fontFileProvider.getFileForStream(is);
            Font af = java.awt.Font.createFont(Font.TRUETYPE_FONT, fontFile);

            // this is a nasty way of getting the style flags
            // for the font, but af.getStyle() always returns 0 !
            int style = 0;
            String psName = af.getPSName();
            if (psName.indexOf("Bold") >= 0) style |= PlatformFont.BOLD;
            if (psName.indexOf("Italic") >= 0) style |= PlatformFont.ITALIC;

            String faceName = af.getFamily();

            String fontFileName = AbstractFont.createFontFileName(faceName, 0,
                style);
            log.info("addFont: Using {} as font {}", fontHandle, fontFileName);

            // should we rollback in case of problem in the next 4 lines ?
            addFontMapEntry(fontMapPage, fontFileName, faceName, psName,
                fontHandle, fontFile.getAbsolutePath());

            // fontMap is still null during initialization
            if (fontMap != null) {
                fontMap.put(fontFileName, fontHandle);
            }

            return fontFileName;

        } catch (IOException ioe) {
            log.error("addFont: Cannot access font page {}", fontHandle, ioe);
        } catch (FontFormatException ffe) {
            log.error("addFont: Cannot read font from {}", fontHandle, ffe);
        }

        return null;
    }

    /**
     * Removes the mapping for the font identified by the
     * <code>fontHandle</code> from the <code>fontMapPage</code>.
     * <p>
     * This method starts a transaction on the map page if needed. It is the
     * duty of the caller of this method to commit or rollback this transaction.
     *
     * @param fontMapPage The {@link Page} containing the font mappings from
     *            which the font handle should be removed.
     * @param fontHandle The handle of the TrueType font page, whose mapping is
     *            to be removed.
     * @return <code>true</code> if removing the mapping succeeded.
     */
    private boolean removeFont(Resource fontMapPage, String fontHandle) {
        // find the font file name
        String fontName = null;
        for (Map.Entry<String, String> entry : fontMap.entrySet()) {
            if (entry.getValue().equals(fontHandle)) {
                fontName = entry.getKey();
                break;
            }
        }

        // if found, remove from map and the page
        if (fontName != null) {
            log.info("removeFont: Removing {} from font map", fontHandle);

            fontMap.remove(fontName);
            removeFontMapEntry(fontMapPage, fontName);
        } else {
            log.info("removeFont: No mapping for {} found.", fontHandle);
        }

        return true;
    }

    // ----------- accessing the font map page

    /**
     * Loads the font mapping from the given page into a <code>Map</code>
     * using the font page handles as the keys and the font names as the values.
     * Returns <code>null</code> if an error occurrs, see log for details in
     * this case.
     * <p>
     *
     * <pre>
     *   .fontmap
     *       +--- &lt;fontname&gt;
     *                +--- Family
     *                +--- PSName
     *                +--- FontPath
     *                +--- FontFile
     * </pre>
     *
     * @param fontMap The {@link Page} containing the font mapping
     * @return A <code>Map</code> containing the mapping from the mapping
     *         page, where the font page handles are used as keys and the font
     *         names as values, or <code>null</code> if a problem occurrs
     *         reading the mapping page.
     */
    private Map<String, String> getFileFontMap(Resource fontMap) {
        log.debug("getFileFontMap: Loading map from {}", fontMap.getPath());

        Map<String, String> fileFontMap = new HashMap<String, String>();

        Iterator<Resource> fi = ResourceUtil.listChildren(fontMap);
        while (fi.hasNext()) {
            Resource fontMapEntry = fi.next();
            Map<?, ?> entry = fontMapEntry.adaptTo(Map.class);
            Object handleObject = entry.get(PROP_FONT_PATH);
            if (handleObject != null) {
                String fontName = ResourceUtil.getName(fontMapEntry);
                log.debug("getFileFontMap: Adding {} => {}", fontName,
                    handleObject);
                fileFontMap.put(handleObject.toString(), fontName);
            }
        }

        return fileFontMap;
    }

    /**
     * Returns the font map <code>Page</code> object for the given folder. If
     * the page does not exist, it is created, if the folder is part of the font
     * path list.
     *
     * @param path The folder for which to get the map <code>Page</code>
     *            object.
     * @return The map <code>Page</code>, which might have been just created
     *         if the path is part of the font path and the page did not already
     *         exist.
     * @throws ContentBusException If a problem occurrs accessing the
     *             ContentBus.
     */
    private Resource getFontMapPage(String path) {

        // build the name of the font list page
        String pageName = path + "/" + FONT_MAP_RESOURCE_NAME;
        log.debug("getFontMapPage: fontlist Page is {}", pageName);

        Resource fontMapResource = systemTicket.getResource(pageName);
        if (fontMapResource == null) {
            // fontListPageName is not contained in the package !!!
            if (true /* obsPackage.contains(path + "/dummy.ttf") */) {
                log.info("getFontMapPage: Creating fontlist page");
                return addFontMapResource(path);
            }

            log.info("getFontMapPage: {} is not a TrueType font path,"
                + " no map needed", path);
            return null;
        }

        log.debug("getFontMapPage: Returning existing page");
        return fontMapResource;
    }

    /**
     * Sets up the <code>ResourceFontCache</code> defining the font map.
     */
    private void setupCache() {
        // get the expanded font path handles
        // ArrayList pathHandles = fontPath.getExpandedHandles(systemTicket);

        // load and synchronize the font page mappings
        this.fontMap = syncFontMappings(fontPath);
    }

    private Resource addFontMapResource(String location) {
        Session session = systemTicket.adaptTo(Session.class);
        if (session != null) {
            try {
                Node locationNode = getOrCreateNode(session, location);
                Node fontMapNode = locationNode.addNode(FONT_MAP_RESOURCE_NAME,
                    NODE_TYPE_DEFAULT);

                // persist all changes
                session.save();

                return systemTicket.getResource(fontMapNode.getPath());

            } catch (RepositoryException re) {
                log.error("addFontMapResource: Cannot create font map", re);
            } finally {
                try {
                    if (session.hasPendingChanges()) {
                        session.refresh(false);
                    }
                } catch (RepositoryException re2) {
                    log.warn(
                        "addFontMapResource: Failed reverting failed changes",
                        re2);
                }
            }
        }

        // fall back to not possible in case of trouble
        return null;
    }

    private void addFontMapEntry(Resource fontMapPage, String fontFileName,
            String faceName, String psName, String fontPath, String fontFilePath) {
        // Currently we only support storing the map in the repository
        Node fontMapNode = fontMapPage.adaptTo(Node.class);
        if (fontMapNode != null) {
            try {
                Node entryNode = fontMapNode.addNode(fontFileName,
                    NODE_TYPE_DEFAULT);

                entryNode.setProperty(PROP_FAMILY, faceName);
                entryNode.setProperty(PROP_PSNAME, psName);
                entryNode.setProperty(PROP_FONT_PATH, fontPath);
                entryNode.setProperty(PROP_FONT_FILE_PATH, fontFilePath);

                fontMapNode.save();
            } catch (RepositoryException re) {
                log.error("createFontMapEntry: Cannot store the entry for "
                    + fontFileName, re);
            } finally {
                if (fontMapNode.isModified()) {
                    try {
                        fontMapNode.refresh(false);
                    } catch (RepositoryException re2) {
                        log.error(
                            "createFontMapEntry: Cannot revert failed changes for "
                                + fontFileName, re2);
                    }
                }
            }
        }
    }

    private void removeFontMapEntry(Resource fontMapPage, String fontFileName) {
        // Currently we only support storing the map in the repository
        Node fontMapNode = fontMapPage.adaptTo(Node.class);
        if (fontMapNode != null) {
            try {
                if (fontMapNode.hasNode(fontFileName)) {
                    fontMapNode.getNode(fontFileName).remove();
                }
                fontMapNode.save();
            } catch (RepositoryException re) {
                log.error("removeFontMapEntry: Cannot remove the entry for "
                    + fontFileName, re);
            } finally {
                if (fontMapNode.isModified()) {
                    try {
                        fontMapNode.refresh(false);
                    } catch (RepositoryException re2) {
                        log.error(
                            "removeFontMapEntry: Cannot revert failed changes for "
                                + fontFileName, re2);
                    }
                }
            }
        }
    }

    /**
     * Returns the node at the requested <code>path</code> or creates it along
     * with any required intermediate parent nodes.
     * <p>
     * This method creates required nodes using the <i>sling:Folder</i> node
     * type, which is an <i>nt:folder</i> but allowing for any children and any
     * properties.
     * <p>
     * If one or more nodes need to be created to return the requested node, the
     * <code>session</code> will have changes when returning. To persist these
     * changes, the session must be saved by the caller.
     *
     * @param session The JCR Session used to access the nodes
     * @param path The absolute path to the requested node
     * @return The node at the requested path.
     * @throws RepositoryException
     */
    private Node getOrCreateNode(Session session, String path)
            throws RepositoryException {
        // check existence first and short cut
        if (session.itemExists(path)) {
            return (Node) session.getItem(path);
        }

        Node parent = getOrCreateNode(session, ResourceUtil.getParent(path));
        return parent.addNode(ResourceUtil.getName(path), "sling:Folder");
    }
}
