/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image.internal.font.resource;

import java.awt.Font;
import java.awt.FontFormatException;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import javax.jcr.observation.EventIterator;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.image.font.AbstractFont;
import com.day.image.font.FontListEntry;
import com.day.image.internal.FontPath;
import com.day.image.internal.font.FontFileProvider;
import com.day.image.internal.font.FontProvider;
import com.day.image.internal.font.PlatformFont;

/**
 * The <code>ResourceFontProvider</code> class implements the
 * {@link FontProvider} interface to create instance of the
 * {@link ContentBusFont} class.
 *
 * @version $Revision: 20173 $, $Date: 2006-05-03 16:47:57 +0200 (Mit, 03 Mai
 *          2006) $
 * @author fmeschbe
 * @since degu
 * @audience wad
 */
public class ResourceFontProvider implements FontProvider {

    /** default log */
    private final Logger log = LoggerFactory.getLogger(getClass());

    /** The font cache used to access the font instances themselves */
    private ResourceFontCache fontCache;

    /** The ticket to access the ContentBus */
    private ResourceResolver ticket;

    /** Provides font files from InputStreams */
    private FontFileProvider fontFileProvider;

    /**
     * Returns the list of available fonts on the platform. For each font found
     * its family name (e.g. Helvetica), point size (??) and style (e.g.
     * Font.BOLD) is returned. Note: We only get the font family names and
     * defined all fonts to be available all styles, as Java2D will create
     * missing styles on the fly.
     *
     * @return List of fonts on the platform
     */
    public List<FontListEntry> getFontList() {
        return fontCache.getFontList();
    }

    /**
     * Tries to load a font with the given face name, size and style flags. This
     * provider returns <code>null</code> if the {@link ContentBusFont} class
     * cannot load the exact TrueType font or derive it from another style for
     * the same font family.
     *
     * @param faceName Name of the font as known to the user. This generally is
     *            not the same as the name of the font file. In fact this MUST
     *            be the name of the Font family. That is "Times New Roman" and
     *            not "Times New Roman Bold" is expected.
     * @param requestedSize Size in points for the font to open
     * @param style Style flags for the new font
     * @return The {@link Font} instance according to the given properties or
     *         <code>null</code> if the {@link ContentBusFont} cannot load the
     *         exact font or derive it from another font of the same family.
     */
    public AbstractFont getFont(String faceName, int requestedSize, int style) {

        Font awtFont;
        synchronized (fontCache) {

            // we need the optionally scaled size for further works on fonts
            int fontSize = PlatformFont.scaleFontSize(requestedSize, style);

            // try to get the correct font (name, size, style) from the cache
            String fontName = AbstractFont.createFontFileName(faceName,
                fontSize, style);
            log.debug("findFont: Try {} from cache", fontName);
            awtFont = fontCache.get(fontName);

            // if not currently in the cache
            if (awtFont == null) {
                // load the font from the ContentBus
                log.debug("findFont: Derive size from {}/{}", faceName,
                    AbstractFont.styleToDescription(style));
                awtFont = loadBaseFont(faceName, style);

                if (awtFont != null) {
                   // size (and possibly style) do(es) not match yet, derive
                    awtFont = awtFont.deriveFont((float) fontSize);

                    // and store in the cache for later use
                    log.debug("findFont: Caching new instance for {}", fontName);
                    fontCache.put(fontName, awtFont);
                }
            }

            if (awtFont != null) {
                // call the constructor with the correctly sized font
                // but with the originally requested size
                return new PlatformFont(faceName, requestedSize, style, awtFont);
            }
        }

        // no Resource based font
        return null;
    }

    // ---------- SCR integration

    public void init(ResourceResolver resolver, String[] fontPath, FontFileProvider ffp) {
        this.ticket = resolver;
        this.fontFileProvider = ffp;

        FontPath fp = new FontPath(ticket, fontPath);

        // sets up the cache from the font path of the FontHelper
        fontCache = new ResourceFontCache(ticket, fontPath, ffp);
    }

    public void onEvent(EventIterator events) {
    }

    public void destroy() {
        if (fontCache != null) {
            fontCache.destroy();
            fontCache = null;
        }
    }

    // ---------- internal

    /**
     * Returns a font instance, which is loaded from the ContentBus. If the a
     * font page for the exact name and style exists, that page is used. If no
     * such page exists, a font is loaded with plain style and the same name
     * from which the desired style is derived. Of course the font from which
     * the font returned has been derived is stored in the cache, too.
     * <p>
     * The font returned from this method is already stored in the cache.
     *
     * @param family The font family name of the font to load.
     * @param style The style flags for the font to open.
     * @return The <code>java.awt.Font</code> object to derive the font to be
     *         used ultimately. If an error occurrs loading the font or if
     *         no font can be found, <code>null</code> is returned.
     */
    private Font loadBaseFont(String family, int style) {

        String baseFont = null;

        // try to get the font for the exact name and style from the cache
        String fontName = AbstractFont.createFontFileName(family, 0, style);
        log.debug("loadBaseFont: Try {} from cache", fontName);
        Font awtFont = fontCache.get(fontName);

        // if it exists already return it
        if (awtFont != null) {
            log.debug("loadBaseFont: Got it");
            return awtFont;
        }

        // else check, whether a mapping for that font/style exists
        String handle = fontCache.getFontHandle(fontName);
        if (handle == null) {
            // no mapping for exact font name and style exists
            log.debug("loadBaseFont: Derive style for {}/{} from plain {}",
                family, AbstractFont.styleToDescription(style));

            // try to get the font page handle for the exact name in plain style
            baseFont = AbstractFont.createFontFileName(family, 0, 0);
            awtFont = fontCache.get(baseFont);

            // if such a font exists in the cache already, return it
            if (awtFont != null) {
                // derive desired style and store in cache
                log.debug("loadBaseFont: Caching and returning derived font style");
                awtFont = awtFont.deriveFont(style);
                fontCache.put(fontName, awtFont);

                return awtFont;
            }

            // not even a plain font is in the cache, check mapping
            log.debug("loadBaseFont: Check mapping for plain {}", family);
            handle = fontCache.getFontHandle(baseFont);
            if (handle == null) {
                // there is no font to derive style from, fail
                log.debug("loadBaseFont: Not even found mapping for plain {}",
                    family);
                return null;
            }

        }
        // invariant : handle is either the handle of the page with exact name
        // and style or with the name but plain style, in which case
        // baseFont contains the font name of this latter font.

        try {
            // laod the TrueType font from the page
            log.debug("loadBaseFont: Load font from {}", handle);
            Resource res = ticket.getResource(handle);
            final InputStream fi = res.adaptTo(InputStream.class);
            if(fi == null) {
                throw new IOException("Resource does not adapt to InputStream:" + res.getPath());
            }
            java.awt.Font af = java.awt.Font.createFont(java.awt.Font.TRUETYPE_FONT,
                    fontFileProvider.getFileForStream(fi));

            // if we have to derive the style, cache the plain version
            if (baseFont != null) {
                log.debug("loadBaseFont: Cache plain style base and derive");
                // this is the plain font with the name
                fontCache.put(baseFont, af);

                // derive the desired style from the plain font
                af = af.deriveFont(style);
            }

            // cache the font with name and style (derived or exact)
            log.debug("loadBaseFont: Cache desired font with style");
            fontCache.put(fontName, af);

            // and finally return
            return af;

            // } catch (ContentBusException cbe) {
            // throw new NoSuchElementException("Cannot access font page "
            // + handle + ": " + cbe.getMessage());
        } catch (IOException ioe) {
            log.warn("IO problem loading the font from " + handle, ioe);
        } catch (FontFormatException ffe) {
            log.warn("The font page " + handle
                + " does not seem to contain a valid TrueType font", ffe);
        }

        // fall back to no ContentBusFont
        return null;
    }

}
