package com.deque.axe.android

import com.deque.axe.android.constants.AxeStatus
import com.deque.axe.android.wrappers.AxeProps

/**
 * Construct a RuleResult from its components.
 * @param ruleId The id of the rule.
 * @param ruleSummary A summary of the rule.
 * @param axeViewId The id of the view it's associated with.
 * @param status The status of the rule (PASS, FAIL, etc)
 * @param impact How sever the issue is.
 * @param axeProps Properties analyzed to come to these conclusions.
 */
open class AxeRuleResult(
        /**
     * The ID of the rule that generated the result. Very often also the ClassName of the Rule.
     */
    @JvmField val ruleId: String?,
        /**
     * The basic description of the Rule that failed.
     */
    @JvmField val ruleSummary: String?,
        /**
     * A unique identifier for an AxeView within a given set of axeRuleResults.
     */
    @JvmField val axeViewId: String?,
        /**
     * One of [PASS, FAIL, INCOMPLETE, INAPPLICABLE].
     */
    @JvmField @AxeStatus val status: String?,
        /**
     * The impact of the rule [0=MINOR, 1=MODERATE, 2=SERIOUS, 3=CRITICAL, 4=BLOCKER].
     */
    @JvmField var impact: Int = 0,
        /**
     * The properties used in determining that the AxeView was in violation.
     */
    @JvmField val props: AxeProps?,
        /**
     * True if the view is visible to the user.
     */
    @JvmField val isVisibleToUser: Boolean = true
) {
    /**
     * A simple interface that allows us to query AxeResult objects.
     */
    interface Matcher {
        fun matches(ruleResult: AxeRuleResult?): Boolean
    }

    constructor(
            @AxeStatus status: String?,
            axeRule: AxeRule?,
            axeProps: AxeProps?,
            axeView: AxeView?
    ) : this(
            axeRule?.id,
            axeRule?.summary,
            axeView?.nodeId,
            status,
            axeRule?.impact ?: 0,
            axeProps,
            axeView != null
                    && (axeView.calculatedProps[CalculatedPropsEnum.IS_VISIBLE_TO_USER.prop] == null ||
                    (axeView.calculatedProps[CalculatedPropsEnum.IS_VISIBLE_TO_USER.prop] == true))
    )
}
