package com.deque.axe.android.moshi

import com.deque.axe.android.AxeConf
import com.deque.axe.android.AxeContext
import com.deque.axe.android.AxeDevice
import com.deque.axe.android.AxeMetaData
import com.deque.axe.android.AxeResult
import com.deque.axe.android.AxeRule
import com.deque.axe.android.AxeRuleResult
import com.deque.axe.android.AxeView
import com.deque.axe.android.colorcontrast.AxeColor
import com.deque.axe.android.conf.IssueFilterConf
import com.deque.axe.android.conf.RuleConf
import com.deque.axe.android.wrappers.AxeRect
import com.deque.axe.android.wrappers.TextBoundsInScreen
import com.deque.networking.models.devtools.CalculatedProps

internal data class AxeResultJSON(
    val axeConf: AxeConfJSON?,
    val axeContext: AxeContextJSON?,
    val axeRuleResults: List<AxeRuleResultJSON>?,
    val scanName: String?,
    val tags: Set<String>?
) {
    constructor(result: AxeResult) : this(
        axeConf = result.axeConf?.let { AxeConfJSON(it) },
        axeContext = result.axeContext?.let { AxeContextJSON(it) },
        axeRuleResults = result.axeRuleResults?.map { AxeRuleResultJSON(it) },
        scanName = result.scanName,
        tags = result.tags
    )
}

internal data class AxeConfJSON(
    val standards: Set<String>,
    val rules: Map<String, RuleConfJSON> = mapOf(),
    val issueFilterConf: IssueFilterConfJSON = IssueFilterConfJSON(false),
    @JvmField @Transient val customRules: Set<Class<out AxeRule>> = mutableSetOf()
) {
    constructor(axeConf: AxeConf) : this(
        standards = axeConf.standards,
        rules = axeConf.rules.fromRuleConf(),
        issueFilterConf = IssueFilterConfJSON(axeConf.issueFilterConf),
        customRules = axeConf.customRules
    )
}

internal data class RuleConfJSON(
    val impact: Int,
    val standard: String,
    val summary: String,
    val ignored: Boolean,
    val experimental: Boolean?
) {
    constructor(ruleConf: RuleConf) : this(
        impact = ruleConf.impact,
        standard = ruleConf.standard,
        summary = ruleConf.summary,
        ignored = ruleConf.ignored,
        experimental = ruleConf.experimental
    )
}

internal fun Map<String, RuleConf>.fromRuleConf(): Map<String, RuleConfJSON> {
    return this.mapValues { entry: Map.Entry<String, RuleConf> -> RuleConfJSON(entry.value) }
}

internal data class IssueFilterConfJSON(
    var onlyShowResultsVisibleToUser: Boolean
) {
    constructor(issueFilterConf: IssueFilterConf) : this(issueFilterConf.onlyShowResultsVisibleToUser)
}

internal data class AxeContextJSON(
    val axeView: AxeViewJSON,
    val axeDevice: AxeDeviceJSON?,
    val screenshot: String?,
    val axeMetaData: AxeMetaDataJSON?
) {
    constructor(axeContext: AxeContext) : this(
        axeView = AxeViewJSON(axeContext.axeView),
        axeDevice = axeContext.axeDevice?.let { AxeDeviceJSON(it) },
        screenshot = axeContext.screenshot?.toBase64Png(),
        axeMetaData = axeContext.axeMetaData?.let { AxeMetaDataJSON(it) }
    )
}

internal data class AxeDeviceJSON(
    val dpi: Float,
    val name: String?,
    val os: String?,
    val osVersion: String,
    val screenHeight: Int,
    val screenWidth: Int
) {
    constructor(axeDevice: AxeDevice) : this(
        dpi = axeDevice.dpi,
        name = axeDevice.name,
        os = axeDevice.os,
        osVersion = axeDevice.osVersion,
        screenHeight = axeDevice.screenHeight,
        screenWidth = axeDevice.screenWidth
    )
}

internal data class AxeMetaDataJSON(
    val appIdentifier: String?,
    val screenTitle: String?,
    val analysisTimestamp: Long?,
    val axeVersion: String?
) {
    constructor(axeMetaData: AxeMetaData) : this(
        appIdentifier = axeMetaData.appIdentifier,
        screenTitle = axeMetaData.screenTitle,
        analysisTimestamp = axeMetaData.analysisTimestamp,
        axeVersion = axeMetaData.axeVersion,
    )
}

internal data class AxeViewJSON(
    val boundsInScreen: AxeRectJSON,
    val className: String?,
    val contentDescription: String?,
    val isAccessibilityFocusable: Boolean,
    val isFocusable: Boolean,
    val isClickable: Boolean,
    val isEnabled: Boolean = true,
    val isImportantForAccessibility: Boolean,
    val labeledBy: AxeViewJSON?,
    val packageName: String?,
    val paneTitle: String?,
    val text: String?,
    val viewIdResourceName: String?,
    val hintText: String?,
    val value: String?,
    val children: List<AxeViewJSON>,
    val overridesAccessibilityDelegate: Boolean = false,
    val isVisibleToUser: Boolean = true,
    val visibility: Int = 0,
    val measuredHeight: Int = 0,
    val measuredWidth: Int = 0,
    val id: Int = 0,
    val textColor: AxeColor?,
    val isLongClickable: Boolean = false,
    val isComposeView: Boolean = false,
    val ignoreRules: Set<String> = setOf(),
    val axeViewId: String,
    val calculatedProps: CalculatedProps = mapOf(),
    val screenTitle: String? = null,
    val mlKitIdentifiedTextAndBoundsInScreen: List<TextBoundsInScreen> = listOf(),
    val screenOrientation: Int? = null,
    val appLabel: String? = null,
    val activityClassName: String? = null,
    val isRootView: Boolean = false,
    val fragmentClassNames: List<String>? = null,
) {
    constructor(axeView: AxeView) : this(
        boundsInScreen = AxeRectJSON(axeView.boundsInScreen),
        className = axeView.className,
        contentDescription = axeView.contentDescription,
        isAccessibilityFocusable = axeView.isAccessibilityFocusable,
        isFocusable = axeView.isFocusable,
        isClickable = axeView.isClickable,
        isEnabled = axeView.isEnabled,
        isImportantForAccessibility = axeView.isImportantForAccessibility,
        labeledBy = axeView.labeledBy?.let { AxeViewJSON(it) },
        packageName = axeView.packageName,
        paneTitle = axeView.paneTitle,
        text = axeView.text,
        viewIdResourceName = axeView.viewIdResourceName,
        hintText = axeView.hintText,
        value = axeView.value,
        children = axeView.children.map { AxeViewJSON(it) },
        overridesAccessibilityDelegate = axeView.overridesAccessibilityDelegate,
        isVisibleToUser = axeView.isVisibleToUser,
        visibility = axeView.visibility,
        measuredHeight = axeView.measuredHeight,
        measuredWidth = axeView.measuredWidth,
        id = axeView.id,
        textColor = axeView.textColor,
        isLongClickable = axeView.isLongClickable,
        isComposeView = axeView.isComposeView,
        ignoreRules = axeView.ignoreRules,
        axeViewId = axeView.axeViewId,
        calculatedProps = axeView.calculatedProps,
        screenTitle = axeView.screenTitle,
        mlKitIdentifiedTextAndBoundsInScreen = axeView.mlKitIdentifiedTextAndBoundsInScreen,
        screenOrientation = axeView.screenOrientation,
        appLabel = axeView.appLabel,
        activityClassName = axeView.activityClassName,
        isRootView = axeView.isRootView,
        fragmentClassNames = axeView.fragmentClassNames,
    )
}

internal data class AxeRectJSON(
    val left: Int,
    val right: Int,
    val top: Int,
    val bottom: Int
) {
    constructor(axeRect: AxeRect) : this(
        left = axeRect.left,
        right = axeRect.right,
        top = axeRect.top,
        bottom = axeRect.bottom
    )
}

internal data class AxeRuleResultJSON(
    val ruleId: String?,
    val ruleSummary: String?,
    val axeViewId: String?,
    val status: String?,
    var impact: Int = 0,
    val props: Map<String, Any?>?,
    val isVisibleToUser: Boolean = true,
) {
    constructor(axeRuleResult: AxeRuleResult) : this(
        ruleId = axeRuleResult.ruleId,
        ruleSummary = axeRuleResult.ruleSummary,
        axeViewId = axeRuleResult.axeViewId,
        status = axeRuleResult.status,
        impact = axeRuleResult.impact,
        props = axeRuleResult.props,
        isVisibleToUser = axeRuleResult.isVisibleToUser,
    )
}