package com.deque.axe.android.wrappers

class AxeRect(left: Int, right: Int, top: Int, bottom: Int) : Comparable<AxeRect> {
    @JvmField val left: Int = if (left < right) left else right
    @JvmField val right: Int = if (left < right) right else left
    @JvmField val top: Int = if (top < bottom) top else bottom
    @JvmField val bottom: Int = if (top < bottom) bottom else top

    constructor(rect: AxeRect) : this(rect.left, rect.right, rect.top, rect.bottom)

    fun isWithin(container: AxeRect): Boolean = container.top <= top && container.bottom >= bottom && container.left <= left && container.right >= right

    fun isMlTextWithin(container: AxeRect): Boolean {
        val midX = left + (width() / 2)
        val midY = top + (height() / 2)

        return container.top <= midY
                && container.bottom >= midY
                && container.left <= midX
                && container.right >= midX
    }

    fun isLeadingEdge(point: AxePoint): Boolean = point.valueX == left

    fun isTrailingEdge(point: AxePoint): Boolean = isTrailingEdge(point.valueX)

    fun width(): Int = right - left

    fun height(): Int = bottom - top

    fun surfaceArea(): Int = width() * height()

    fun binaryRowSearch(maxWidthBetweenRows: Int): Iterable<AxePoint> {
        return Iterable { binaryRowSearch(top, bottom, maxWidthBetweenRows) }
    }

    private fun isTrailingEdge(column: Int): Boolean = column == right

    private fun binaryRowSearch(y1: Int, y2: Int, widthBetweenRows: Int): Iterator<AxePoint> {
        val maxY = if (y1 < y2) y2 else y1
        val minY = if (y1 < y2) y1 else y2
        return if (maxY - minY <= widthBetweenRows) {
            object : Iterator<AxePoint> {
                override fun hasNext(): Boolean {
                    return false
                }

                @Throws(NoSuchElementException::class)
                override fun next(): AxePoint {
                    throw NoSuchElementException()
                }
            }
        } else object : Iterator<AxePoint> {
            @JvmField var valueX = left
            @JvmField val middle = (minY + maxY) / 2
            @JvmField val topHalf = binaryRowSearch(minY, middle, widthBetweenRows)
            @JvmField val bottomHalf = binaryRowSearch(middle, maxY, widthBetweenRows)
            override fun hasNext(): Boolean {
                return if (!isTrailingEdge(valueX - 1)) {
                    true
                } else topHalf.hasNext() || bottomHalf.hasNext()
            }

            @Throws(NoSuchElementException::class)
            override fun next(): AxePoint {
                if (!isTrailingEdge(valueX - 1)) {
                    return AxePoint(valueX++, middle)
                }
                if (topHalf.hasNext()) {
                    return topHalf.next()
                }
                if (bottomHalf.hasNext()) {
                    return bottomHalf.next()
                }
                throw NoSuchElementException()
            }
        }
    }

    override fun equals(other: Any?): Boolean {
        if (other === this) {
            return true
        }
        if (other !is AxeRect) {
            return false
        }
        return other.compareTo(this) == 0
    }

    override fun hashCode(): Int {
        var result = left
        result = 31 * result + right
        result = 31 * result + top
        result = 31 * result + bottom
        return result
    }

    override fun compareTo(other: AxeRect): Int {
        var temp = top.compareTo(other.top)
        if (temp != 0) {
            return temp
        }
        temp = left.compareTo(other.left)
        if (temp != 0) {
            return temp
        }
        temp = bottom.compareTo(other.bottom)
        return if (temp != 0) {
            temp
        } else right.compareTo(other.right)
    }
    override fun toString(): String = ("Rect($left, $right  - $top, $bottom)")
}