package com.deque.networking

import com.deque.networking.models.devtools.serializable.AxeDevToolsResultKey
import kotlinx.coroutines.TimeoutCancellationException
import retrofit2.HttpException

class AxeLogger {
    companion object {
        private const val tag: String = "Axe DevTools Android"

        private fun log(message: String) {
            println("$tag: $message")
        }

        fun notAuthenticated(e: Throwable) {
            log("${e.message} User is not authenticated! Please use valid api key or credentials before connecting.")
        }

        fun notAuthenticatedAtScan() {
            log("Could not start the scan because the network client isn't authenticated yet. Please reconnect again.")
        }

        fun scanNotReturningBecauseResultIsNull() {
            log("Scan returning null because scan result is incomplete.")
        }

        fun activityContentNotFoundAtScanTime() {
            log("Could not scan because it could not find contentView of Activity.")
        }

        fun networkError(e: Throwable) {
            log("Network Error: ${e.message}")
        }

        fun clientNotInitialized() {
            log("Network Client is not initialized! Try connecting again.")
        }

        fun desktopDown() {
            log("Could not connect to the Desktop app, please restart it and wait until it's running.")
        }

        fun instrumentationRegistryIsNull() {
            log("Could not scan, please provide Instrumentation Registry.")
        }

        fun nodeInfoIsNull() {
            log("Could not scan, please make sure that entire view is populated on the screen before scanning.")
        }

        fun serverDown() {
            log("Could not connect to the Dashboard, please run the app again.")
        }

        fun logNetworkError(e: Throwable) {
            when  {
                e is HttpException && e.message?.contains("401") == true -> notAuthenticated(e)
                else -> networkError(e)
            }
        }

        fun errorWhileDoingPixelCopy(e: Throwable) {
            log("Error while taking screenshot on the view, trying canvas instead, ${e.message}")
        }

        fun errorWhileScreenshot(e: Throwable) {
            log("Error while using canvas to take screenshot on the view, ${e.message}")
        }

        fun errorWhileCreatingScreenshot(e: Throwable) {
            log("Error while capturing and creating screenshot: ${e.message}")
        }

        fun errorWhileSaving(e: Throwable) {
            log("Error while saving file: ${e.message}")
        }

        fun errorWhileRunningMLkitTextDetection(e: Throwable) {
            log("Error while running mlKit Text Detection: ${e.message}")
        }

        fun axeResultSavedAt(location: String?) {
            log("AxeResult saved at: $location")
        }

        fun noClassDefFoundError(msg: String, e: Throwable) {
            log("${e.localizedMessage} at $msg")
        }

        fun composeNotSupportedAtLessThanAPI26() {
            log("Not able to scan because the library does not support API less than 26.")
        }

        fun resourceNameNotFound(resId: String) {
            log("Resource Name not found this view with id: $resId")
        }

        fun resultKeyIsNotValid(resultKey: AxeDevToolsResultKey) {
            log("AxeDevToolsResultKey is returning invalid path: $resultKey")
        }

        fun errorWhileRunningComposeAxeRule(ruleId: String, e: Throwable) {
            log("$ruleId incomplete, error reason: ${e.localizedMessage}")
        }

        fun deleteResultFailure(resultKey: AxeDevToolsResultKey, e: Throwable) {
            log(
                "Could not delete result with id ${resultKey.resultId}: ${e.localizedMessage}",
            )
        }

        fun fromPathRequiresThreeElements(pathElements: Array<String>) {
            log("AxeDevToolsResultKey's fromPath method requires 3 elements in the pathElements array: $pathElements")
        }

        fun errorWhileBuildingView(e: Exception) {
            log("Error while creating Axe View: ${e.localizedMessage}")
        }

        fun dialogInflationTimeout(e: TimeoutCancellationException) {
            log("AxeDevTools timed out while attempting to scan your dialog.")
            e.printStackTrace()
        }
    }
}