package com.deque.networking.interfaces

import com.deque.axe.android.AxeResult
import com.deque.networking.models.DequeRetrofitBuilder
import com.deque.networking.models.auth.AuthSource
import com.deque.networking.models.devtools.TagsSet
import com.deque.networking.models.devtools.serializable.AxeDevToolsResult
import com.deque.networking.models.devtools.serializable.AxeDevToolsResultKey
import com.deque.networking.models.devtools.serializable.AxeDevToolsResultSummaryResponse
import com.deque.networking.models.devtools.serializable.UserInfo

internal class AxeDevToolsDashboardClient(
    authSource: AuthSource, backendUrl: String
) : AxeDevToolsClient {

    private val axeDevToolsDashboardInterface: AxeDevToolsDashboardInterface =
        DequeRetrofitBuilder(backendUrl, AuthHeaderInterceptor(authSource))
            .okHttpBuilder()
            .client(AxeDevToolsDashboardInterface::class.java)

    override suspend fun getResult(
        axeDevToolsResultKey: AxeDevToolsResultKey
    ): Result<AxeDevToolsResult> {
        val callResult = axeDevToolsDashboardInterface.getAxeDevToolsResultAxe(
            axeDevToolsResultKey.userId,
            axeDevToolsResultKey.packageName,
            axeDevToolsResultKey.resultId
        )

        val axeDevToolsResult = callResult.getOrNull()

        return if (axeDevToolsResult != null) {
            Result.success(
                AxeDevToolsResult(
                    axeDevToolsResult,
                    axeDevToolsResultKey
                )
            )
        } else {
            Result.failure(callResult.exceptionOrNull()!!)
        }
    }

    override suspend fun postResult(axeResult: AxeResult): Result<AxeDevToolsResultKey> {
        val callResult = axeDevToolsDashboardInterface.uploadResult(axeResult)
        val axeDevToolsResultKey = callResult.getOrNull()

        return if (axeDevToolsResultKey != null) {
            Result.success(
                axeDevToolsResultKey
            )
        } else {
            Result.failure(callResult.exceptionOrNull()!!)
        }
    }

    override suspend fun deleteResult(axeDevToolsResultKey: AxeDevToolsResultKey): Result<Unit> {
        return axeDevToolsDashboardInterface.deleteResult(
            axeDevToolsResultKey.userId,
            axeDevToolsResultKey.packageName,
            axeDevToolsResultKey.resultId
        )
    }

    override suspend fun tag(
        axeDevToolsResultKey: AxeDevToolsResultKey,
        tags: TagsSet
    ): Result<AxeDevToolsResultKey> {
        return axeDevToolsDashboardInterface.postAxeDevToolsResultTag(
            axeDevToolsResultKey.userId,
            axeDevToolsResultKey.packageName,
            axeDevToolsResultKey.resultId,
            tags
        )
    }

    override suspend fun setScanName(
        axeDevToolsResultKey: AxeDevToolsResultKey,
        scanName: String
    ): Result<AxeDevToolsResultSummaryResponse> {
        return axeDevToolsDashboardInterface.setScanName(
            axeDevToolsResultKey.userId,
            axeDevToolsResultKey.packageName,
            axeDevToolsResultKey.resultId,
            scanName
        )
    }

    override suspend fun getUserInfo(): Result<UserInfo> {
        return axeDevToolsDashboardInterface.getUserInfo()
    }
}