package com.deque.networking.interfaces

import com.deque.networking.AxeLogger
import com.deque.networking.analytics.AmplitudeEventConstant
import com.deque.networking.analytics.AmplitudeEventProps
import com.deque.networking.analytics.AmplitudeEventType
import com.deque.networking.analytics.AnalyticsService
import com.deque.networking.interfaces.AxeDevToolsClient.Companion.BASE_FRONTEND_URL
import com.deque.networking.models.auth.AuthSource
import com.deque.networking.models.auth.AuthSource.Companion.API_KEY_HEADER
import com.deque.networking.models.devtools.TagsSet
import com.deque.networking.models.devtools.serializable.AxeDevToolsResult
import com.deque.networking.models.devtools.serializable.AxeDevToolsResultKey
import com.deque.networking.models.devtools.serializable.AxeDevToolsResultSummaryResponse
import com.deque.networking.models.devtools.serializable.UserInfo
import kotlinx.coroutines.runBlocking

private const val ADT_CLIENT_INITIALIZATION_ERROR = "Client Disconnected"

class ResultsDashboard {

    companion object {
        @JvmStatic
        var axeDevToolsClient: AxeDevToolsClient? = null
    }

    fun authorize(authSource: AuthSource, serverConfig: ConnectionConfig): ResultsDashboard {
        axeDevToolsClient = initializeClient(authSource, serverConfig.dbUrl)
        return this
    }

    fun authorizeLocally(port: String): ResultsDashboard {
        axeDevToolsClient = initializeClient(port)
        return this
    }

    val isConnected: Boolean
        get() = runBlocking {
            axeDevToolsClient?.let { isUserAuthenticated(it) } ?: false
        }

    suspend fun getResult(axeDevToolsResultKey: AxeDevToolsResultKey): Result<AxeDevToolsResult> {
        AnalyticsService.sendEvent(AmplitudeEventType.SCAN_RETRIEVE)
        return axeDevToolsClient?.getResult(axeDevToolsResultKey)?.onFailure {
            AnalyticsService.sendEvent(
                AmplitudeEventType.SCAN_RETRIEVE_ERROR,
                AmplitudeEventProps(reason = it.message)
            )
        } ?: Result.failure(UninitializedPropertyAccessException(ADT_CLIENT_INITIALIZATION_ERROR))
    }

    suspend fun postResult(axeResult: com.deque.axe.android.AxeResult): Result<AxeDevToolsResultKey> {
        AnalyticsService.sendEvent(AmplitudeEventType.SCAN_SEND)
        return axeDevToolsClient?.postResult(axeResult)?.onSuccess {
            println("ADT result at: ${BASE_FRONTEND_URL}?userId=${it.userId}&packageName=${it.packageName}&resultId=${it.resultId}")
        }?.onFailure {
            AnalyticsService.sendEvent(
                AmplitudeEventType.SCAN_SEND_ERROR,
                AmplitudeEventProps(reason = it.message)
            )
        } ?: Result.failure(UninitializedPropertyAccessException(ADT_CLIENT_INITIALIZATION_ERROR))
    }

    suspend fun deleteResult(axeDevToolsResultKey: AxeDevToolsResultKey): Result<Unit> {
        AnalyticsService.sendEvent(AmplitudeEventType.SCAN_DELETE)
        return axeDevToolsClient?.deleteResult(axeDevToolsResultKey)?.onFailure {
            AnalyticsService.sendEvent(
                AmplitudeEventType.SCAN_DELETE_ERROR,
                AmplitudeEventProps(reason = it.message)
            )
        } ?: Result.failure(UninitializedPropertyAccessException(ADT_CLIENT_INITIALIZATION_ERROR))
    }

    suspend fun tag(
        axeDevToolsResultKey: AxeDevToolsResultKey,
        tags: TagsSet
    ): Result<AxeDevToolsResultKey> {
        AnalyticsService.sendEvent(
            AmplitudeEventType.SCAN_TAG,
            eventProps = AmplitudeEventProps(
                tags = tags.toList()
            )
        )

        return axeDevToolsClient?.tag(axeDevToolsResultKey, tags.toSet())?.onFailure {
            AnalyticsService.sendEvent(
                AmplitudeEventType.SCAN_TAG_ERROR,
                AmplitudeEventProps(reason = it.message)
            )
        } ?: Result.failure(UninitializedPropertyAccessException(ADT_CLIENT_INITIALIZATION_ERROR))
    }

    suspend fun setScanName(
        axeDevToolsResultKey: AxeDevToolsResultKey,
        scanName: String
    ): Result<AxeDevToolsResultSummaryResponse> {
        AnalyticsService.sendEvent(
            AmplitudeEventType.SCAN_UPDATE_NAME,
            eventProps = AmplitudeEventProps(
                scan_name = scanName
            )
        )

        return axeDevToolsClient?.setScanName(axeDevToolsResultKey, scanName)?.onFailure {
            AnalyticsService.sendEvent(
                AmplitudeEventType.SCAN_UPDATE_NAME_ERROR,
                AmplitudeEventProps(
                    reason = it.message
                )
            )
        } ?: Result.failure(UninitializedPropertyAccessException(ADT_CLIENT_INITIALIZATION_ERROR))
    }

    suspend fun getUserInfo(): Result<UserInfo> {
        return axeDevToolsClient?.getUserInfo() ?: Result.failure(
            UninitializedPropertyAccessException(
                ADT_CLIENT_INITIALIZATION_ERROR
            )
        )
    }

    fun disconnect() {
        axeDevToolsClient = null
    }

    private fun initializeClient(authSource: AuthSource, dbUrl: String): AxeDevToolsClient {
        val axeDevToolsClient: AxeDevToolsClient = AxeDevToolsDashboardClient(authSource, dbUrl)

        val errorEventType = if (authSource.getAuthHeader() == API_KEY_HEADER)
            AmplitudeEventType.LOGIN_API_KEY_ERROR else AmplitudeEventType.LOGIN_USERNAME_ERROR

        checkConnectionStatusAndLogResult(axeDevToolsClient, errorEventType)

        return axeDevToolsClient
    }

    private fun initializeClient(port: String): AxeDevToolsClient {
        return AxeDevToolsDashboardClient(
            object : AuthSource {
                override fun getToken(): String = ""
                override fun getAuthHeader(): String = ""
            },
            "http://localhost:$port"
        )
    }

    private fun checkConnectionStatusAndLogResult(
        axeDevToolsClient: AxeDevToolsClient,
        errorEventType: AmplitudeEventType
    ) {
        val userAuthenticated = runBlocking { isUserAuthenticated(axeDevToolsClient) }
        if (!userAuthenticated) {
            AnalyticsService.sendEvent(
                errorEventType,
                AmplitudeEventProps(
                    reason = AmplitudeEventConstant.USER_NOT_AUTH
                )
            )
        }
    }

    private suspend fun isUserAuthenticated(axeDevToolsClient: AxeDevToolsClient): Boolean {
        val userInfo = axeDevToolsClient.getUserInfo().onFailure {
            AxeLogger.notAuthenticated(it)
        }.getOrNull()

        return userInfo?.userId?.isNotBlank() ?: false
    }
}