/*
 * Code licensed under new-style BSD (see LICENSE).
 * All code up to tags/original: Copyright (c) 2006, Wojciech Gradkowski
 * All code after tags/original: Copyright (c) 2015, DiffPlug
 */
package com.jmatio.types;

import java.io.UnsupportedEncodingException;
import java.util.Arrays;

import javax.annotation.Nonnull;

import com.jmatio.common.MatDataTypes;

public class MLArray {

	/* Matlab Array Types (Classes) */
	public static final int mxUNKNOWN_CLASS = 0;
	public static final int mxCELL_CLASS = 1;
	public static final int mxSTRUCT_CLASS = 2;
	public static final int mxOBJECT_CLASS = 3;
	public static final int mxCHAR_CLASS = 4;
	public static final int mxSPARSE_CLASS = 5;
	public static final int mxDOUBLE_CLASS = 6;
	public static final int mxSINGLE_CLASS = 7;
	public static final int mxINT8_CLASS = 8;
	public static final int mxUINT8_CLASS = 9;
	public static final int mxINT16_CLASS = 10;
	public static final int mxUINT16_CLASS = 11;
	public static final int mxINT32_CLASS = 12;
	public static final int mxUINT32_CLASS = 13;
	public static final int mxINT64_CLASS = 14;
	public static final int mxUINT64_CLASS = 15;
	public static final int mxFUNCTION_CLASS = 16;
	public static final int mxOPAQUE_CLASS = 17;

	public static final int mtFLAG_COMPLEX = 0x0800;
	public static final int mtFLAG_GLOBAL = 0x0400;
	public static final int mtFLAG_LOGICAL = 0x0200;
	public static final int mtFLAG_TYPE = 0xff;

	@Nonnull
	protected final int dims[];
	@Nonnull
	public String name;
	protected final int attributes;
	protected final int type;
	private final int dimStrides[];  // Used to convert multidimensional index to linear index

	public static final String DEFAULT_NAME = "@";

	public MLArray(String name, int[] dims, int type, int attributes) {
		this.dims = Arrays.copyOf(dims, dims.length);

		if (name != null && !name.equals("")) {
			this.name = name;
		} else {
			this.name = DEFAULT_NAME; //default name
		}

		this.type = type;
		this.attributes = attributes;

		// the indices are processed in order from 0 to max in order to match Matlab's 
		// column major ordering of storage in the .mat file
		dimStrides = new int[dims.length];
		for (int dimIx = 0, f = 1; dimIx < dims.length; dimIx++) {
			dimStrides[dimIx] = f;
			f *= dims[dimIx];
		}
	}

	/**
	 * Returns the one-dim index for the multi-dimensional indexes.  Compatible with matlab multi-dimensional indexing.
	 * 
	 * Note: this performs the same logical function as getIndex, but the indices are computed in column major order
	 * for compatibility with .mat files generated by Matlab.
	 * 
	 * @param indexes Length must be same as number of dimensions. Element value must be >= 0 and < dimension size for the corresponding dimension.
	 * @return The linear index
	 */
	public int getIndex(int... indexes) {
		if (indexes.length != dims.length) {
			throw new IllegalArgumentException("Cannot use " + indexes.length + " indexes for " + dims.length + " dimensions.");
		}
		int ix = 0;
		for (int dimIx = 0; dimIx < indexes.length; dimIx++) {
			ix += dimStrides[dimIx] * validateDimSize(dimIx, indexes[dimIx]);
		}
		return ix;
	}

	private int validateDimSize(int dimIx, int ixInDim) {
		if (dims[dimIx] > ixInDim) {
			return ixInDim;
		}
		throw new IllegalArgumentException("Index " + ixInDim + " does not exist for dimension " + dimIx);
	}

	/**
	 * Gets array name
	 * 
	 * @return - array name
	 */
	public String getName() {
		return name;
	}

	public int getFlags() {
		return type & mtFLAG_TYPE | attributes & 0xffffff00;
	}

	public byte[] getNameToByteArray() {
		try {
			return name.getBytes(MatDataTypes.CHARSET);
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}
	}

	public int[] getDimensions() {
		return Arrays.copyOf(dims, dims.length);
	}

	public int getM() {
		return dims[0];
	}

	public int getN() {
		int i = dims[1];
		for (int j = 2; j < dims.length; j++) {
			i *= dims[j];
		}
		return i;
	}

	public int getNDimensions() {
		return dims.length;
	}

	public int getSize() {
		return getM() * getN();
	}

	public int getType() {
		return type;
	}

	public boolean isEmpty() {
		return getN() == 0;
	}

	public static final String typeToString(int type) {
		String s;
		// @formatter:off
		switch (type) {
		case mxUNKNOWN_CLASS:	s = "unknown";			break;
		case mxCELL_CLASS:		s = "cell";				break;
		case mxSTRUCT_CLASS:	s = "struct";			break;
		case mxCHAR_CLASS:		s = "char";				break;
		case mxSPARSE_CLASS:	s = "sparse";			break;
		case mxDOUBLE_CLASS:	s = "double";			break;
		case mxSINGLE_CLASS:	s = "single";			break;
		case mxINT8_CLASS:		s = "int8";				break;
		case mxUINT8_CLASS:		s = "uint8";			break;
		case mxINT16_CLASS:		s = "int16";			break;
		case mxUINT16_CLASS:	s = "uint16";			break;
		case mxINT32_CLASS:		s = "int32";			break;
		case mxUINT32_CLASS:	s = "uint32";			break;
		case mxINT64_CLASS:		s = "int64";			break;
		case mxUINT64_CLASS:	s = "uint64";			break;
		case mxFUNCTION_CLASS:	s = "function_handle";	break;
		case mxOPAQUE_CLASS:	s = "opaque";			break;
		case mxOBJECT_CLASS:	s = "object";			break;
		default:				s = "unknown";			break;
		}
		// @formatter:on
		return s;
	}

	public boolean isCell() {
		return type == mxCELL_CLASS;
	}

	public boolean isChar() {
		return type == mxCHAR_CLASS;
	}

	public boolean isComplex() {
		return (attributes & mtFLAG_COMPLEX) == mtFLAG_COMPLEX;
	}

	public boolean isSparse() {
		return type == mxSPARSE_CLASS;
	}

	public boolean isStruct() {
		return type == mxSTRUCT_CLASS;
	}

	public boolean isDouble() {
		return type == mxDOUBLE_CLASS;
	}

	public boolean isSingle() {
		return type == mxSINGLE_CLASS;
	}

	public boolean isInt8() {
		return type == mxINT8_CLASS;
	}

	public boolean isUint8() {
		return type == mxUINT8_CLASS;
	}

	public boolean isInt16() {
		return type == mxINT16_CLASS;
	}

	public boolean isUint16() {
		return type == mxUINT16_CLASS;
	}

	public boolean isInt32() {
		return type == mxINT32_CLASS;
	}

	public boolean isUint32() {
		return type == mxUINT32_CLASS;
	}

	public boolean isInt64() {
		return type == mxINT64_CLASS;
	}

	public boolean isUint64() {
		return type == mxUINT64_CLASS;
	}

	public boolean isObject() {
		return type == mxOBJECT_CLASS;
	}

	public boolean isOpaque() {
		return type == mxOPAQUE_CLASS;
	}

	public boolean isLogical() {
		return (attributes & mtFLAG_LOGICAL) == mtFLAG_LOGICAL;
	}

	public boolean isFunctionObject() {
		return type == mxFUNCTION_CLASS;
	}

	public boolean isUnknown() {
		return type == mxUNKNOWN_CLASS;
	}

	protected int getIndex(int m, int n) {
		return m + n * getM();
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append('[');
		if (dims.length > 3) {
			sb.append(dims.length);
			sb.append('D');
		} else {
			sb.append(dims[0]);
			sb.append('x');
			sb.append(dims[1]);
			if (dims.length == 3) {
				sb.append('x');
				sb.append(dims[2]);
			}
		}
		sb.append("  ");
		sb.append(typeToString(type));
		if (isLogical()) {
			sb.append(" (logical)");
		}
		sb.append(" array");
		if (isSparse()) {
			sb.append(" (sparse");
			if (isComplex()) {
				sb.append(" complex");
			}
			sb.append(")");
		} else if (isComplex()) {
			sb.append(" (complex)");
		}
		sb.append(']');
		return sb.toString();
	}

	public String contentToString() {
		return "content cannot be displayed";
	}

	/** Clears any memory used by this MLArray.  The array cannot be used after this has been called. */
	public void dispose() {}
}
