// Copyright (c) 1999-2004 Brian Wellington (bwelling@xbill.org)

package org.xbill.DNS;

import java.io.*;
import java.net.*;

/**
 * IPv6 Address Record - maps a domain name to an IPv6 address
 *
 * @author Brian Wellington
 */

public class AAAARecord extends Record {

private static final long serialVersionUID = -4588601512069748050L;

private InetAddress address;

AAAARecord() {}

Record
getObject() {
	return new AAAARecord();
}

/**
 * Creates an AAAA Record from the given data
 * @param address The address suffix
 */
public
AAAARecord(Name name, int dclass, long ttl, InetAddress address) {
	super(name, Type.AAAA, dclass, ttl);
	if (Address.familyOf(address) != Address.IPv6)
		throw new IllegalArgumentException("invalid IPv6 address");
	this.address = address;
}

void
rrFromWire(DNSInput in) throws IOException {
	if (name == null)
		address = InetAddress.getByAddress(in.readByteArray(16));
	else
		address = InetAddress.getByAddress(name.toString(),
						   in.readByteArray(16));
}

void
rdataFromString(Tokenizer st, Name origin) throws IOException {
	address = st.getAddress(Address.IPv6);
}

/** Converts rdata to a String 
 * @return the string
 * */
String
rrToString() {
	return address.getHostAddress();
}

/** Returns the address 
 * @return the InetAddress
 * */
public InetAddress
getAddress() {
	return address;
}

void
rrToWire(DNSOutput out, Compression c, boolean canonical) {
	byte[] thisAddressBytes = address.getAddress();
	if(thisAddressBytes.length == 4) {
		byte[] templateIPV6 = new byte[16];
		templateIPV6[10] = (byte)0xFF;
		templateIPV6[11] = (byte)0xFF;
		System.arraycopy(thisAddressBytes, 0, templateIPV6, 12, 4);
		out.writeByteArray(templateIPV6);
	} else {
		out.writeByteArray(thisAddressBytes);
	}
}

}
