/**
 */

package smithy4s.meta;

import software.amazon.smithy.model.node.ExpectationNotMetException;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.traits.AbstractTrait;
import software.amazon.smithy.model.traits.AbstractTraitBuilder;
import software.amazon.smithy.model.traits.Trait;
import software.amazon.smithy.utils.SmithyBuilder;
import software.amazon.smithy.utils.SmithyGenerated;
import software.amazon.smithy.utils.ToSmithyBuilder;

/**
 * Marks the given member shape as one that was added to the structure _after_ it was initially created and its generated
 * code was published.
 * Adding such members is a change that keeps binary compatibility.
 */
@SmithyGenerated
public final class BincompatAddedTrait extends AbstractTrait implements ToSmithyBuilder<BincompatAddedTrait> {
    public static final ShapeId ID = ShapeId.from("smithy4s.meta#bincompatAdded");

    private final String version;

    private BincompatAddedTrait(Builder builder) {
        super(ID, builder.getSourceLocation());
        this.version = SmithyBuilder.requiredState("version", builder.version);
    }

    @Override
    protected Node createNode() {
        return Node.objectNodeBuilder()
            .sourceLocation(getSourceLocation())
            .withMember("version", Node.from(version))
            .build();
    }

    /**
     * Creates a {@link BincompatAddedTrait} from a {@link Node}.
     *
     * @param node Node to create the BincompatAddedTrait from.
     * @return Returns the created BincompatAddedTrait.
     * @throws ExpectationNotMetException if the given Node is invalid.
     */
    public static BincompatAddedTrait fromNode(Node node) {
        Builder builder = builder();
        node.expectObjectNode()
            .expectStringMember("version", builder::version);

        return builder.build();
    }

    /**
     * Used to determine which members of the structure were added, and in what order.
     * Members marked with the same version will be grouped together, and appended to the previous version's members in any
     * generated constructors.
     * You must not add new members with a version number that's already been published.
     * The version must consist of a sequence of dot-separated numbers, e.g. "1.0", "1.2.3", "2.0.0", "1.2.3.4".
     */
    public String getVersion() {
        return version;
    }

    /**
     * Creates a builder used to build a {@link BincompatAddedTrait}.
     */
    public SmithyBuilder<BincompatAddedTrait> toBuilder() {
        return builder().sourceLocation(getSourceLocation())
            .version(version);
    }

    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder for {@link BincompatAddedTrait}.
     */
    public static final class Builder extends AbstractTraitBuilder<BincompatAddedTrait, Builder> {
        private String version;

        private Builder() {}

        public Builder version(String version) {
            this.version = version;
            return this;
        }

        @Override
        public BincompatAddedTrait build() {
            return new BincompatAddedTrait(this);
        }
    }

    public static final class Provider extends AbstractTrait.Provider {
        public Provider() {
            super(ID);
        }

        @Override
        public Trait createTrait(ShapeId target, Node value) {
            BincompatAddedTrait result = BincompatAddedTrait.fromNode(value);
            result.setNodeCache(value);
            return result;
        }
    }
}
