/*
 * Decompiled with CFR 0.152.
 */
package com.dnastack.auth;

import com.dnastack.auth.JwtTokenParser;
import com.dnastack.auth.TokenRevocationChecker;
import com.dnastack.auth.cache.CachingConcurrentHashMap;
import com.dnastack.auth.client.PolicyEvaluationClient;
import com.dnastack.auth.exception.UnmetPolicyRequirementsException;
import com.dnastack.auth.model.AccessCheck;
import com.dnastack.auth.model.AccessDecision;
import com.dnastack.auth.model.AnonymousPrincipal;
import com.dnastack.auth.model.Evaluation;
import com.dnastack.auth.model.EvaluationRequest;
import com.dnastack.auth.model.IssuerInfo;
import com.dnastack.auth.model.JwtTokenParsers;
import com.dnastack.auth.model.Resource;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.JwtException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

public class PermissionChecker {
    private static final Duration CLOCK_SKEW = Duration.of(15L, ChronoUnit.SECONDS);
    private final Clock clock = Clock.systemUTC();
    private final JwtTokenParsers jwtTokenParsers;
    private final TokenRevocationChecker tokenRevocationChecker;
    private final String policyEvaluationRequester;
    private final PolicyEvaluationClient policyEvaluationClient;
    private final long fetchRevokedEvaluationsIntervalMs;
    private long lastEvaluationRevocationCheck;
    private final long expirePolicyCacheAfter;
    private final Map<List<Resource>, Evaluation> evaluationCache;

    public PermissionChecker(JwtTokenParsers jwtTokenParsers, TokenRevocationChecker tokenRevocationChecker, String policyEvaluationRequester, PolicyEvaluationClient policyEvaluationClient, long expirePolicyCacheAfter, long fetchRevokedEvaluationsIntervalMs) {
        this.jwtTokenParsers = jwtTokenParsers;
        this.tokenRevocationChecker = tokenRevocationChecker;
        this.policyEvaluationRequester = policyEvaluationRequester;
        this.policyEvaluationClient = policyEvaluationClient;
        this.fetchRevokedEvaluationsIntervalMs = fetchRevokedEvaluationsIntervalMs;
        this.expirePolicyCacheAfter = expirePolicyCacheAfter;
        this.evaluationCache = new CachingConcurrentHashMap<List<Resource>, Evaluation>(expirePolicyCacheAfter, 10000L, val -> val != null && val.getExp() != null && val.getExp().isAfter(Instant.now()));
    }

    public void checkPermissions(String jwtToken) {
        JwtTokenParser jwtTokenParser = this.jwtTokenParsers.getParser();
        this.checkPermissions(jwtToken, jwtTokenParser);
        Jws<Claims> jwsClaims = jwtTokenParser.parseTokenClaims(jwtToken);
        this.validateRevocation(jwtTokenParser.parseTokenJti(jwsClaims), jwtTokenParser.parseTokenIssuer(jwsClaims));
    }

    private void checkPermissions(String jwtToken, JwtTokenParser jwtTokenParser) {
        if (Objects.isNull(jwtToken) || jwtToken.isBlank()) {
            throw new IllegalArgumentException("Argument jwtToken is required and cannot be null.");
        }
        Jws<Claims> jwsClaims = jwtTokenParser.parseTokenClaims(jwtToken);
        this.validateExpiry(jwtTokenParser.parseTokenExpiration(jwsClaims));
        this.validateIssuers(jwtTokenParser.getAllowedIssuers(), jwtTokenParser.parseTokenIssuer(jwsClaims));
        this.validateAudiences(jwtTokenParser.parseTokenIssuer(jwsClaims), jwtTokenParser.parseTokenAudiences(jwsClaims), jwtTokenParser.getAllowedIssuers());
    }

    public void checkPermissions(String jwtToken, Collection<String> requiredScopes, Map<String, Set<String>> requiredActionsForResources) {
        if (!Objects.isNull(jwtToken)) {
            if (jwtToken.isBlank()) {
                throw new IllegalArgumentException("Argument jwtToken cannot be blank.");
            }
            if (Objects.isNull(requiredScopes) || requiredScopes.isEmpty()) {
                throw new IllegalArgumentException("Argument requiredScopes is required and cannot be null.");
            }
            if (Objects.isNull(requiredActionsForResources) || requiredActionsForResources.isEmpty()) {
                throw new IllegalArgumentException("Argument requiredActionsForResources is required and cannot be null.");
            }
            requiredActionsForResources.forEach((resource, requiredActionsForResource) -> {
                JwtTokenParser jwtTokenParser = this.jwtTokenParsers.getParserForResource((String)resource);
                this.checkPermissions(jwtToken, jwtTokenParser);
                Jws<Claims> jwsClaims = jwtTokenParser.parseTokenClaims(jwtToken);
                this.validateScopes(requiredScopes, jwtTokenParser.parseTokenScopes(jwsClaims));
                this.validateActions(Map.of(resource, requiredActionsForResource), jwtTokenParser.parseTokenActions(jwsClaims));
            });
        } else {
            this.evaluateAnonymousPrincipal(requiredActionsForResources);
        }
    }

    public void checkPermissions(String jwtToken, Map<String, Set<String>> requiredActionsForResources) {
        if (!Objects.isNull(jwtToken)) {
            if (jwtToken.isBlank()) {
                throw new IllegalArgumentException("Argument jwtToken cannot be blank.");
            }
            if (Objects.isNull(requiredActionsForResources) || requiredActionsForResources.isEmpty()) {
                throw new IllegalArgumentException("Argument requiredActionsForResources is required and cannot be null.");
            }
            requiredActionsForResources.forEach((resource, requiredActionsForResource) -> {
                JwtTokenParser jwtTokenParser = this.jwtTokenParsers.getParserForResource((String)resource);
                this.checkPermissions(jwtToken, jwtTokenParser);
                Jws<Claims> jwsClaims = jwtTokenParser.parseTokenClaims(jwtToken);
                this.validateActions(Map.of(resource, requiredActionsForResource), jwtTokenParser.parseTokenActions(jwsClaims));
            });
        } else {
            this.evaluateAnonymousPrincipal(requiredActionsForResources);
        }
    }

    public void evaluateAnonymousPrincipal(Map<String, Set<String>> requiredActionsForResources) {
        if (Objects.isNull(requiredActionsForResources) || requiredActionsForResources.isEmpty()) {
            throw new IllegalArgumentException("Argument requiredActionsForResources is required and cannot be null.");
        }
        List<Resource> resources = requiredActionsForResources.keySet().stream().map(resource -> Resource.builder().uri((String)resource).build()).collect(Collectors.toList());
        Evaluation evaluation = this.getEvaluationForAnonymousPrincipal(resources);
        this.validateActions(requiredActionsForResources, evaluation.getActions());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Evaluation getEvaluationForAnonymousPrincipal(List<Resource> resources) {
        EvaluationRequest evaluationRequest = EvaluationRequest.builder().requester(this.policyEvaluationRequester).resources(resources).subject(new AnonymousPrincipal()).build();
        PermissionChecker permissionChecker = this;
        synchronized (permissionChecker) {
            long now = System.currentTimeMillis();
            if (this.lastEvaluationRevocationCheck + this.fetchRevokedEvaluationsIntervalMs < now) {
                this.removeRevokedCacheEntries();
            }
        }
        return this.evaluationCache.computeIfAbsent(resources, key -> this.policyEvaluationClient.getEvaluation(evaluationRequest));
    }

    private void removeRevokedCacheEntries() {
        long lastChecked = this.lastEvaluationRevocationCheck != 0L ? this.lastEvaluationRevocationCheck : System.currentTimeMillis() - this.expirePolicyCacheAfter;
        this.lastEvaluationRevocationCheck = System.currentTimeMillis();
        Set<List<Resource>> revokedResources = this.getRevokedEvaluations(lastChecked);
        for (List<Resource> resource : revokedResources) {
            this.evaluationCache.remove(resource);
        }
    }

    private Set<List<Resource>> getRevokedEvaluations(long after) {
        List<String> audiences = this.policyEvaluationClient.getRevokedEvaluationsAfter(after);
        HashSet<List<Resource>> revokedResources = new HashSet<List<Resource>>();
        Set<List<Resource>> evaluationCacheKeys = this.evaluationCache.keySet();
        audiences.forEach(audience -> evaluationCacheKeys.forEach(resources -> {
            boolean revoked = resources.stream().map(Resource::getUri).anyMatch(uri -> uri.startsWith((String)audience));
            if (revoked) {
                revokedResources.add((List<Resource>)resources);
            }
        }));
        return revokedResources;
    }

    private void validateRevocation(String jti, String issuerUri) {
        if (this.tokenRevocationChecker.isTokenRevoked(jti, issuerUri)) {
            throw new JwtException("Token has been revoked");
        }
    }

    private void validateExpiry(Date tokenExpiresAt) {
        Instant expiresAt;
        if (tokenExpiresAt != null && (expiresAt = tokenExpiresAt.toInstant()).isBefore(Instant.now(this.clock).minus(CLOCK_SKEW))) {
            throw new JwtException(String.format("Token is expired. Expired at %s.", expiresAt));
        }
    }

    private void validateIssuers(Collection<IssuerInfo> allowedIssuers, String tokenIssuer) {
        List allowedIssuersUris = allowedIssuers.stream().map(IssuerInfo::getIssuerUri).collect(Collectors.toUnmodifiableList());
        if (allowedIssuersUris.stream().noneMatch(allowedIssuer -> allowedIssuer.equals(tokenIssuer))) {
            throw new JwtException(String.format("Expected following issuers %s to be present in given token. Instead found %s.", allowedIssuersUris, tokenIssuer));
        }
    }

    private void validateAudiences(String tokenIssuer, Collection<String> tokenAudiences, Collection<IssuerInfo> allowedIssuers) {
        Collection<String> allowedAudiences = this.getAllowedAudiencesForIssuer(tokenIssuer, allowedIssuers);
        if (allowedAudiences.stream().noneMatch(tokenAudiences::contains)) {
            throw new JwtException(String.format("Expected following audiences %s to be present in given token. Instead found %s.", allowedAudiences, tokenAudiences));
        }
    }

    private Collection<String> getAllowedAudiencesForIssuer(String tokenIssuer, Collection<IssuerInfo> allowedIssuers) {
        return allowedIssuers.stream().filter(issuer -> issuer.getIssuerUri().equals(tokenIssuer)).findFirst().orElseThrow().getAllowedAudiences();
    }

    private void validateScopes(Collection<String> allowedScopes, Collection<String> tokenScopes) {
        if (!tokenScopes.containsAll(allowedScopes)) {
            throw new UnmetPolicyRequirementsException(String.format("Expected following scopes %s to be present in given token. Instead found %s.", allowedScopes, tokenScopes));
        }
    }

    private void validateActions(Map<String, Set<String>> requiredActions, Map<String, Set<String>> providedActions) {
        requiredActions.forEach((resource, requiredActionsForResource) -> {
            Set<String> grantedActions = this.getGrantedActions(providedActions, (String)resource);
            if (!grantedActions.containsAll((Collection<?>)requiredActionsForResource)) {
                throw new UnmetPolicyRequirementsException(String.format("Expected following actions %s to be present on resource [%s] in given token. Instead found %s.", requiredActionsForResource, resource, grantedActions));
            }
        });
    }

    private Set<String> getGrantedActions(Map<String, Set<String>> providedActionsForResources, String resource) {
        return providedActionsForResources.entrySet().stream().filter(entry -> {
            String tokenResource = (String)entry.getKey();
            return resource.equals(tokenResource) || tokenResource.endsWith("/") && resource.startsWith(tokenResource) || resource.endsWith("/") && tokenResource.startsWith(resource);
        }).map(Map.Entry::getValue).flatMap(Collection::stream).collect(Collectors.toUnmodifiableSet());
    }

    public boolean hasPermissions(String jwtToken, Collection<String> requiredScopes, String requiredResource, Set<String> requiredActions) {
        try {
            this.checkPermissions(jwtToken, requiredScopes, Map.of(requiredResource, requiredActions));
            return true;
        }
        catch (UnmetPolicyRequirementsException | JwtException e) {
            return false;
        }
    }

    public AccessDecision hasPermissions(String jwtToken, AccessCheck accessCheck) {
        if (Objects.isNull(accessCheck)) {
            throw new IllegalArgumentException("Argument accessCheck cannot be null.");
        }
        try {
            this.checkPermissions(jwtToken, accessCheck.getRequiredScopes(), Map.of(accessCheck.getResourceUri(), accessCheck.getRequiredActions()));
            return new AccessDecision(accessCheck.getResourceUri(), true);
        }
        catch (UnmetPolicyRequirementsException | JwtException e) {
            return new AccessDecision(accessCheck.getResourceUri(), false);
        }
    }

    public List<AccessDecision> hasPermissions(String jwtToken, Collection<AccessCheck> accessChecks) {
        if (Objects.isNull(accessChecks)) {
            throw new IllegalArgumentException("Argument accessChecks cannot be null.");
        }
        if (!Objects.isNull(jwtToken)) {
            return accessChecks.stream().map(accessCheck -> this.hasPermissions(jwtToken, (AccessCheck)accessCheck)).collect(Collectors.toList());
        }
        return this.hasAnonymousPrincipalPermissions(accessChecks);
    }

    public AccessDecision hasAnonymousPrincipalPermissions(AccessCheck accessCheck) {
        if (Objects.isNull(accessCheck)) {
            throw new IllegalArgumentException("Argument accessCheck cannot be null.");
        }
        Resource resource = Resource.builder().uri(accessCheck.getResourceUri()).build();
        Evaluation evaluation = this.getEvaluationForAnonymousPrincipal(List.of(resource));
        return this.getAccessDecision(accessCheck, evaluation);
    }

    public List<AccessDecision> hasAnonymousPrincipalPermissions(Collection<AccessCheck> accessChecks) {
        if (Objects.isNull(accessChecks)) {
            throw new IllegalArgumentException("Argument accessChecks cannot be null.");
        }
        List<Resource> resources = accessChecks.stream().map(accessCheck -> Resource.builder().uri(accessCheck.getResourceUri()).build()).collect(Collectors.toList());
        Evaluation evaluation = this.getEvaluationForAnonymousPrincipal(resources);
        return this.getAccessDecisions(accessChecks, evaluation);
    }

    private List<AccessDecision> getAccessDecisions(Collection<AccessCheck> accessChecks, Evaluation evaluation) {
        return accessChecks.stream().map(accessCheck -> this.getAccessDecision((AccessCheck)accessCheck, evaluation)).collect(Collectors.toList());
    }

    private AccessDecision getAccessDecision(AccessCheck accessCheck, Evaluation evaluation) {
        Set<String> grantedActions = this.getGrantedActions(evaluation.getActions(), accessCheck.getResourceUri());
        boolean hasAccess = grantedActions.containsAll(accessCheck.getRequiredActions());
        return new AccessDecision(accessCheck.getResourceUri(), hasAccess);
    }

    public boolean hasPermissions(String jwtToken, String requiredResource, Set<String> requiredActions) {
        try {
            this.checkPermissions(jwtToken, Map.of(requiredResource, requiredActions));
            return true;
        }
        catch (UnmetPolicyRequirementsException | JwtException e) {
            return false;
        }
    }

    public boolean hasAnonymousPrincipalPermissions(String requiredResource, Set<String> requiredActions) {
        try {
            this.evaluateAnonymousPrincipal(Map.of(requiredResource, requiredActions));
            return true;
        }
        catch (UnmetPolicyRequirementsException ex) {
            return false;
        }
    }
}

