package com.domeke.core.db.sql;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.beetl.core.BeetlKit;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.util.FileCopyUtils;

import com.domeke.core.kit.JaxbKit;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jfinal.kit.PathKit;
import com.jfinal.plugin.activerecord.Table;

public class SqlKit {
	protected static final Logger logger = LoggerFactory.getLogger(SqlKit.class);

	/**
	 * xml中所有的sql语句
	 */
	private static final Map<String, String> sqlMap = new HashMap<String, String>();

	/**
	 * 过滤掉的sql关键字
	 */
	private static final List<String> badKeyWordList = new ArrayList<String>();

	/**
	 * 加载关键字到List
	 */
	static {
		String badStr = "'|and|exec|execute|insert|select|delete|update|count|drop|*|%|chr|mid|master|truncate|"
				+ "char|declare|sitename|net user|xp_cmdshell|;|or|-|+|,|like'|and|exec|execute|insert|create|drop|"
				+ "table|from|grant|use|group_concat|column_name|"
				+ "information_schema.columns|table_schema|union|where|select|delete|update|order|by|count|*|"
				+ "chr|mid|master|truncate|char|declare|or|;|-|--|+|,|like|//|/|%|#";
		badKeyWordList.addAll(Arrays.asList(badStr.split("\\|")));
	}

	/**
	 * sql查询关键字过滤效验
	 * 
	 * @param queryStr
	 * @return
	 */
	public static boolean keywordVali(String queryStr) {
		queryStr = queryStr.toLowerCase();// 统一转为小写
		for (String badKeyWord : badKeyWordList) {
			if (queryStr.indexOf(badKeyWord) >= 0) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 获取SQL，固定SQL
	 * 
	 * @param sqlId
	 * @return
	 */
	public static String getSql(String sqlId) {
		String sql = sqlMap.get(sqlId);
		if (null == sql || sql.isEmpty()) {
			logger.error("sql语句不存在：sql id是" + sqlId);
		}

		return sql.replaceAll("[\\s]{2,}", " ");
	}

	/**
	 * 获取SQL，动态SQL
	 * 
	 * @param sqlId
	 * @param param
	 * @return
	 */
	public static String getSql(String sqlId, Map<String, Object> param) {
		String sqlTemplete = sqlMap.get(sqlId);
		if (null == sqlTemplete || sqlTemplete.isEmpty()) {
			logger.error("sql语句不存在：sql id是" + sqlId);
		}
		String sql = BeetlKit.render(sqlTemplete, param);

		Set<String> keySet = param.keySet();
		for (String key : keySet) {
			if (param.get(key) == null) {
				continue;
			}
			String value = (String) param.get(key);
			value = value.replace("'", "").replace(";", "").replace("--", "");
			sql = sql.replace("#" + key + "#", value);
		}

		return sql.replaceAll("[\\s]{2,}", " ");
	}

	/**
	 * 获取SQL，动态SQL
	 * 
	 * @param sqlId
	 * @param param
	 *            查询参数
	 * @param list
	 *            用于接收预处理的值
	 * @return
	 */
	public static String getSql(String sqlId, Map<String, String> param, LinkedList<Object> list) {
		String sqlTemplete = sqlMap.get(sqlId);
		if (null == sqlTemplete || sqlTemplete.isEmpty()) {
			logger.error("sql语句不存在：sql id是" + sqlId);
		}

		Map<String, Object> paramMap = Maps.newHashMap();
		;
		Set<String> paramKeySet = param.keySet();
		for (String paramKey : paramKeySet) {
			paramMap.put(paramKey, (Object) param.get(paramKey));
		}
		String sql = BeetlKit.render(sqlTemplete, paramMap);

		Pattern pattern = Pattern.compile("#[\\w\\d\\$\\'\\%\\_]+#"); // #[\\w\\d]+#
																		// \\$
		Pattern pattern2 = Pattern.compile("\\$[\\w\\d\\_]+\\$");

		Matcher matcher = pattern.matcher(sql);

		while (matcher.find()) {
			String clounm = matcher.group(0); // 得到的结果形式：#'%$names$%'#

			Matcher matcher2 = pattern2.matcher(clounm);
			matcher2.find();
			String clounm2 = matcher2.group(0); // 得到的结果形式：$names$

			String clounm3 = clounm2.replace("$", "");

			if (clounm.equals("#" + clounm2 + "#")) { // 数值型，可以对应处理int、long、bigdecimal、double等等
				String val = (String) param.get(clounm3);
				try {
					Integer.parseInt(val);
					sql = sql.replace(clounm, val);
				} catch (NumberFormatException e) {
					logger.error("查询参数值错误，整型值传入了字符串，非法字符串是：" + val);
					return null;
				}

			} else { // 字符串，主要是字符串模糊查询、日期比较的查询
				String val = (String) param.get(clounm3);

				String clounm4 = clounm.replace("#", "").replace("'", "").replace(clounm2, val);
				list.add(clounm4);

				sql = sql.replace(clounm, "?");
			}
		}

		return sql.replaceAll("[\\s]{2,}", " ");
	}

	/**
	 * 清除加载的sql
	 */
	public static void destory() {
		sqlMap.clear();
	}

	/**
	 * 初始化加载sql语句到map
	 */
	public static synchronized void init(boolean isInit) {
		Long startTime = System.currentTimeMillis();

		getResourceInJar();

		File file = new File(PathKit.getRootClassPath());

		List<File> files = new ArrayList<File>();
		try {
			listFiles(file, files);
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		String key = "";
		String sql = "";
		SqlStatement group = null;
		for (File xmlfile : files) {
			group = JaxbKit.unmarshal(xmlfile, SqlStatement.class);
			String filename = xmlfile.getName();;
			String namespace = group.namespace;
			if (namespace == null || namespace.trim().equals("")) {
				namespace = xmlfile.getName();
			}
			for (SqlItem sqlItem : group.sqlItems) {
				key = namespace + "." + sqlItem.id;
				sql = sqlItem.value;
				sqlMap.put(key, sql);
			}

			if (null == namespace || namespace.trim().isEmpty()) {
				logger.error("sql xml文件" + filename + "的命名空间不能为空");
				continue;
			}
			sql = sql.replaceAll("[\\s]{2,}", " ");
			sqlMap.put(key, sql);
			logger.debug("sql加载, sql file = " + filename + ", sql key = " +key  + ", sql content = " + sql);
		}

		long endTime = System.currentTimeMillis();

		logger.debug("初始化 SQL 文件耗时  =" + (endTime - startTime));
	}

	public static Object[] filterDataStateParams(Object... paras) {
		List<Object> list = Lists.newArrayList();
		if (paras.length > 0) {

		}
		for (int i = 0; i < paras.length; i++) {
			list.add(paras[i]);
		}
		list.add(Long.valueOf(1));
		Object[] array = list.toArray();
		return array;
	}

	public static String filterDataState(Table table, String sql) {

		if (sql.contains("where")) {
			if (table.hasColumnLabel("datastate")) {
				sql += " and datastate = ?";
			}
		} else {
			if (table.hasColumnLabel("datastate")) {
				sql += " where datastate = ?";
			}
		}
		return sql;
	}

	public static void listFiles(File file, List<File> sqlXmlFiles) throws IOException {

		if (file.getName().endsWith(".sql.xml")) {
			sqlXmlFiles.add(file);
		} else if (file.isDirectory()) {
			File[] subFiles = file.listFiles();
			for (File subFile : subFiles) {
				listFiles(subFile, sqlXmlFiles);
			}
		}

	}

	private static void getResourceInJar() {
		ResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
		Resource[] resources = null;
		;
		try {
			resources = resolver.getResources("classpath*:com/domeke/core/model/*.sql.xml");
			File e = null;
			FileOutputStream os = null;
			InputStream inputStream = null;
			File file = null;
			String path = PathKit.getRootClassPath();
			for (Resource resource : resources) {
				file = new File(path + File.separator + resource.getFilename());
				os = new FileOutputStream(file);
				inputStream = resource.getInputStream();
				FileCopyUtils.copy(inputStream, os);
			}
		} catch (IOException e1) {
			e1.printStackTrace();
		}

	}

}
