package com.domeke.core.model;

import java.lang.reflect.Field;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.domeke.core.db.sql.SqlKit;
import com.domeke.core.kit.CommonKit;
import com.google.common.collect.Maps;
import com.jfinal.plugin.activerecord.Model;
import com.jfinal.plugin.activerecord.Table;
import com.jfinal.plugin.activerecord.TableMapping;

/**
 * Model基础类
 * 
 * @author lijiasen@126.com
 * @param <M>
 */
public abstract class BaseModel<M extends Model<M>> extends Model<M> {

	private static final long serialVersionUID = -900378319414539856L;

	private static Logger log = LoggerFactory.getLogger(BaseModel.class);

	/**
	 * 获取SQL，固定SQL
	 * 
	 * @param sqlId
	 * @return
	 */
	protected String getSql(String sqlId) {
		return SqlKit.getSql(sqlId);
	}

	/**
	 * 获取SQL，动态SQL
	 * 
	 * @param sqlId
	 * @param param
	 * @return
	 */
	protected String getSql(String sqlId, Map<String, Object> param) {
		return SqlKit.getSql(sqlId, param);
	}

	/**
	 * 获取SQL，动态SQL
	 * 
	 * @param sqlId
	 * @param param
	 *            查询参数
	 * @param list
	 *            用于接收预处理的值
	 * @return
	 */
	protected String getSql(String sqlId, Map<String, String> param, LinkedList<Object> list) {
		return SqlKit.getSql(sqlId, param, list);
	}

	// /**
	// * 根据i18n参数查询获取哪个字段的值
	// *
	// * @param i18n
	// * @return
	// */
	// protected String i18n(String i18n) {
	// return I18NPlugin.i18n(i18n);
	// }

	/**
	 * 获取表映射对象
	 * 
	 * @return
	 */
	protected Table getTable() {
		return TableMapping.me().getTable(getClass());
	}

	/**
	 * 获取主键值
	 * 
	 * @return
	 */
	public String getPKValue() {
		return this.getStr(getTable().getPrimaryKey()[0]);
	}

	/**
	 * 重写save方法
	 */
	public boolean save() {
		this.set(getTable().getPrimaryKey()[0], CommonKit.getUuidByJdk(true)); // 设置主键值
		if (getTable().hasColumnLabel("version")) { // 是否需要乐观锁控制
			this.set("version", Long.valueOf(0)); // 初始化乐观锁版本号
		}
		if (getTable().hasColumnLabel("create_time")) {
			this.set("create_time", getCurrentTime());
		}

		if (getTable().hasColumnLabel("modify_time")) {
			this.set("modify_time", getCurrentTime());
		}

		if (getTable().hasColumnLabel("datastate")) {
			this.set("datastate", Long.valueOf(1));
		}

		return super.save();
	}

	/**
	 * 获得当前时间
	 * 
	 * @return
	 */
	public Timestamp getCurrentTime() {
		DateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		String time = sdf.format(new Date());
		// 创建时间
		Timestamp nowTime = Timestamp.valueOf(time);
		return nowTime;
	}

	/**
	 * 重写update方法
	 */
	@SuppressWarnings("unchecked")
	public boolean update() {
		Table table = getTable();
		boolean hasVersion = table.hasColumnLabel("version");

		if (hasVersion) {// 是否需要乐观锁控制，表是否有version字段
			String name = table.getName();
			String pk = table.getPrimaryKey()[0];

			// 1.数据是否还存在
			Map<String, Object> param = Maps.newHashMap();;
			param.put("table", name);
			param.put("pk", pk);
			String sql = SqlKit.getSql("domeke.core.baseModel.version", param);
			Model<M> modelOld = findFirst(sql, getPKValue());
			if (null == modelOld) { // 数据已经被删除
				throw new RuntimeException("数据库中此数据不存在，可能数据已经被删除，请刷新数据后在操作");
			}

			// 2.乐观锁控制
			Set<String> modifyFlag = null;
			try {
				Field field = this.getClass().getSuperclass().getSuperclass().getDeclaredField("modifyFlag");
				field.setAccessible(true);
				Object object = field.get(this);
				if (null != object) {
					modifyFlag = (Set<String>) object;
				}
				field.setAccessible(false);
			} catch (NoSuchFieldException | SecurityException e) {
				log.error("业务Model类必须继承BaseModel");
				e.printStackTrace();
				throw new RuntimeException("业务Model类必须继承BaseModel");
			} catch (IllegalArgumentException | IllegalAccessException e) {
				log.error("BaseModel访问modifyFlag异常");
				e.printStackTrace();
				throw new RuntimeException("BaseModel访问modifyFlag异常");
			}
			boolean versionModify = modifyFlag.contains("version"); // 表单是否包含version字段
			if (versionModify) {
				Long versionDB = modelOld.getNumber("version").longValue(); // 数据库中的版本号
				Long versionForm = getNumber("version").longValue() + 1; // 表单中的版本号
				if (!(versionForm > versionDB)) {
					throw new RuntimeException("表单数据版本号和数据库数据版本号不一致，可能数据已经被其他人修改，请重新编辑");
				}
			}

			if (getTable().hasColumnLabel("modify_time")) {
				this.set("modify_time", getCurrentTime());
			}
		}

		return super.update();
	}

	/**
	 * 是否可以物理删除，如果含有datastate字段，则只能物理删除
	 */
	public boolean delete() {
		Table table = getTable();
		boolean isDelEnable = table.hasColumnLabel("datastate");
		// if (isDelEnable) {
		// this.set("datastate", Long.valueOf(0));
		// return update();
		// }
		return super.delete();
	}

	/**
	 * 是否可以物理删除，如果含有datastate字段，则只能物理删除
	 */
	public boolean deleteById(Object id) {
		Table table = getTable();
		boolean isDelEnable = table.hasColumnLabel("datastate");
		// if (isDelEnable) {
		// this.set("datastate", Long.valueOf(0));
		// return update();
		// }
		return super.deleteById(id);
	}

	@Override
	public List<M> find(String sql, Object... paras) {
		Table table = getTable();
		Object[] array = paras;
		if (table.hasColumnLabel("datastate")) {
			// sql = DbKit.filterDataState(getTable(),sql);
			// array = DbKit.filterDataStateParams(paras);
		}
		return super.find(sql, array);
	}

	public boolean isNotEmpty() {
		Object[] attrValues = getAttrValues();
		return attrValues.length > 0 ? true : false;
	}

	public boolean isEmpty() {
		return isNotEmpty();
	}

}
