package com.domeke.core.service;

import java.util.List;
import java.util.Map;

import org.apache.shiro.crypto.RandomNumberGenerator;
import org.apache.shiro.crypto.SecureRandomNumberGenerator;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.domeke.core.base.config.DictKeys;
import com.domeke.core.model.Company;
import com.domeke.core.model.User;
import com.domeke.core.service.cache.ICacheService;
import com.jfinal.plugin.activerecord.Page;
import com.jfinal.plugin.ehcache.CacheKit;
import com.jfinal.plugin.redis.Redis;

/**
 * 用户service
 * 
 * @author Administrator
 *
 */
public class UserService extends BaseService implements ICacheService {

	private static final Logger logger = LoggerFactory.getLogger(UserService.class);

	public static final UserService SERVICE = new UserService();

	public User login(String username) {
		return User.dao.login(username);
	}

	/**
	 * 添加用户
	 * 
	 * @param user
	 */
	public void save(User user) {
		String password = user.getStr("password");
		RandomNumberGenerator rng = new SecureRandomNumberGenerator();
		String salt = rng.nextBytes().toHex();
		String hashedPasswordBase64 = new Sha256Hash(password).toHex();
		user.set("password", hashedPasswordBase64);
		user.set("salt", salt);
		user.save();

		addOrUpdateCache(user);
	}

	private void addOrUpdateCache(User user) {
		Redis.use().del(CACHESTART_USER + user.getPKValue());
		Redis.use().del(CACHESTART_USER + user.getStr("username"));
		Redis.use().del(CACHESTART_USER + user.getStr("email"));
		Redis.use().del(CACHESTART_USER + user.getStr("mobile"));

		Redis.use().set(CACHESTART_USER + user.getPKValue(), user);
		Redis.use().set(CACHESTART_USER + user.getStr("username"), user);
		Redis.use().set(CACHESTART_USER + user.getStr("email"), user);
		Redis.use().set(CACHESTART_USER + user.getStr("mobile"), user);
	}

	/**
	 * 用户更新
	 * 
	 * @param user
	 */
	public void update(User user) {
		String password = user.getStr("password");
		// 如果修改密码加密（密码没有改变不动，否则会导致提交过来的表单密码再次加密保存）
		if (password != null) {
			String userId = user.getStr("userid");
			User upUser = User.dao.findFirst(getSql("domeke.core.user.selectById"), userId);
			if (!password.equals(upUser.get("password"))) {
				// String salt = upUser.get("salt");
				String userName = user.get("username");
				if (userName == null)
					userName = upUser.get("username");
				String hashedPasswordBase64 = new Sha256Hash(password).toHex();
				user.set("password", hashedPasswordBase64);
			} else {
				user.remove("password");
			}
		}
		user.update();
		addOrUpdateCache(user);
	}

	/**
	 * 检查用户名、邮箱是否重复
	 * 
	 * @param userName
	 * @param email
	 * @return
	 */
	public String checkUserNameAndEmail(String userName, String email) {
		User checkName = User.dao.findFirst(getSql("domeke.core.user.selectCountByName"), userName);
		User checkEmail = User.dao.findFirst(getSql("domeke.core.user.selectCountByEmail"), email);
		if (checkName != null)
			return "用户名已存在";
		if (checkEmail != null)
			return "邮箱已存在";
		return null;
	}

	/**
	 * 获取用户信息
	 * 
	 * @param id
	 * @return
	 */
	public User query(Object id) {
		return User.dao.findById(id);
	}

	public List<User> queryAllUser() {
		return User.dao.find(getSql("domeke.core.user.queryAll"));
	}

	public List<User> queryByFilter(Map<String, Object> filter) {
		return User.dao.find(getSql("domeke.core.user.queryByFilter", filter));
	}

	public List<User> queryUserIdNameList() {
		return User.dao.find(getSql("domeke.core.user.queryUserIdNameMap"));
	}

	public Page<User> queryUserList(int pageNumber, Map<String, Object> filter) {
		return User.dao.queryUserList(pageNumber, filter);
	}

	public List<Company> queryCompanyList() {
		return Company.dao.queryCompanyCodedata();
	}

	@Override
	public void cacheAll() {
		logger.info("缓存加载：User start");
		String sql = getSql("domeke.core.user.queryAll");
		List<User> userList = User.dao.find(sql);
		for (User user : userList) {
			CacheKit.put(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + user.getPKValue(), user);
			CacheKit.put(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + user.getStr("username"), user);
			CacheKit.put(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + user.getStr("email"), user);
			CacheKit.put(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + user.getStr("mobile"), user);
			user = null;
		}
		logger.info("缓存加载：User end, size = " + userList.size());
		userList = null;
	}

	/**
	 * 删除缓存
	 */
	public void removeCache(String ids) {
		User user = User.dao.findById(ids);
		CacheKit.remove(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + ids);
		CacheKit.remove(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + user.getStr("username"));
		CacheKit.remove(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + user.getStr("email"));
		CacheKit.remove(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + user.getStr("mobile"));
	}

	/**
	 * 获取缓存
	 * 
	 * @param ids
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public User getCache(String cacheName) {
		User user = CacheKit.get(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER + cacheName);
		return user;
	}

	@Override
	public void removeCache() {
		CacheKit.remove(DictKeys.CACHE_NAME_SYSTEM, CACHESTART_USER);
	}

}
